(:
 Copyright (c) 2020 MarkLogic Corporation
:)
xquery version "1.0-ml";

module namespace es="http://marklogic.com/entity-services";

import module namespace esi="http://marklogic.com/entity-services-impl"
       at "entity-services-impl.xqy";
import module namespace es-codegen="http://marklogic.com/entity-services-codegen"
       at "entity-services-codegen.xqy";
import module namespace inst="http://marklogic.com/entity-services-instance"
       at "entity-services-instance.xqy";

declare default function namespace "http://www.w3.org/2005/xpath-functions";
declare namespace sem="http://marklogic.com/semantics";
declare namespace m="http://marklogic.com/entity-services/mapping";
declare namespace search="http://marklogic.com/appservices/search";
declare namespace xq="http://www.w3.org/2012/xquery";

(: declare option xdmp:mapping "false"; :)
declare option xq:require-feature "xdmp:three-one";


(:~
 : Validates a model.  This function will validate documents or nodes, or
 : in-memory instances of models as map:map or json:object
 :
 : @param $model Any representation of a model.
 : @return A valid representation of the model, converted to map:map as needed.
 : @throws Validation errors.
 :)
declare function es:model-validate(
    $model-descriptor
) as map:map
{
    esi:model-validate($model-descriptor)
};


(:~
 : Creates a model from an XML document or element
 : For JSON documents, this is equivalent to xdmp:json with validation.
 : For XML documents, we transform the input as well.
 :
 : @param $node A JSON or XML document containing an entity model.
 :)
declare function es:model-from-xml(
    $node
) as map:map
{
    if ($node instance of document-node())
    then esi:modernize(esi:model-from-xml($node/node()))
    else esi:modernize(esi:model-from-xml($node))
};

(:~
 : Given a model, returns its XML representation
 : @param A model
 :)
declare function es:model-to-xml(
    $model as map:map
) as element(es:model)
{
    esi:model-to-xml($model)
};

(:~
 : Given a model, ensure it is a valid JSON Schema for the given entity
 : If the model is already a JSON Schema, return it as is
 : @param A model
 : $param The name of the entity to validate
 :)
declare function es:model-to-json-schema(
    $model as map:map,
    $entity-type-name as xs:string
) as map:map
{
    esi:model-to-json-schema($model, $entity-type-name)
};

(:~
 : Given a model, ensure it is a valid JSON Schema
 : The JSON Schema will accept an instance of any entity in the model
 : If the model is already a JSON Schema, return it as is
 : @param A model
 :)
declare function es:model-to-json-schema(
    $model as map:map
) as map:map
{
    esi:model-to-json-schema($model)
};

(: experiment :)
declare function es:model-to-triples(
    $model as map:map
)
{
    esi:model-to-triples($model)
};

(:~
 : Generates an XQuery module that can be customized and used
 : to support transforms associated with a model
 : @param $model  A model.
 : @return An XQuery module (text) that can be edited and installed in a modules database.
 :)
declare function es:instance-converter-generate(
    $model as map:map
) as document-node()
{
    es-codegen:instance-converter-generate($model)
};

(:~
 : Generate one test instance in XML for each entity type in the
 : model.
 : @param A model.
 :)
declare function es:model-get-test-instances(
    $model as map:map
) as element()*
{
    esi:model-get-test-instances($model)
};



(:~
 : Generate a JSON node that can be used with the Management Client API
 : to configure a database for this model
 : Portions of this complete database properties file can be used
 : as building-blocks for the completed database properties
 : index configuration.
 : @param A model.
 :)
declare function es:database-properties-generate(
    $model as map:map
) as document-node()
{
    esi:database-properties-generate($model)
};



(:~
 : Generate a schema that can validate entity instance documents.
 : @param A model.
 : @return An XSD schema that can validate entity instances in XML form.
 :)
declare function es:schema-generate(
    $model as map:map
) as element()*
{
    esi:schema-generate($model)
};


(:~
 : Generates an extraction template that can surface the entity.
 : instance as a view in the rows index.
 : @param A model.
 :)
declare function es:extraction-template-generate(
    $model as map:map
) as document-node()
{
    document {
        esi:extraction-template-generate($model)
    }
};


(:~
 : Generates an element for configuring Search API applications
 : Intended as a starting point for developing a search grammar tailored
 : for entity and relationship searches.
 : @param An entity model
 :)
declare function es:search-options-generate(
    $model as map:map
)
{
    esi:search-options-generate($model)
};

(:~
 : Generates an XQuery module that can create an instance of one
 : type from documents saved by code from another type.
 : @param A model that describes the target type of the conversion.
 : @param A model that describes the source type of the conversion.
 :)
declare function es:version-translator-generate(
    $source-model as map:map,
    $target-model as map:map
) as document-node()
{
    es-codegen:version-translator-generate(
           $source-model,
           $target-model)
};


(:~
 : Generates a configuration for use with ELS and the mgmt client API
 : to secure individual properties with the 'pii' role
 :)
declare function es:pii-generate(
    $model as map:map
) as document-node()
{
    document {
        esi:pii-generate($model)
    }
};


(:~
 : Given a document, gets the instance data
 : from it and returns instances as maps.
 : @param a document, usually es:envelope.
 : @return zero or more entity instances extracted from the document.
 :)
declare function es:instance-from-document(
    $document as document-node()
) as map:map*
{
    inst:instance-from-document($document)
};

(:~
 : Return the canonical XML representation of an instance from
 : a document.  This function does not transform; it's just a
 : projection of elements from a document.
 : @param a document, usually es:envelope.
 : @return zero or more elements that represent instances.
 :)
declare function es:instance-xml-from-document(
    $document as document-node()
) as element()*
{
    inst:instance-xml-from-document($document)
};

(:~
 : Returns the JSON serialization of an instance from
 : a document.
 : @param a document, usually es:envelope.
 : @return zero or more JSON structures that represent instances.
 :)
declare function es:instance-json-from-document(
    $document as document-node()
) as object-node()*
{
    inst:instance-json-from-document($document)
};


(:~
 : Return the attachments from within an es:envelope document.
 : @param a document, usually es:envelope.
 : @return Anything that is contained within the es:attachments
 : element.
 :)
declare function es:instance-get-attachments(
    $document as document-node()
) as item()*
{
    inst:instance-get-attachments($document)
};


(:~
 : Fluent method to add key/value pairs to an entity instance, if the value exists.
 : @param $instance An instance of map:map to add a key to.
 : @param $property-key  The key to add to $instance.
 : @param $value The value to add to $instance for the given key.
:)
declare function es:optional(
    $instance as map:map,
    $property-key as xs:string,
    $value as item()*
) as map:map
{
    typeswitch($value)
    case empty-sequence() return ()
    (: this case handles empty extractions :)
    case map:map return
        if (map:contains($value, "$ref") and empty(map:get($value, "$ref")))
        then ()
        else map:put($instance, $property-key, $value)
    default return map:put($instance, $property-key, $value)
    ,
    $instance
};


(:~
 : Extract values from a sequence of nodes into an property of type array.
 : If there are no nodes on input, then this function returns the empty sequence.
 : @param $source-nodes The node(s) from which to extract values into an array.
 : @param $fn The function to be applied to each sequence item
 : @param $value - the value to add to $instance for the given key.
 :)
declare function es:extract-array(
    $raw-nodes as item()*,
    $fn as function(*)
) as json:array?
{
    let $source-nodes := $raw-nodes[not(. instance of null-node())]
    return
        if (empty($source-nodes)) then ()
        else json:to-array($source-nodes ! $fn(.))
};


(:~
 : Examine an incoming data source to normalize the extraction context.
 : @param $source An incoming source node, which may be an element, JSON node or document.
 : @param $entity-type-name The expected entity type name.
 : @return Either the incoming node intact, or its contents if it's already canonicalized.
 :)
declare function es:init-source(
    $source as item()*,
    $entity-type-name as xs:string
) as item()*
{
    if ( ($source instance of document-node()) )
    then $source/node()
    else if (exists ($source/element()[fn:local-name(.) eq $entity-type-name] ))
    then $source/*
    else $source
};

(:~
 : Initializes an instance data structure, by adding a type key and, if appropriate,
 : a ref key.
 : @param $source-node The input to extractions. Used to determine whether to extract a pointer or an instance node.
 : @param $entity-type-name  The name of this instance's type.
 : @return A json object with $type key, and, if appropriate, a $ref key.
 :)
declare function es:init-instance(
    $source-node as item()*,
    $entity-type-name as xs:string
) as json:object
{
    let $source-node := es:init-source($source-node, $entity-type-name)
    let $instance := json:object()
            =>map:with('$type', $entity-type-name)
    return
        if (empty($source-node/*))
        then $instance=>map:with('$ref', $source-node/data())
        (: Otherwise, this source node contains instance data. Populate it. :)
        else $instance
};


(:~
 : Adds namespace information to an instance.
 : @param $instance the existing instance
 : @param $namespace The namespace uri to be included as metadata.
 : @param $namespace-prefix A prefix to be passed to the instance for its namespace
 : @return The $instance, with namespace info appended
 :)
declare function es:with-namespace(
    $instance as map:map,
    $namespace as xs:string?,
    $namespace-prefix as xs:string?
) as json:object
{
    let $_ :=
        if ($namespace-prefix)
        then (
            map:put($instance, "$namespace", $namespace),
            map:put($instance, "$namespacePrefix", $namespace-prefix)
        )
        else ()
    return $instance
};


(:~
 : Adds the original source document to the entity instance.
 : @param $instance The instance data, to which the source will be attached.
 : @param $source-node The extraction context for the incoming data
 : @param $source The unmodified source document.
 :)
declare function es:add-attachments(
    $instance as map:map,
    $source as item()*
) as map:map
{
    $instance=>map:with('$attachments', $source)
};

(:~
 : Initializes the context to convert instances from one version to another
 : @param $source Zero or more envelopes or canonical instances.
 : @param $entity-type-name The name of the expected Entity Type
 : @return Zero or more sources expected to contain the canonical data of the given type.
 :)
declare function es:init-translation-source(
    $source as item()*,
    $entity-type-name as xs:string
) as item()*
{
    if ( ($source//es:instance/element()[fn:local-name(.) eq $entity-type-name]))
    then $source//es:instance/element()[fn:local-name(.) eq $entity-type-name]
    else $source
};


(:~
 : Copies attachments from an envelope document to a new intance.
 : @param $instance  The target to which to attach source data.
 : @param $source The envelope or canonical instance from which to copy attachments.
 :)
declare function es:copy-attachments(
    $instance as map:map,
    $source as item()*
) as map:map
{
    let $attachments := $source ! fn:root(.)/es:envelope/es:attachments/node()
    return
    if (exists($attachments))
    then $instance=>map:with('$attachments', $attachments)
    else $instance
};


(:~
 : Serializes attachments for storage in an envelope document.
 : @param $instance  The isntance holding attachment data.
 : @param $format The format of the envlosing envlelope.
 : @return If the format does not match that supplied in $format, then the attachment
 :     is returned as a quoted string.  Otherwise, the attachment is returned for inline
       inclusion as a node.
 :)
declare function es:serialize-attachments(
    $instance as map:map,
    $envelope-format as xs:string
) as item()*
{
    switch ($envelope-format)
    case "json" return
        object-node { 'attachments' :
            array-node {
                for $attachment in $instance=>map:get('$attachments')
                let $attachment :=
                    if ($attachment instance of document-node())
                    then $attachment/node()
                    else $attachment
                return
                    typeswitch ($attachment)
                    case object-node() return $attachment
                    case array-node() return $attachment
                    default return xdmp:quote($attachment)
            }
        }
    case "xml" return
        element es:attachments {
            for $attachment in $instance=>map:get('$attachments')
            let $attachment :=
                if ($attachment instance of document-node())
                then $attachment/node()
                else $attachment
            return
                typeswitch ($attachment)
                case element() return $attachment
                default return xdmp:quote($attachment)
        }
    default return fn:error((), "ES-ENV-BADFORMAT", $envelope-format)
};

(:== Experimental ==:)

declare variable $es:MAPPING_COLLECTION := "http://marklogic.com/entity-services/mapping/compiled";
declare variable $es:FUNCTIONDEF_COLLECTION := "http://marklogic.com/entity-services/function-metadata/compiled";

declare %private variable $es:fetch := '
  declare variable $uri as xs:string external;
  fn:doc($uri)
';

declare %private variable $es:fetch-compiled := '
  declare variable $uri as xs:string external;
  declare variable $xslt-uri as xs:string := $uri||".xslt";
  fn:doc($xslt-uri)
';

declare %private variable $es:put := '
  declare variable $uri as xs:string external;
  declare variable $node as node() external;
  declare variable $collection as xs:string external;
  declare variable $xslt-uri as xs:string := $uri||".xslt";
  declare variable $options := 
    <options xmlns="xdmp:document-insert">
      <permissions>{xdmp:document-get-permissions($uri,"elements")}</permissions>
      <collections>{for $c in xdmp:document-get-collections($uri) return <collection>{$c}</collection>,
        <collection>{$collection}</collection>
      }</collections>
    </options>;

  xdmp:document-insert($xslt-uri, $node, $options)
';

declare %private variable $es:delete := '
  declare variable $uri as xs:string external;
  declare variable $xslt-uri as xs:string := $uri||".xslt";

  for $uri in ($uri,$xslt-uri) return xdmp:document-delete($uri)
';

declare function es:mapping-put(
    $uri as xs:string
) as empty-sequence()
{
  let $source :=
    xdmp:eval(
      $es:fetch,
      map:map()=>map:with("uri",$uri),
      map:map()=>map:with("database",xdmp:modules-database()))
  let $compiled := es:mapping-compile($source)
  return
    xdmp:eval($es:put,
      map:map()=>map:with("uri",$uri)=>
        map:with("node",$compiled)=>
        map:with("collection",$es:MAPPING_COLLECTION)
      ,
      map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:mapping-delete(
    $uri as xs:string
) as empty-sequence()
{
  xdmp:eval($es:delete,
    map:map()=>map:with("uri",$uri),
    map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:mapping-compile(
    $mapping as node()
) as node()
{
  let $input := es:mapping-validate($mapping)
  return
    xdmp:xslt-invoke("/data-hub/core/entity-services/mapping-compile.xsl", $input)
};

declare function es:mapping-get(
    $uri as xs:string
) as empty-sequence()
{
  xdmp:eval(
    $es:fetch,
    map:map()=>map:with("uri",$uri),
    map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:compiled-mapping-get(
    $uri as xs:string
) as empty-sequence()
{
  xdmp:eval(
    $es:fetch-compiled,
    map:map()=>map:with("uri",$uri),
    map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:mapping-validate(
  $mapping as node()
)
{
   esi:mapping-validate($mapping)
};

declare function es:function-metadata-put(
    $uri as xs:string
) as empty-sequence()
{
  let $source :=
    xdmp:eval(
      $es:fetch,
      map:map()=>map:with("uri",$uri),
      map:map()=>map:with("database",xdmp:modules-database()))
  let $compiled := es:function-metadata-compile($source)
  return
    xdmp:eval($es:put,
      map:map()=>map:with("uri",$uri)=>
        map:with("node",$compiled)=>
        map:with("collection",$es:FUNCTIONDEF_COLLECTION)
      ,
      map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:function-metadata-delete(
    $uri as xs:string
) as empty-sequence()
{
  xdmp:eval($es:delete,
    map:map()=>map:with("uri",$uri),
    map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:function-metadata-compile(
    $function-metadata as node()
) as node()
{
  let $input := es:function-metadata-validate($function-metadata)
  return
    xdmp:xslt-invoke("/data-hub/core/entity-services/function-metadata.xsl", $input)
};

declare function es:function-metadata-get(
    $uri as xs:string
) as empty-sequence()
{
  xdmp:eval(
    $es:fetch,
    map:map()=>map:with("uri",$uri),
    map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:compiled-function-metadata-get(
    $uri as xs:string
) as empty-sequence()
{
  xdmp:eval(
    $es:fetch-compiled,
    map:map()=>map:with("uri",$uri),
    map:map()=>map:with("database",xdmp:modules-database()))
};

declare function es:function-metadata-validate(
  $function-metadata as node()
)
{
  (: TODO: higher level errors :)
  let $input := 
    typeswitch ($function-metadata)
    case document-node() return
      if ($function-metadata/m:function-defs)
      then $function-metadata
      else fn:error((),"ES-FUNCTION-METADATA-BADFORMAT")
    case element(m:function-defs) return
      document {$function-metadata}
    default return fn:error((),"ES-FUNCTION-METADATA-BADFORMAT")
  return
    validate {$input}
};

declare function es:map-to-canonical(
  $source-instance as node(),
  $mapping-uri as xs:string,
  $options as map:map
) as node()
{
  let $target-entity-name := $options=>map:get("entity")
  let $format := $options=>map:get("format")
  let $format :=
    if (empty($format)) then
      typeswitch ($source-instance)
      case document-node() return
        if ($source-instance/element()) then "xml"
        else "json"
      case element() return "xml"
      default return "json"
    else $format
  let $input :=
    typeswitch ($source-instance)
    case document-node() return $source-instance
    case element() return document { $source-instance }
    default return document { $source-instance }
  let $parms :=
    if (empty($target-entity-name)) then ()
    else map:map()=>map:with("template", $target-entity-name)
  let $results := 
    xdmp:xslt-invoke($mapping-uri||".xslt", $input, (), $parms)
  return
    if ($format="xml")
    then inst:canonical-xml($results)
    else inst:canonical-json($results)
};

declare function es:map-to-canonical(
  $source-instance as node(),
  $mapping-uri as xs:string
) as node()
{
  es:map-to-canonical($source-instance, $mapping-uri, map:map())
};

declare function es:node-map-to-canonical(
  $source-instance as node(),
  $mapping as node(),
  $options as map:map
) as node()
{
  let $target-entity-name := $options=>map:get("entity")
  let $format := $options=>map:get("format")
  let $format :=
    if (empty($format)) then
      typeswitch ($source-instance)
      case document-node() return
        if ($source-instance/element()) then "xml"
        else "json"
      case element() return "xml"
      default return "json"
    else $format
  let $input :=
    typeswitch ($source-instance)
    case document-node() return $source-instance
    default return document { $source-instance }
  let $compiled := es:mapping-compile($mapping)
  let $parms := 
    if (empty($target-entity-name)) then ()
    else map:map()=>map:with("template", $target-entity-name)
  let $results := xdmp:xslt-eval($compiled, $input, (), $parms)
  return
    if ($format="xml")
    then inst:canonical-xml($results)
    else inst:canonical-json($results)
};

declare function es:node-map-to-canonical(
  $source-instance as node(),
  $mapping as node()
) as node()
{
  es:node-map-to-canonical($source-instance, $mapping, map:map())
};

declare %private variable $es:ORACLEMAP := map:map()=>
  map:with("source-entity", es:default-source-entity-oracle#3)=>
  map:with("source-property", es:default-source-property-oracle#3);

declare %private function
es:default-source-entity-oracle(
  $target-entity-name as xs:string,
  $target-entity as map:map,
  $source-model as map:map
) as map:map?
{
  $source-model=>map:get("definitions")=>map:get($target-entity-name)
};

declare %private function
es:default-source-property-oracle(
  $target-property as xs:string,
  $target-entity as map:map,
  $source-entity as map:map?
) as map:map?
{
  if (empty($source-entity)) then ()
  else $source-entity=>map:get("properties")=>map:get($target-property)
};

declare function es:mapping-generate(
  $source-model as map:map?,
  $target-model as map:map,
  $oracle as map:map
) as node()
{
  let $_defaults := (
    if (map:contains($oracle, "source-entity")) then ()
    else map:put($oracle, "source-entity", es:default-source-entity-oracle#3)
    ,
    if (map:contains($oracle, "source-property")) then ()
    else map:put($oracle, "source-property", es:default-source-property-oracle#3)
  )
  return
    es-codegen:mapping-generate($source-model, $target-model, $oracle)
};

declare function es:mapping-generate(
  $source-model as map:map?,
  $target-model as map:map
) as node()
{
  es-codegen:mapping-generate(($source-model,map:map())[1], $target-model, $es:ORACLEMAP)
};

declare function es:function-metadata-generate(
    $uri as xs:string) as node()
{
  es-codegen:introspect-javascript-module($uri)
};

declare function es:function-metadata-generate(
    $namespace as xs:string,
    $uri as xs:string) as node()
{
  es-codegen:introspect-module($namespace, $uri)
};

declare function es:modernize(
     $model as map:map) as map:map
{
  esi:modernize($model)
};

declare function es:instance-validate(
     $instance as node(),
     $model as map:map
) as node()
{
  esi:instance-validate($instance, $model)
};

declare function es:identify-instance-model(
     $doc as node(),
     $deep as xs:boolean
) as map:map*
{
  esi:identify-instance-model($doc, $deep, false()(: no debug :))
};
    
(: For testing :)
declare function es:demodernize(
     $model as map:map) as map:map
{
  esi:demodernize($model)
};

declare function es:demodernize-xml(
     $model as node()) as node()
{
  esi:demodernize-xml($model)
};

