/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.site.util;

import com.liferay.application.list.PanelAppRegistry;
import com.liferay.application.list.PanelCategoryRegistry;
import com.liferay.application.list.constants.PanelCategoryKeys;
import com.liferay.application.list.display.context.logic.PanelCategoryHelper;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.model.LayoutConstants;
import com.liferay.portal.kernel.model.LayoutSet;
import com.liferay.portal.kernel.security.permission.ActionKeys;
import com.liferay.portal.kernel.service.LayoutLocalService;
import com.liferay.portal.kernel.service.LayoutSetLocalService;
import com.liferay.portal.kernel.service.permission.GroupPermissionUtil;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.Http;
import com.liferay.portal.kernel.util.Portal;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.util.WebKeys;

import javax.portlet.PortletRequest;
import javax.portlet.PortletURL;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Julio Camarero
 */
@Component(immediate = true, service = GroupURLProvider.class)
public class GroupURLProvider {

	public String getDisplayURL(
		Group group, ThemeDisplay themeDisplay, boolean privateLayout,
		int layoutsCount) {

		try {
			if ((layoutsCount > 0) ||
				(group.isUser() &&
				 (_layoutLocalService.getLayoutsCount(group, privateLayout) >
					 0))) {

				LayoutSet layoutSet = _layoutSetLocalService.getLayoutSet(
					group.getGroupId(), privateLayout);

				String groupFriendlyURL = _portal.getGroupFriendlyURL(
					layoutSet, themeDisplay);

				if (group.isUser()) {
					return _portal.addPreservedParameters(
						themeDisplay, groupFriendlyURL, false, true);
				}

				return _portal.addPreservedParameters(
					themeDisplay, groupFriendlyURL);
			}
		}
		catch (PortalException pe) {
			_log.error(pe, pe);
		}

		return StringPool.BLANK;
	}

	public String getGroupAdministrationURL(
		Group group, PortletRequest portletRequest) {

		PanelCategoryHelper panelCategoryHelper = new PanelCategoryHelper(
			_panelAppRegistry, _panelCategoryRegistry);

		ThemeDisplay themeDisplay = (ThemeDisplay)portletRequest.getAttribute(
			WebKeys.THEME_DISPLAY);

		String portletId = panelCategoryHelper.getFirstPortletId(
			PanelCategoryKeys.SITE_ADMINISTRATION,
			themeDisplay.getPermissionChecker(), group);

		if (Validator.isNotNull(portletId)) {
			PortletURL groupAdministrationURL =
				_portal.getControlPanelPortletURL(
					portletRequest, group, portletId, 0, 0,
					PortletRequest.RENDER_PHASE);

			if (groupAdministrationURL != null) {
				return groupAdministrationURL.toString();
			}
		}

		return null;
	}

	public String getGroupLayoutsURL(
		Group group, boolean privateLayout, PortletRequest portletRequest) {

		ThemeDisplay themeDisplay = (ThemeDisplay)portletRequest.getAttribute(
			WebKeys.THEME_DISPLAY);

		String groupDisplayURL = group.getDisplayURL(
			themeDisplay, privateLayout);

		if (Validator.isNotNull(groupDisplayURL)) {
			return groupDisplayURL;
		}

		return null;
	}

	public String getGroupURL(Group group, PortletRequest portletRequest) {
		return getGroupURL(group, portletRequest, true);
	}

	public int getLayoutsCount(Group group) {
		int layoutsCount = 0;

		try {
			LayoutSet layoutSet = _layoutSetLocalService.getLayoutSet(
				group.getGroupId(), false);

			layoutsCount = layoutSet.getPageCount();

			if (layoutsCount != 0) {
				layoutsCount = _layoutLocalService.getLayoutsCount(
					group, false,
					new String[] {
						StringPool.CONTENT, LayoutConstants.TYPE_EMBEDDED,
						LayoutConstants.TYPE_LINK_TO_LAYOUT,
						LayoutConstants.TYPE_FULL_PAGE_APPLICATION,
						LayoutConstants.TYPE_PANEL,
						LayoutConstants.TYPE_PORTLET, LayoutConstants.TYPE_URL
					});
			}
		}
		catch (PortalException pe) {
			_log.error(pe, pe);
		}

		return layoutsCount;
	}

	public String getLiveGroupURL(Group group, PortletRequest portletRequest) {
		return getGroupURL(group, portletRequest, false);
	}

	protected String getGroupURL(
		Group group, PortletRequest portletRequest,
		boolean includeStagingGroup) {

		ThemeDisplay themeDisplay = (ThemeDisplay)portletRequest.getAttribute(
			WebKeys.THEME_DISPLAY);

		int layoutsCount = getLayoutsCount(group);

		String groupDisplayURL = getDisplayURL(
			group, themeDisplay, false, layoutsCount);

		if (Validator.isNotNull(groupDisplayURL)) {
			return _http.removeParameter(groupDisplayURL, "p_p_id");
		}

		groupDisplayURL = group.getDisplayURL(themeDisplay, true);

		if (Validator.isNotNull(groupDisplayURL)) {
			return _http.removeParameter(groupDisplayURL, "p_p_id");
		}

		if (includeStagingGroup && group.hasStagingGroup()) {
			try {
				if (GroupPermissionUtil.contains(
						themeDisplay.getPermissionChecker(), group,
						ActionKeys.VIEW_STAGING)) {

					return getGroupURL(group.getStagingGroup(), portletRequest);
				}
			}
			catch (PortalException pe) {
				_log.error(
					"Unable to check permission on group " + group.getGroupId(),
					pe);
			}
		}

		return getGroupAdministrationURL(group, portletRequest);
	}

	@Reference(unbind = "-")
	protected void setPanelAppRegistry(PanelAppRegistry panelAppRegistry) {
		_panelAppRegistry = panelAppRegistry;
	}

	@Reference(unbind = "-")
	protected void setPanelCategoryRegistry(
		PanelCategoryRegistry panelCategoryRegistry) {

		_panelCategoryRegistry = panelCategoryRegistry;
	}

	private static final Log _log = LogFactoryUtil.getLog(
		GroupURLProvider.class);

	@Reference
	private Http _http;

	@Reference
	private LayoutLocalService _layoutLocalService;

	@Reference
	private LayoutSetLocalService _layoutSetLocalService;

	private PanelAppRegistry _panelAppRegistry;
	private PanelCategoryRegistry _panelCategoryRegistry;

	@Reference
	private Portal _portal;

}