/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.segments.criteria.contributor;

import com.liferay.portal.kernel.language.LanguageUtil;
import com.liferay.portal.odata.entity.EntityModel;
import com.liferay.segments.criteria.Criteria;
import com.liferay.segments.field.Field;

import java.util.List;
import java.util.Locale;

import javax.portlet.PortletRequest;

/**
 * Provides an interface for extending the segment's {@link Criteria} by adding
 * more filters.
 *
 * @author Eduardo García
 */
public interface SegmentsCriteriaContributor {

	/**
	 * Contributes the criterion to a segment's criteria.
	 *
	 * @param criteria the segment's criteria
	 * @param filterString the criterion's filter as a string
	 * @param conjunction the criterion's conjunction
	 */
	public default void contribute(
		Criteria criteria, String filterString,
		Criteria.Conjunction conjunction) {

		criteria.addCriterion(getKey(), getType(), filterString, conjunction);
		criteria.addFilter(getType(), filterString, conjunction);
	}

	/**
	 * Returns the contributed criterion from the criteria.
	 *
	 * @param  criteria the segment's criteria
	 * @return the contributed criterion
	 */
	public default Criteria.Criterion getCriterion(Criteria criteria) {
		return criteria.getCriterion(getKey());
	}

	/**
	 * Returns the entity model associated with the contributor.
	 *
	 * @return the entity model associated with the contributor
	 */
	public EntityModel getEntityModel();

	/**
	 * Returns the name of the entity model associated with the contributor.
	 *
	 * @return the name of the entity model associated with the contributor
	 */
	public String getEntityName();

	/**
	 * Returns the list of fields that are supported by this contributor.
	 *
	 * @param  portletRequest the portlet request
	 * @return the list of fields that are supported by this contributor
	 */
	public List<Field> getFields(PortletRequest portletRequest);

	/**
	 * Returns the contributor's unique key.
	 *
	 * @return the contributor's unique key
	 */
	public String getKey();

	/**
	 * Returns the label displayed in the user interface based on the locale.
	 *
	 * @param  locale the locale to apply for the label
	 * @return the label displayed in the user interface
	 */
	public default String getLabel(Locale locale) {
		return LanguageUtil.get(locale, "contributor." + getKey());
	}

	/**
	 * Returns the contributor's type.
	 *
	 * @return the contributor's type
	 * @see    Criteria.Type
	 */
	public Criteria.Type getType();

}