/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
/*
 * Copyright 2005 Sun Microsystems, Inc. All rights reserved.
 */
/*
 * $Id: DOMKeyInfoFactory.java 1667527 2015-03-18 12:54:20Z mullan $
 */
package org.apache.jcp.xml.dsig.internal.dom;

import java.math.BigInteger;
import java.security.KeyException;
import java.security.PublicKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.List;

import javax.xml.crypto.*;
import javax.xml.crypto.dom.DOMCryptoContext;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.keyinfo.*;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * DOM-based implementation of KeyInfoFactory.
 *
 * @author Sean Mullan
 */
public final class DOMKeyInfoFactory extends KeyInfoFactory {

    public DOMKeyInfoFactory() { }

    @Override
    @SuppressWarnings("rawtypes")
    public KeyInfo newKeyInfo(List content) {
        return newKeyInfo(content, null);
    }

    @Override
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public KeyInfo newKeyInfo(List content, String id) {
        return new DOMKeyInfo(content, id);
    }

    @Override
    public KeyName newKeyName(String name) {
        return new DOMKeyName(name);
    }

    @Override
    public KeyValue newKeyValue(PublicKey key)  throws KeyException {
        String algorithm = key.getAlgorithm();
        if (algorithm.equals("DSA")) {
            return new DOMKeyValue.DSA((DSAPublicKey) key);
        } else if (algorithm.equals("RSA")) {
            return new DOMKeyValue.RSA((RSAPublicKey) key);
        } else if (algorithm.equals("EC")) {
            return new DOMKeyValue.EC((ECPublicKey) key);
        } else {
            throw new KeyException("unsupported key algorithm: " + algorithm);
        }
    }

    @Override
    public PGPData newPGPData(byte[] keyId) {
        return newPGPData(keyId, null, null);
    }

    @Override
    @SuppressWarnings({ "rawtypes", "unchecked" })
    public PGPData newPGPData(byte[] keyId, byte[] keyPacket, List other) {
        return new DOMPGPData(keyId, keyPacket, other);
    }

    @Override
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public PGPData newPGPData(byte[] keyPacket, List other) {
        return new DOMPGPData(keyPacket, other);
    }

    @Override
    public RetrievalMethod newRetrievalMethod(String uri) {
        return newRetrievalMethod(uri, null, null);
    }

    @Override
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public RetrievalMethod newRetrievalMethod(String uri, String type,
        List transforms) {
        if (uri == null) {
            throw new NullPointerException("uri must not be null");
        }
        return new DOMRetrievalMethod(uri, type, transforms);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public X509Data newX509Data(List content) {
        return new DOMX509Data(content);
    }

    @Override
    public X509IssuerSerial newX509IssuerSerial(String issuerName, 
        BigInteger serialNumber) {
        return new DOMX509IssuerSerial(issuerName, serialNumber);
    }

    @Override
    public boolean isFeatureSupported(String feature) {
        if (feature == null) {
            throw new NullPointerException();
        } else {
            return false;
        }
    }

    @Override
    public URIDereferencer getURIDereferencer() {
        return DOMURIDereferencer.INSTANCE;
    }

    @Override
    public KeyInfo unmarshalKeyInfo(XMLStructure xmlStructure) 
        throws MarshalException {
        if (xmlStructure == null) {
            throw new NullPointerException("xmlStructure cannot be null");
        }
        if (!(xmlStructure instanceof javax.xml.crypto.dom.DOMStructure)) {
            throw new ClassCastException("xmlStructure must be of type DOMStructure");
        }
        Node node =
            ((javax.xml.crypto.dom.DOMStructure) xmlStructure).getNode();
        node.normalize();

        Element element = null;
        if (node.getNodeType() == Node.DOCUMENT_NODE) {
            element = ((Document) node).getDocumentElement();
        } else if (node.getNodeType() == Node.ELEMENT_NODE) {
            element = (Element) node;
        } else {
            throw new MarshalException
                ("xmlStructure does not contain a proper Node");
        }

        // check tag
        String tag = element.getLocalName();
        String namespace = element.getNamespaceURI();
        if (tag == null || namespace == null) {
            throw new MarshalException("Document implementation must " +
                "support DOM Level 2 and be namespace aware");
        }
        if (tag.equals("KeyInfo") && XMLSignature.XMLNS.equals(namespace)) {
            return new DOMKeyInfo(element, new UnmarshalContext(), getProvider());
        } else {
            throw new MarshalException("invalid KeyInfo tag: " + namespace + ":" + tag);
        }
    }

    private static class UnmarshalContext extends DOMCryptoContext {
        UnmarshalContext() {}
    }

}
