/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.saml.opensaml.integration.internal.provider;

import com.liferay.portal.kernel.security.auth.CompanyThreadLocal;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.saml.opensaml.integration.internal.util.SamlUtil;
import com.liferay.saml.persistence.exception.NoSuchIdpSpConnectionException;
import com.liferay.saml.persistence.exception.NoSuchSpIdpConnectionException;
import com.liferay.saml.persistence.model.SamlIdpSpConnection;
import com.liferay.saml.persistence.model.SamlSpIdpConnection;
import com.liferay.saml.persistence.service.SamlIdpSpConnectionLocalService;
import com.liferay.saml.persistence.service.SamlSpIdpConnectionLocalService;
import com.liferay.saml.runtime.configuration.SamlProviderConfigurationHelper;

import java.io.StringReader;

import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;
import net.shibboleth.utilities.java.support.xml.ParserPool;

import org.opensaml.core.criterion.EntityIdCriterion;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.XMLObjectSupport;
import org.opensaml.saml.metadata.resolver.MetadataResolver;
import org.opensaml.saml.metadata.resolver.filter.MetadataFilter;
import org.opensaml.saml.metadata.resolver.impl.AbstractMetadataResolver;
import org.opensaml.saml.saml2.metadata.EntityDescriptor;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;

/**
 * @author Mika Koivisto
 */
@Component(scope = ServiceScope.PROTOTYPE, service = MetadataResolver.class)
public class DBMetadataResolver extends AbstractMetadataResolver {

	@Nonnull
	@Override
	public Iterable<EntityDescriptor> resolve(@Nullable CriteriaSet criteriaSet)
		throws ResolverException {

		if (criteriaSet == null) {
			return Collections.emptyList();
		}

		EntityIdCriterion entityIdCriterion = criteriaSet.get(
			EntityIdCriterion.class);

		if (entityIdCriterion == null) {
			throw new ResolverException("Entity ID criterion is null");
		}

		try {
			EntityDescriptor entityDescriptor = getEntityDescriptor(
				entityIdCriterion.getEntityId());

			if (isValid(entityDescriptor)) {
				return Collections.singletonList(entityDescriptor);
			}

			return Collections.emptyList();
		}
		catch (Exception e) {
			throw new ResolverException(e);
		}
	}

	@Override
	public void setParserPool(ParserPool parserPool) {
		super.setParserPool(parserPool);

		_parserPool = parserPool;
	}

	protected EntityDescriptor getEntityDescriptor(String entityID)
		throws Exception {

		return SamlUtil.getEntityDescriptorById(
			entityID, getMetadata(entityID));
	}

	protected XMLObject getMetadata(String entityID) throws Exception {
		String metadataXml = getMetadataXml(entityID);

		if (Validator.isNull(metadataXml)) {
			return null;
		}

		XMLObject metadataXmlObject = XMLObjectSupport.unmarshallFromReader(
			_parserPool, new StringReader(metadataXml));

		MetadataFilter metadataFilter = getMetadataFilter();

		if (metadataFilter != null) {
			metadataXmlObject = metadataFilter.filter(metadataXmlObject);
		}

		return metadataXmlObject;
	}

	protected String getMetadataXml(String entityId) throws Exception {
		long companyId = CompanyThreadLocal.getCompanyId();

		if (_samlProviderConfigurationHelper.isRoleIdp()) {
			try {
				SamlIdpSpConnection samlIdpSpConnection =
					_samlIdpSpConnectionLocalService.getSamlIdpSpConnection(
						companyId, entityId);

				if (!samlIdpSpConnection.isEnabled()) {
					return null;
				}

				return samlIdpSpConnection.getMetadataXml();
			}
			catch (NoSuchIdpSpConnectionException nsisce) {
				return null;
			}
		}
		else if (_samlProviderConfigurationHelper.isRoleSp()) {
			try {
				SamlSpIdpConnection samlSpIdpConnection =
					_samlSpIdpConnectionLocalService.getSamlSpIdpConnection(
						companyId, entityId);

				if (!samlSpIdpConnection.isEnabled()) {
					return null;
				}

				return samlSpIdpConnection.getMetadataXml();
			}
			catch (NoSuchSpIdpConnectionException nssice) {
				return null;
			}
		}

		return null;
	}

	@Nonnull
	@Override
	protected List<EntityDescriptor> lookupEntityID(@Nonnull String entityID)
		throws ResolverException {

		try {
			EntityDescriptor entityDescriptor = getEntityDescriptor(entityID);

			if (entityDescriptor == null) {
				return Collections.emptyList();
			}

			return Collections.singletonList(entityDescriptor);
		}
		catch (Exception e) {
			throw new ResolverException(e);
		}
	}

	@Reference
	private ParserPool _parserPool;

	@Reference
	private SamlIdpSpConnectionLocalService _samlIdpSpConnectionLocalService;

	@Reference
	private SamlProviderConfigurationHelper _samlProviderConfigurationHelper;

	@Reference
	private SamlSpIdpConnectionLocalService _samlSpIdpConnectionLocalService;

}