/**
 * SPDX-FileCopyrightText: (c) 2025 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.headless.admin.site.internal.dto.v1_0.util;

import com.liferay.fragment.entry.processor.constants.FragmentEntryProcessorConstants;
import com.liferay.fragment.entry.processor.util.EditableFragmentEntryProcessorUtil;
import com.liferay.fragment.model.FragmentEntryLink;
import com.liferay.headless.admin.site.dto.v1_0.FragmentEditableElement;
import com.liferay.headless.admin.site.dto.v1_0.FragmentEditableElementValue;
import com.liferay.headless.admin.site.dto.v1_0.FragmentInlineValue;
import com.liferay.headless.admin.site.dto.v1_0.FragmentMappedValue;
import com.liferay.headless.admin.site.dto.v1_0.TextFragmentEditableElementValue;
import com.liferay.headless.admin.site.dto.v1_0.TextFragmentValue;
import com.liferay.headless.admin.site.dto.v1_0.TextInlineFragmentValue;
import com.liferay.headless.admin.site.dto.v1_0.TextMappedFragmentValue;
import com.liferay.info.item.InfoItemServiceRegistry;
import com.liferay.petra.function.transform.TransformUtil;
import com.liferay.portal.kernel.json.JSONFactoryUtil;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.json.JSONUtil;
import com.liferay.portal.kernel.util.MapUtil;
import com.liferay.portal.vulcan.util.LocalizedMapUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author Rubén Pulido
 */
public class FragmentEditableElementUtil {

	public static FragmentEditableElement[] getFragmentEditableElements(
		long companyId, FragmentEntryLink fragmentEntryLink,
		InfoItemServiceRegistry infoItemServiceRegistry, long scopeGroupId) {

		JSONObject editableValuesJSONObject =
			fragmentEntryLink.getEditableValuesJSONObject();

		if (editableValuesJSONObject == null) {
			return null;
		}

		List<FragmentEditableElement> fragmentEditableElements =
			new ArrayList<>();

		JSONObject editableFragmentEntryProcessorJSONObject =
			editableValuesJSONObject.getJSONObject(
				FragmentEntryProcessorConstants.
					KEY_EDITABLE_FRAGMENT_ENTRY_PROCESSOR);

		if (editableFragmentEntryProcessorJSONObject == null) {
			return fragmentEditableElements.toArray(
				new FragmentEditableElement[0]);
		}

		fragmentEditableElements.addAll(
			_getTextFragmentEditableElements(
				companyId,
				EditableFragmentEntryProcessorUtil.getEditableTypes(
					fragmentEntryLink.getHtml()),
				infoItemServiceRegistry,
				editableFragmentEntryProcessorJSONObject, scopeGroupId));

		return fragmentEditableElements.toArray(new FragmentEditableElement[0]);
	}

	public static JSONObject
		getFragmentEditableElementsEditableValuesJSONObject(
			long companyId, FragmentEditableElement[] fragmentEditableElements,
			InfoItemServiceRegistry infoItemServiceRegistry,
			long scopeGroupId) {

		return JSONUtil.put(
			FragmentEntryProcessorConstants.
				KEY_EDITABLE_FRAGMENT_ENTRY_PROCESSOR,
			() -> {
				JSONObject editableFragmentEntryProcessorJSONObject =
					_getEditableFragmentEntryProcessorJSONObject(
						companyId, fragmentEditableElements,
						infoItemServiceRegistry, scopeGroupId);

				if (editableFragmentEntryProcessorJSONObject.length() > 0) {
					return editableFragmentEntryProcessorJSONObject;
				}

				return null;
			});
	}

	private static JSONObject _getEditableFragmentEntryProcessorJSONObject(
			long companyId, FragmentEditableElement[] fragmentEditableElements,
			InfoItemServiceRegistry infoItemServiceRegistry, long scopeGroupId)
		throws Exception {

		JSONObject jsonObject = JSONFactoryUtil.createJSONObject();

		if (fragmentEditableElements == null) {
			return jsonObject;
		}

		for (FragmentEditableElement fragmentEditableElement :
				fragmentEditableElements) {

			if ((fragmentEditableElement == null) ||
				(fragmentEditableElement.getId() == null)) {

				continue;
			}

			FragmentEditableElementValue fragmentEditableElementValue =
				fragmentEditableElement.getFragmentEditableElementValue();

			if ((fragmentEditableElementValue == null) ||
				(fragmentEditableElementValue.getType() !=
					FragmentEditableElementValue.Type.TEXT)) {

				jsonObject.put(
					fragmentEditableElement.getId(), (JSONObject)null);

				continue;
			}

			jsonObject.put(
				fragmentEditableElement.getId(),
				_getFragmentEditableElementJSONObject(
					companyId, infoItemServiceRegistry, scopeGroupId,
					(TextFragmentEditableElementValue)
						fragmentEditableElementValue));
		}

		return jsonObject;
	}

	private static JSONObject _getFragmentEditableElementJSONObject(
			long companyId, InfoItemServiceRegistry infoItemServiceRegistry,
			long scopeGroupId,
			TextFragmentEditableElementValue textFragmentEditableElementValue)
		throws Exception {

		JSONObject jsonObject = JSONFactoryUtil.createJSONObject();

		if (textFragmentEditableElementValue == null) {
			return jsonObject;
		}

		TextFragmentValue textFragmentValue =
			textFragmentEditableElementValue.getTextFragmentValue();

		if (textFragmentValue == null) {
			return jsonObject;
		}

		if (textFragmentValue instanceof TextInlineFragmentValue) {
			TextInlineFragmentValue textInlineFragmentValue =
				(TextInlineFragmentValue)textFragmentValue;

			FragmentInlineValue fragmentInlineValue =
				textInlineFragmentValue.getFragmentInlineValue();

			if (fragmentInlineValue == null) {
				return jsonObject;
			}

			Map<String, String> languageIdMap =
				LocalizedMapUtil.getLanguageIdMap(
					LocalizedMapUtil.getLocalizedMap(
						fragmentInlineValue.getValue_i18n()));

			for (Map.Entry<String, String> entry : languageIdMap.entrySet()) {
				jsonObject.put(entry.getKey(), entry.getValue());
			}

			return jsonObject;
		}

		if (!(textFragmentValue instanceof TextMappedFragmentValue)) {
			return jsonObject;
		}

		TextMappedFragmentValue textMappedFragmentValue =
			(TextMappedFragmentValue)textFragmentValue;

		FragmentMappedValue fragmentMappedValue =
			textMappedFragmentValue.getFragmentMappedValue();

		if (fragmentMappedValue == null) {
			return jsonObject;
		}

		return FragmentMappingUtil.getFragmentMappedValueJSONObject(
			companyId, infoItemServiceRegistry,
			fragmentMappedValue.getMapping(), scopeGroupId);
	}

	private static List<FragmentEditableElement>
		_getTextFragmentEditableElements(
			long companyId, Map<String, String> editableTypes,
			final InfoItemServiceRegistry infoItemServiceRegistry,
			JSONObject jsonObject, long scopeGroupId) {

		return TransformUtil.transform(
			jsonObject.keySet(),
			textId -> new FragmentEditableElement() {
				{
					setFragmentEditableElementValue(
						() -> {
							String type = editableTypes.getOrDefault(
								textId, "text");

							if (!Objects.equals(type, "text")) {
								return null;
							}

							return _toTextFragmentEditableElementValue(
								companyId, infoItemServiceRegistry,
								jsonObject.getJSONObject(textId), scopeGroupId);
						});
					setId(() -> textId);
				}
			});
	}

	private static TextFragmentEditableElementValue
		_toTextFragmentEditableElementValue(
			long companyId, InfoItemServiceRegistry infoItemServiceRegistry,
			JSONObject jsonObject, long scopeGroupId) {

		if (jsonObject == null) {
			return null;
		}

		TextFragmentEditableElementValue textFragmentEditableElementValue =
			new TextFragmentEditableElementValue();

		textFragmentEditableElementValue.setTextFragmentValue(
			() -> _toTextFragmentValue(
				companyId, infoItemServiceRegistry, jsonObject, scopeGroupId));

		return textFragmentEditableElementValue;
	}

	private static TextFragmentValue _toTextFragmentValue(
		long companyId, InfoItemServiceRegistry infoItemServiceRegistry,
		JSONObject jsonObject, long scopeGroupId) {

		if (jsonObject == null) {
			return null;
		}

		if (FragmentMappingUtil.isMappedValue(jsonObject)) {
			TextMappedFragmentValue textMappedFragmentValue =
				new TextMappedFragmentValue();

			textMappedFragmentValue.setFragmentMappedValue(
				() -> FragmentMappingUtil.toFragmentMappedValue(
					companyId, infoItemServiceRegistry, jsonObject,
					scopeGroupId));

			return textMappedFragmentValue;
		}

		Map<String, String> i18nMap = LocalizedMapUtil.getI18nMap(
			true,
			LocalizedMapUtil.populateLocalizedMap(
				JSONUtil.toStringMap(jsonObject)));

		if (MapUtil.isEmpty(i18nMap)) {
			return null;
		}

		TextInlineFragmentValue textInlineFragmentValue =
			new TextInlineFragmentValue();

		textInlineFragmentValue.setFragmentInlineValue(
			() -> {
				FragmentInlineValue fragmentInlineValue =
					new FragmentInlineValue();

				fragmentInlineValue.setValue_i18n(() -> i18nMap);

				return fragmentInlineValue;
			});

		return textInlineFragmentValue;
	}

}