/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.frontend.token.definition.internal;

import com.liferay.client.extension.constants.ClientExtensionEntryConstants;
import com.liferay.client.extension.model.ClientExtensionEntryRel;
import com.liferay.client.extension.service.ClientExtensionEntryRelLocalService;
import com.liferay.client.extension.type.ThemeCSSCET;
import com.liferay.frontend.token.definition.FrontendTokenDefinition;
import com.liferay.frontend.token.definition.FrontendTokenDefinitionRegistry;
import com.liferay.frontend.token.definition.constants.FrontendTokenDefinitionConstants;
import com.liferay.frontend.token.definition.internal.validator.FrontendTokenDefinitionJSONValidator;
import com.liferay.osgi.util.ServiceTrackerFactory;
import com.liferay.petra.concurrent.DCLSingleton;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.json.validator.JSONValidatorException;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.feature.flag.FeatureFlagManagerUtil;
import com.liferay.portal.kernel.json.JSONException;
import com.liferay.portal.kernel.json.JSONFactory;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.Layout;
import com.liferay.portal.kernel.model.LayoutSet;
import com.liferay.portal.kernel.model.PortletConstants;
import com.liferay.portal.kernel.model.Theme;
import com.liferay.portal.kernel.resource.bundle.ResourceBundleLoader;
import com.liferay.portal.kernel.resource.bundle.ResourceBundleLoaderUtil;
import com.liferay.portal.kernel.util.HashMapBuilder;
import com.liferay.portal.kernel.util.Portal;
import com.liferay.portal.kernel.util.URLUtil;
import com.liferay.portal.kernel.util.Validator;

import java.io.IOException;

import java.net.URL;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.util.tracker.BundleTracker;
import org.osgi.util.tracker.BundleTrackerCustomizer;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;

/**
 * @author Iván Zaera
 */
@Component(service = FrontendTokenDefinitionRegistry.class)
public class FrontendTokenDefinitionRegistryImpl
	implements FrontendTokenDefinitionRegistry {

	@Override
	public FrontendTokenDefinition getFrontendTokenDefinition(Layout layout) {
		String cetExternalReferenceCode = null;

		if (FeatureFlagManagerUtil.isEnabled(
				layout.getCompanyId(), "LPD-30204")) {

			cetExternalReferenceCode = _getCETExternalReferenceCode(
				layout.getClassNameId(), layout.getPlid());

			if (cetExternalReferenceCode != null) {
				return _getThemeCSSCETFrontendTokenDefinition(
					layout.getCompanyId(), cetExternalReferenceCode);
			}

			if (layout.getMasterLayoutPlid() > 0) {
				cetExternalReferenceCode = _getCETExternalReferenceCode(
					_portal.getClassNameId(Layout.class),
					layout.getMasterLayoutPlid());

				if (cetExternalReferenceCode != null) {
					return _getThemeCSSCETFrontendTokenDefinition(
						layout.getCompanyId(), cetExternalReferenceCode);
				}
			}
		}

		LayoutSet layoutSet = layout.getLayoutSet();

		cetExternalReferenceCode = _getCETExternalReferenceCode(
			_portal.getClassNameId(LayoutSet.class),
			layoutSet.getLayoutSetId());

		if (cetExternalReferenceCode != null) {
			return _getThemeCSSCETFrontendTokenDefinition(
				layoutSet.getCompanyId(), cetExternalReferenceCode);
		}

		Theme theme = null;

		try {
			theme = layout.getTheme();

			return _getBundleFrontendTokenDefinition(theme.getThemeId());
		}
		catch (PortalException portalException) {
			_log.error(
				"Unable to get the theme for layout with layout ID " +
					layout.getLayoutId(),
				portalException);
		}

		return null;
	}

	public FrontendTokenDefinition getFrontendTokenDefinition(
		LayoutSet layoutSet) {

		return _getFrontendTokenDefinition(
			layoutSet.getCompanyId(),
			_getCETExternalReferenceCode(
				_portal.getClassNameId(LayoutSet.class),
				layoutSet.getLayoutSetId()),
			layoutSet.getThemeId());
	}

	@Override
	public FrontendTokenDefinition getFrontendTokenDefinition(
		long companyId, String themeId) {

		FrontendTokenDefinition frontendTokenDefinition =
			_frontendTokenDefinitions.get(themeId);

		if (frontendTokenDefinition != null) {
			return frontendTokenDefinition;
		}

		Map<String, FrontendTokenDefinition> frontendTokenDefinitions =
			_frontendTokenDefinitionsMap.get(companyId);

		if (frontendTokenDefinitions == null) {
			return null;
		}

		return frontendTokenDefinitions.get(themeId);
	}

	@Override
	public List<FrontendTokenDefinition> getFrontendTokenDefinitions(
		long companyId) {

		List<FrontendTokenDefinition> frontendTokenDefinitions =
			new ArrayList<>(_frontendTokenDefinitions.values());

		Map<String, FrontendTokenDefinition> frontendTokenDefinitionsMap =
			_frontendTokenDefinitionsMap.get(companyId);

		if (frontendTokenDefinitionsMap != null) {
			frontendTokenDefinitions.addAll(
				frontendTokenDefinitionsMap.values());
		}

		return frontendTokenDefinitions;
	}

	@Activate
	protected void activate(BundleContext bundleContext) {
		_bundleTracker = new BundleTracker<>(
			bundleContext, Bundle.ACTIVE, _bundleTrackerCustomizer);

		_serviceTracker = ServiceTrackerFactory.open(
			bundleContext, ThemeCSSCET.class,
			new ServiceTrackerCustomizer<ThemeCSSCET, ThemeCSSCET>() {

				@Override
				public ThemeCSSCET addingService(
					ServiceReference<ThemeCSSCET> serviceReference) {

					ThemeCSSCET themeCSSCET = bundleContext.getService(
						serviceReference);

					if (Validator.isNull(
							themeCSSCET.getFrontendTokenDefinitionJSON())) {

						return themeCSSCET;
					}

					_addingService(themeCSSCET);

					return themeCSSCET;
				}

				@Override
				public void modifiedService(
					ServiceReference<ThemeCSSCET> serviceReference,
					ThemeCSSCET themeCSSCET) {
				}

				@Override
				public void removedService(
					ServiceReference<ThemeCSSCET> serviceReference,
					ThemeCSSCET themeCSSCET) {

					bundleContext.ungetService(serviceReference);

					_removedService(themeCSSCET);
				}

			});
	}

	@Deactivate
	protected void deactivate() {
		_bundleTracker.close();

		_serviceTracker.close();
	}

	protected List<FrontendTokenDefinitionImpl> getFrontendTokenDefinitionImpls(
		Bundle bundle) {

		String json = _getFrontendTokenDefinitionJSON(bundle);

		if (json == null) {
			return Collections.emptyList();
		}

		try {
			ResourceBundleLoader resourceBundleLoader =
				ResourceBundleLoaderUtil.
					getResourceBundleLoaderByBundleSymbolicName(
						bundle.getSymbolicName());

			if (resourceBundleLoader == null) {
				resourceBundleLoader =
					ResourceBundleLoaderUtil.getPortalResourceBundleLoader();
			}

			List<FrontendTokenDefinitionImpl> frontendTokenDefinitionImpls =
				new ArrayList<>();

			for (Map<String, String> themeMap : getThemeMaps(bundle)) {
				frontendTokenDefinitionImpls.add(
					new FrontendTokenDefinitionImpl(
						jsonFactory.createJSONObject(json), jsonFactory,
						resourceBundleLoader, themeMap.get("id"),
						themeMap.get("name"),
						FrontendTokenDefinitionConstants.THEME_TYPE_BUNDLE));
			}

			return frontendTokenDefinitionImpls;
		}
		catch (JSONException | RuntimeException exception) {
			_log.error(
				"Unable to parse frontend token definitions for bundle " +
					bundle.getSymbolicName(),
				exception);
		}

		return Collections.emptyList();
	}

	protected String getServletContextName(Bundle bundle) {
		Dictionary<String, String> headers = bundle.getHeaders(
			StringPool.BLANK);

		String webContextPath = headers.get("Web-ContextPath");

		if (webContextPath == null) {
			return null;
		}

		if (webContextPath.startsWith(StringPool.SLASH)) {
			webContextPath = webContextPath.substring(1);
		}

		return webContextPath;
	}

	protected List<Map<String, String>> getThemeMaps(Bundle bundle) {
		URL url = bundle.getEntry("WEB-INF/liferay-look-and-feel.xml");

		if (url == null) {
			return Collections.emptyList();
		}

		try {
			List<Map<String, String>> themeMaps = new ArrayList<>();

			String servletContextName = getServletContextName(bundle);

			String xml = URLUtil.toString(url);

			xml = xml.replaceAll(StringPool.NEW_LINE, StringPool.SPACE);

			Matcher matcher = _themePattern.matcher(xml);

			while (matcher.find()) {
				String themeId = matcher.group(1);

				if (servletContextName != null) {
					themeId +=
						PortletConstants.WAR_SEPARATOR + servletContextName;
				}

				themeMaps.add(
					HashMapBuilder.put(
						"id", portal.getJsSafePortletId(themeId)
					).put(
						"name", matcher.group(2)
					).build());
			}

			return themeMaps;
		}
		catch (IOException ioException) {
			throw new RuntimeException(
				"Unable to read WEB-INF/liferay-look-and-feel.xml",
				ioException);
		}
	}

	@Reference
	protected JSONFactory jsonFactory;

	@Reference
	protected Portal portal;

	private void _addingService(ThemeCSSCET themeCSSCET) {
		try {
			_frontendTokenDefinitionJSONValidator.validate(
				themeCSSCET.getFrontendTokenDefinitionJSON());

			Map<String, FrontendTokenDefinition> frontendTokenDefinitions =
				_frontendTokenDefinitionsMap.computeIfAbsent(
					themeCSSCET.getCompanyId(),
					entry -> new ConcurrentHashMap<>());

			frontendTokenDefinitions.put(
				themeCSSCET.getExternalReferenceCode(),
				new FrontendTokenDefinitionImpl(
					jsonFactory.createJSONObject(
						themeCSSCET.getFrontendTokenDefinitionJSON()),
					jsonFactory,
					ResourceBundleLoaderUtil.getPortalResourceBundleLoader(),
					themeCSSCET.getExternalReferenceCode(),
					themeCSSCET.getName(),
					FrontendTokenDefinitionConstants.THEME_TYPE_THEME_CSS_CET));
		}
		catch (JSONException | JSONValidatorException exception) {
			_log.error(
				"Unable to parse theme CSS client extension frontend token " +
					"definition",
				exception);
		}
	}

	private FrontendTokenDefinition _getBundleFrontendTokenDefinition(
		String themeId) {

		Map<String, FrontendTokenDefinition> frontendTokenDefinitions =
			_frontendTokenDefinitionsDCLSingleton.getSingleton(
				() -> {
					_bundleTracker.open();

					return _frontendTokenDefinitions;
				});

		return frontendTokenDefinitions.get(themeId);
	}

	private String _getCETExternalReferenceCode(
		long classNameId, long classPK) {

		ClientExtensionEntryRel clientExtensionEntryRel =
			_clientExtensionEntryRelLocalService.fetchClientExtensionEntryRel(
				classNameId, classPK,
				ClientExtensionEntryConstants.TYPE_THEME_CSS);

		if (clientExtensionEntryRel == null) {
			return null;
		}

		return clientExtensionEntryRel.getCETExternalReferenceCode();
	}

	private FrontendTokenDefinition _getFrontendTokenDefinition(
		long companyId, String externalReferenceCode, String themeId) {

		if (externalReferenceCode != null) {
			FrontendTokenDefinition frontendTokenDefinition =
				_getThemeCSSCETFrontendTokenDefinition(
					companyId, externalReferenceCode);

			if (frontendTokenDefinition != null) {
				return frontendTokenDefinition;
			}
		}

		return _getBundleFrontendTokenDefinition(themeId);
	}

	private String _getFrontendTokenDefinitionJSON(Bundle bundle) {
		URL url = bundle.getEntry("WEB-INF/frontend-token-definition.json");

		if (url == null) {
			return null;
		}

		try {
			return URLUtil.toString(url);
		}
		catch (IOException ioException) {
			throw new RuntimeException(
				"Unable to read WEB-INF/frontend-token-definition.json",
				ioException);
		}
	}

	private Map<String, FrontendTokenDefinition> _getFrontendTokenDefinitions(
		long companyId) {

		return _frontendTokenDefinitionsMap.getOrDefault(
			companyId, new ConcurrentHashMap<>());
	}

	private FrontendTokenDefinition _getThemeCSSCETFrontendTokenDefinition(
		long companyId, String externalReferenceCode) {

		Map<String, FrontendTokenDefinition> frontendTokenDefinitions =
			_getFrontendTokenDefinitions(companyId);

		return frontendTokenDefinitions.get(externalReferenceCode);
	}

	private void _removedService(ThemeCSSCET themeCSSCET) {
		Map<String, FrontendTokenDefinition> frontendTokenDefinitions =
			_getFrontendTokenDefinitions(themeCSSCET.getCompanyId());

		frontendTokenDefinitions.remove(themeCSSCET.getExternalReferenceCode());
	}

	private static final Log _log = LogFactoryUtil.getLog(
		FrontendTokenDefinitionRegistryImpl.class);

	private static final Pattern _themePattern = Pattern.compile(
		"<theme id=\"([^\"]*)\"[^>]* name=\"([^\"]*)\"[^>]*>");

	private BundleTracker<List<FrontendTokenDefinitionImpl>> _bundleTracker;

	private final BundleTrackerCustomizer<List<FrontendTokenDefinitionImpl>>
		_bundleTrackerCustomizer =
			new BundleTrackerCustomizer<List<FrontendTokenDefinitionImpl>>() {

				@Override
				public List<FrontendTokenDefinitionImpl> addingBundle(
					Bundle bundle, BundleEvent bundleEvent) {

					List<FrontendTokenDefinitionImpl>
						frontendTokenDefinitionImpls =
							getFrontendTokenDefinitionImpls(bundle);

					for (FrontendTokenDefinitionImpl
							frontendTokenDefinitionImpl :
								frontendTokenDefinitionImpls) {

						if (frontendTokenDefinitionImpl.getThemeId() == null) {
							continue;
						}

						_frontendTokenDefinitions.put(
							frontendTokenDefinitionImpl.getThemeId(),
							frontendTokenDefinitionImpl);
					}

					return frontendTokenDefinitionImpls;
				}

				@Override
				public void modifiedBundle(
					Bundle bundle, BundleEvent bundleEvent,
					List<FrontendTokenDefinitionImpl>
						frontendTokenDefinitionImpls) {
				}

				@Override
				public void removedBundle(
					Bundle bundle, BundleEvent bundleEvent,
					List<FrontendTokenDefinitionImpl>
						frontendTokenDefinitionImpls) {

					for (FrontendTokenDefinitionImpl
							frontendTokenDefinitionImpl :
								frontendTokenDefinitionImpls) {

						_frontendTokenDefinitions.remove(
							frontendTokenDefinitionImpl.getThemeId());
					}
				}

			};

	@Reference
	private ClientExtensionEntryRelLocalService
		_clientExtensionEntryRelLocalService;

	private final FrontendTokenDefinitionJSONValidator
		_frontendTokenDefinitionJSONValidator =
			new FrontendTokenDefinitionJSONValidator();
	private final Map<String, FrontendTokenDefinition>
		_frontendTokenDefinitions = new ConcurrentHashMap<>();
	private final DCLSingleton<Map<String, FrontendTokenDefinition>>
		_frontendTokenDefinitionsDCLSingleton = new DCLSingleton<>();
	private final Map<Long, Map<String, FrontendTokenDefinition>>
		_frontendTokenDefinitionsMap = new ConcurrentHashMap<>();

	@Reference
	private Portal _portal;

	private ServiceTracker<ThemeCSSCET, ThemeCSSCET> _serviceTracker;

}