/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import ClayDatePicker from '@clayui/date-picker';
import ClayForm from '@clayui/form';
import ClayModal from '@clayui/modal';
import classNames from 'classnames';
import {format, getYear, isBefore, isEqual} from 'date-fns';
import React, {useEffect, useState} from 'react';

import {IDateFilter, IField, IFilter} from '../../../utils/types';
import Configuration from './Configuration';
import Footer from './Footer';

function Header() {
	return <>{Liferay.Language.get('new-date-range-filter')}</>;
}

interface IBodyProps {
	closeModal: Function;
	fieldNames?: string[];
	fields: IField[];
	filter?: IFilter;
	namespace: string;
	onSave: Function;
}

function Body({
	closeModal,
	fieldNames,
	fields,
	filter,
	namespace,
	onSave,
}: IBodyProps) {
	const [fieldInUseValidationError, setFieldInUseValidationError] =
		useState<boolean>(false);
	const [fieldValidationError, setFieldValidationError] =
		useState<boolean>(false);
	const [labelValidationError, setLabelValidationError] = useState(false);

	const [saveButtonDisabled, setSaveButtonDisabled] =
		useState<boolean>(false);
	const fdsFilterLabelTranslations = filter?.label_i18n ?? {};
	const [i18nFilterLabels, setI18nFilterLabels] = useState(
		fdsFilterLabelTranslations
	);

	const inUseFields: (string | undefined)[] = fields.map((item) =>
		fieldNames?.includes(item.name) ? item.name : undefined
	);

	const [selectedField, setSelectedField] = useState<IField | undefined>(
		fields.find((item) => item.name === filter?.fieldName)
	);
	const [from, setFrom] = useState<string>(
		(filter as IDateFilter)?.from ?? ''
	);
	const [to, setTo] = useState<string>((filter as IDateFilter)?.to ?? '');
	const [isValidDateRange, setIsValidDateRange] = useState<boolean>(true);

	const fromFormElementId = `${namespace}From`;
	const toFormElementId = `${namespace}To`;

	useEffect(() => {
		let isValid = true;

		const dateTo = new Date(to);

		const dateFrom = new Date(from);

		if (to && from) {
			isValid = isBefore(dateFrom, dateTo) || isEqual(dateFrom, dateTo);
		}

		setIsValidDateRange(isValid);
	}, [from, to]);

	const isi18nFilterLabelsValid = (
		i18nFilterLabels: Partial<Liferay.Language.FullyLocalizedValue<string>>
	) => {
		let isValid = true;

		if (!i18nFilterLabels || !Object.values(i18nFilterLabels).length) {
			isValid = false;
		}

		Object.values(i18nFilterLabels).forEach((value) => {
			if (!value) {
				isValid = false;
			}
		});

		return isValid;
	};

	const validate = () => {
		let isValid = true;

		if (Liferay.FeatureFlags['LPD-10754']) {
			const isLabelValid = isi18nFilterLabelsValid(i18nFilterLabels);
			setLabelValidationError(!isLabelValid);

			isValid = isLabelValid;
		}

		if (!selectedField) {
			setFieldValidationError(true);

			isValid = false;
		}

		if (selectedField && !filter) {
			if (inUseFields.includes(selectedField?.name)) {
				setFieldInUseValidationError(true);

				isValid = false;
			}
		}

		const dateTo = new Date(to);
		const dateFrom = new Date(from);

		if (to && from) {
			const isValidRange =
				isBefore(dateFrom, dateTo) || isEqual(dateFrom, dateTo);

			setIsValidDateRange(isValidRange);

			if (!isValidRange) {
				isValid = false;
			}
		}

		return isValid;
	};

	const saveDateRangeFilter = () => {
		setSaveButtonDisabled(true);

		const success = validate();

		if (success) {
			const formData = {
				fieldName: selectedField?.name,
				from,
				label_i18n: i18nFilterLabels,
				to,
				type: selectedField?.format,
			};

			onSave(formData);
		}
		else {
			setSaveButtonDisabled(false);
		}
	};

	return (
		<>
			<ClayModal.Body>
				<Configuration
					fieldInUseValidationError={fieldInUseValidationError}
					fieldNames={fieldNames}
					fieldValidationError={fieldValidationError}
					fields={fields}
					filter={filter}
					labelValidationError={labelValidationError}
					namespace={namespace}
					onBlur={() => {
						setLabelValidationError(
							!isi18nFilterLabelsValid(i18nFilterLabels)
						);
					}}
					onChangeField={(newValue) => {
						setSelectedField(newValue);
						setFieldValidationError(!newValue);
						setFieldInUseValidationError(
							newValue
								? inUseFields.includes(newValue.name)
								: false
						);
					}}
					onChangeLabel={(newValue) => {
						setI18nFilterLabels(newValue);
					}}
				/>

				{!fieldInUseValidationError && (
					<ClayForm.Group className="form-group-autofit">
						<div
							className={classNames('form-group-item', {
								'has-error': !isValidDateRange,
							})}
						>
							<label htmlFor={fromFormElementId}>
								{Liferay.Language.get('from')}
							</label>

							<ClayDatePicker
								inputName={fromFormElementId}
								onChange={(value: any) => {
									setFrom(value);
								}}
								placeholder="YYYY-MM-DD"
								value={
									from
										? format(new Date(from), 'yyyy-MM-dd')
										: ''
								}
								years={{
									end: getYear(new Date()) + 25,
									start: getYear(new Date()) - 50,
								}}
							/>

							{!isValidDateRange && (
								<ClayForm.FeedbackGroup>
									<ClayForm.FeedbackItem>
										<ClayForm.FeedbackIndicator symbol="exclamation-full" />

										{Liferay.Language.get(
											'date-range-is-invalid.-from-must-be-before-to'
										)}
									</ClayForm.FeedbackItem>
								</ClayForm.FeedbackGroup>
							)}
						</div>

						<div className="form-group-item">
							<label htmlFor={toFormElementId}>
								{Liferay.Language.get('to')}
							</label>

							<ClayDatePicker
								inputName={toFormElementId}
								onChange={(value: any) => {
									setTo(value);
								}}
								placeholder="YYYY-MM-DD"
								value={
									to ? format(new Date(to), 'yyyy-MM-dd') : ''
								}
								years={{
									end: getYear(new Date()) + 25,
									start: getYear(new Date()) - 50,
								}}
							/>
						</div>
					</ClayForm.Group>
				)}
			</ClayModal.Body>

			<Footer
				closeModal={closeModal}
				onSave={saveDateRangeFilter}
				saveButtonDisabled={saveButtonDisabled}
			/>
		</>
	);
}

export default {
	Body,
	Header,
};
