/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.dynamic.data.mapping.internal.util;

import com.liferay.dynamic.data.mapping.model.DDMForm;
import com.liferay.dynamic.data.mapping.model.DDMFormField;
import com.liferay.dynamic.data.mapping.model.DDMStructure;
import com.liferay.dynamic.data.mapping.model.LocalizedValue;
import com.liferay.dynamic.data.mapping.model.UnlocalizedValue;
import com.liferay.dynamic.data.mapping.model.Value;
import com.liferay.dynamic.data.mapping.storage.DDMFormFieldValue;
import com.liferay.dynamic.data.mapping.storage.DDMFormValues;
import com.liferay.dynamic.data.mapping.storage.Field;
import com.liferay.dynamic.data.mapping.storage.Fields;
import com.liferay.dynamic.data.mapping.util.DDMFieldsCounter;
import com.liferay.dynamic.data.mapping.util.FieldsToDDMFormValuesConverter;
import com.liferay.dynamic.data.mapping.util.NumericDDMFormFieldUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.util.StringUtil;

import java.io.Serializable;

import java.text.DecimalFormat;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.osgi.service.component.annotations.Component;

/**
 * @author Marcellus Tavares
 */
@Component(immediate = true, service = FieldsToDDMFormValuesConverter.class)
public class FieldsToDDMFormValuesConverterImpl
	implements FieldsToDDMFormValuesConverter {

	@Override
	public DDMFormValues convert(DDMStructure ddmStructure, Fields fields)
		throws PortalException {

		DDMForm ddmForm = ddmStructure.getFullHierarchyDDMForm();

		Map<String, DDMFormField> ddmFormFieldsMap =
			ddmForm.getDDMFormFieldsMap(true);

		DDMFormValues ddmFormValues = createDDMFormValues(
			ddmForm, fields.getAvailableLocales(), fields.getDefaultLocale());

		DDMFieldsCounter ddmFieldsCounter = new DDMFieldsCounter();

		for (String fieldName :
				_getDDMFormFieldNames(ddmForm.getDDMFormFields())) {

			int repetitions = _countDDMFieldRepetitions(
				ddmFormFieldsMap, fields, fieldName, null, -1);

			for (int i = 0; i < repetitions; i++) {
				DDMFormFieldValue ddmFormFieldValue = createDDMFormFieldValue(
					fieldName);

				DDMFormField ddmFormField = ddmFormFieldsMap.get(fieldName);

				if (ddmFormField != null) {
					ddmFormFieldValue.setFieldReference(
						ddmFormField.getFieldReference());
				}

				_setDDMFormFieldValueProperties(
					ddmFormFieldValue, ddmFormFieldsMap, fields,
					ddmFieldsCounter);

				ddmFormValues.addDDMFormFieldValue(ddmFormFieldValue);
			}
		}

		return ddmFormValues;
	}

	protected DDMFormFieldValue createDDMFormFieldValue(String name) {
		DDMFormFieldValue ddmFormFieldValue = new DDMFormFieldValue();

		ddmFormFieldValue.setName(name);

		return ddmFormFieldValue;
	}

	protected DDMFormValues createDDMFormValues(
		DDMForm ddmForm, Set<Locale> availableLocales, Locale defaultLocale) {

		DDMFormValues ddmFormValues = new DDMFormValues(ddmForm);

		ddmFormValues.setAvailableLocales(availableLocales);
		ddmFormValues.setDefaultLocale(defaultLocale);

		return ddmFormValues;
	}

	protected String[] splitFieldsDisplayValue(Field fieldsDisplayField) {
		String value = (String)fieldsDisplayField.getValue();

		return StringUtil.split(value);
	}

	private int _countDDMFieldRepetitions(
			Map<String, DDMFormField> ddmFormFieldsMap, Fields ddmFields,
			String fieldName, String parentFieldName, int parentOffset)
		throws PortalException {

		Field ddmFieldsDisplayField = ddmFields.get(
			DDMImpl.FIELDS_DISPLAY_NAME);

		if (ddmFieldsDisplayField == null) {
			if (ddmFields.contains(fieldName)) {
				return 1;
			}

			return 0;
		}

		String[] ddmFieldsDisplayValues = _getDDMFieldsDisplayValues(
			ddmFormFieldsMap, ddmFieldsDisplayField);

		int offset = -1;

		int repetitions = 0;

		for (String fieldDisplayName : ddmFieldsDisplayValues) {
			if (offset > parentOffset) {
				break;
			}

			if (fieldDisplayName.equals(parentFieldName)) {
				offset++;
			}

			if (fieldDisplayName.equals(fieldName) &&
				(offset == parentOffset)) {

				repetitions++;
			}
		}

		return repetitions;
	}

	private String _getDDMFieldInstanceId(
		Fields ddmFields, String fieldName, int index) {

		Field ddmFieldsDisplayField = ddmFields.get(
			DDMImpl.FIELDS_DISPLAY_NAME);

		if (ddmFieldsDisplayField == null) {
			return StringUtil.randomString();
		}

		String prefix = fieldName.concat(DDMImpl.INSTANCE_SEPARATOR);

		String[] ddmFieldsDisplayValues = StringUtil.split(
			(String)ddmFieldsDisplayField.getValue());

		for (String ddmFieldsDisplayValue : ddmFieldsDisplayValues) {
			if (ddmFieldsDisplayValue.startsWith(prefix)) {
				index--;

				if (index < 0) {
					return StringUtil.extractLast(
						ddmFieldsDisplayValue, DDMImpl.INSTANCE_SEPARATOR);
				}
			}
		}

		return null;
	}

	private String[] _getDDMFieldsDisplayValues(
			Map<String, DDMFormField> ddmFormFieldsMap,
			Field ddmFieldsDisplayField)
		throws PortalException {

		try {
			List<String> fieldsDisplayValues = new ArrayList<>();

			String[] values = splitFieldsDisplayValue(ddmFieldsDisplayField);

			for (String value : values) {
				String fieldName = StringUtil.extractFirst(
					value, DDMImpl.INSTANCE_SEPARATOR);

				if (ddmFormFieldsMap.containsKey(fieldName)) {
					fieldsDisplayValues.add(fieldName);
				}
			}

			return fieldsDisplayValues.toArray(new String[0]);
		}
		catch (Exception exception) {
			throw new PortalException(exception);
		}
	}

	private String _getDDMFieldValueString(
		Field ddmField, Locale locale, int index) {

		Serializable fieldValue = ddmField.getValue(locale, index);

		if (fieldValue instanceof Date) {
			Date valueDate = (Date)fieldValue;

			fieldValue = valueDate.getTime();
		}
		else if ((fieldValue instanceof Number) &&
				 !(fieldValue instanceof Integer)) {

			DecimalFormat decimalFormat =
				NumericDDMFormFieldUtil.getDecimalFormat(locale);

			Number number = (Number)fieldValue;

			if (number instanceof Double || number instanceof Float) {
				decimalFormat.setMaximumFractionDigits(Integer.MAX_VALUE);
				decimalFormat.setMinimumFractionDigits(1);
			}

			return decimalFormat.format(number.doubleValue());
		}

		return String.valueOf(fieldValue);
	}

	private List<String> _getDDMFormFieldNames(
		List<DDMFormField> ddmFormFields) {

		List<String> fieldNames = new ArrayList<>();

		for (DDMFormField ddmFormField : ddmFormFields) {
			fieldNames.add(ddmFormField.getName());
		}

		return fieldNames;
	}

	private void _setDDMFormFieldValueInstanceId(
		DDMFormFieldValue ddmFormFieldValue, Fields ddmFields,
		DDMFieldsCounter ddmFieldsCounter) {

		String name = ddmFormFieldValue.getName();

		ddmFormFieldValue.setInstanceId(
			_getDDMFieldInstanceId(
				ddmFields, name, ddmFieldsCounter.get(name)));
	}

	private void _setDDMFormFieldValueLocalizedValue(
		DDMFormFieldValue ddmFormFieldValue, Field ddmField, int index) {

		Value value = new LocalizedValue(ddmField.getDefaultLocale());

		for (Locale availableLocale : ddmField.getAvailableLocales()) {
			String valueString = _getDDMFieldValueString(
				ddmField, availableLocale, index);

			value.addString(availableLocale, valueString);
		}

		ddmFormFieldValue.setValue(value);
	}

	private void _setDDMFormFieldValueProperties(
			DDMFormFieldValue ddmFormFieldValue,
			Map<String, DDMFormField> ddmFormFieldsMap, Fields ddmFields,
			DDMFieldsCounter ddmFieldsCounter)
		throws PortalException {

		_setDDMFormFieldValueInstanceId(
			ddmFormFieldValue, ddmFields, ddmFieldsCounter);

		_setNestedDDMFormFieldValues(
			ddmFormFieldValue, ddmFormFieldsMap, ddmFields, ddmFieldsCounter);

		_setDDMFormFieldValueValues(
			ddmFormFieldValue, ddmFormFieldsMap, ddmFields, ddmFieldsCounter);
	}

	private void _setDDMFormFieldValueUnlocalizedValue(
		DDMFormFieldValue ddmFormFieldValue, Field ddmField, int index) {

		String valueString = _getDDMFieldValueString(
			ddmField, ddmField.getDefaultLocale(), index);

		Value value = new UnlocalizedValue(valueString);

		ddmFormFieldValue.setValue(value);
	}

	private void _setDDMFormFieldValueValues(
			DDMFormFieldValue ddmFormFieldValue,
			Map<String, DDMFormField> ddmFormFieldMap, Fields ddmFields,
			DDMFieldsCounter ddmFieldsCounter)
		throws PortalException {

		String fieldName = ddmFormFieldValue.getName();

		DDMFormField ddmFormField = ddmFormFieldMap.get(fieldName);

		Field field = ddmFields.get(fieldName);

		if (!ddmFormField.isTransient() && (field != null)) {
			if (ddmFormField.isLocalizable()) {
				_setDDMFormFieldValueLocalizedValue(
					ddmFormFieldValue, field, ddmFieldsCounter.get(fieldName));
			}
			else {
				_setDDMFormFieldValueUnlocalizedValue(
					ddmFormFieldValue, field, ddmFieldsCounter.get(fieldName));
			}
		}

		ddmFieldsCounter.incrementKey(fieldName);
	}

	private void _setNestedDDMFormFieldValues(
			DDMFormFieldValue ddmFormFieldValue,
			Map<String, DDMFormField> ddmFormFieldsMap, Fields ddmFields,
			DDMFieldsCounter ddmFieldsCounter)
		throws PortalException {

		String fieldName = ddmFormFieldValue.getName();

		int parentOffset = ddmFieldsCounter.get(fieldName);

		DDMFormField parentDDMFormField = ddmFormFieldsMap.get(fieldName);

		List<String> nestedFieldNames = _getDDMFormFieldNames(
			parentDDMFormField.getNestedDDMFormFields());

		for (String nestedFieldName : nestedFieldNames) {
			int repetitions = _countDDMFieldRepetitions(
				ddmFormFieldsMap, ddmFields, nestedFieldName, fieldName,
				parentOffset);

			for (int i = 0; i < repetitions; i++) {
				DDMFormFieldValue nestedDDMFormFieldValue =
					createDDMFormFieldValue(nestedFieldName);

				DDMFormField nestedDDMFormField = ddmFormFieldsMap.get(
					nestedFieldName);

				if (nestedDDMFormField != null) {
					nestedDDMFormFieldValue.setFieldReference(
						nestedDDMFormField.getFieldReference());
				}

				_setDDMFormFieldValueProperties(
					nestedDDMFormFieldValue, ddmFormFieldsMap, ddmFields,
					ddmFieldsCounter);

				ddmFormFieldValue.addNestedDDMFormFieldValue(
					nestedDDMFormFieldValue);
			}
		}
	}

}