/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

import {PagesVisitor} from 'dynamic-data-mapping-form-renderer';

export const formatFieldName = (instanceId, languageId, value) => {
	return `ddm$$${value}$${instanceId}$0$$${languageId}`;
};

export const generateInstanceId = length => {
	let text = '';

	const possible =
		'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';

	for (let i = 0; i < length; i++) {
		text += possible.charAt(Math.floor(Math.random() * possible.length));
	}

	return text;
};

export const normalizeSettingsContextPages = (
	pages,
	editingLanguageId,
	fieldType,
	generatedFieldName
) => {
	const visitor = new PagesVisitor(pages);

	return visitor.mapFields(field => {
		const {fieldName} = field;

		if (fieldName === 'name') {
			field = {
				...field,
				value: generatedFieldName,
				visible: true,
			};
		}
		else if (fieldName === 'label') {
			field = {
				...field,
				localizedValue: {
					...field.localizedValue,
					[editingLanguageId]: fieldType.label,
				},
				type: 'text',
				value: fieldType.label,
			};
		}
		else if (fieldName === 'type') {
			field = {
				...field,
				value: fieldType.name,
			};
		}
		else if (fieldName === 'validation') {
			field = {
				...field,
				validation: {
					...field.validation,
					fieldName: generatedFieldName,
				},
			};
		}

		return {
			...field,
		};
	});
};

export const getFieldProperties = (
	{pages},
	defaultLanguageId,
	editingLanguageId
) => {
	const properties = {};
	const visitor = new PagesVisitor(pages);

	visitor.mapFields(
		({fieldName, localizable, localizedValue = {}, type, value}) => {
			if (localizable && localizedValue[editingLanguageId]) {
				properties[fieldName] = localizedValue[editingLanguageId];
			}
			else if (localizable && localizedValue[defaultLanguageId]) {
				properties[fieldName] = localizedValue[defaultLanguageId];
			}
			else if (type == 'options') {
				if (!value[editingLanguageId] && value[defaultLanguageId]) {
					properties[fieldName] = value[defaultLanguageId];
				}
				else {
					properties[fieldName] = value[editingLanguageId];
				}
			}
			else {
				properties[fieldName] = value;
			}
		}
	);

	return properties;
};

export const localizeField = (field, defaultLanguageId, editingLanguageId) => {
	let value = field.value;

	if (field.localizable && field.localizedValue) {
		let localizedValue = field.localizedValue[editingLanguageId];

		if (localizedValue === undefined) {
			localizedValue = field.localizedValue[defaultLanguageId];
		}

		if (localizedValue !== undefined) {
			value = localizedValue;
		}
	}
	else if (
		field.dataType === 'ddm-options' &&
		value[editingLanguageId] === undefined
	) {
		value = {
			...value,
			[editingLanguageId]: value[defaultLanguageId],
		};
	}

	return {
		...field,
		defaultLanguageId,
		editingLanguageId,
		localizedValue: {
			...(field.localizedValue || {}),
			[editingLanguageId]: value,
		},
		value,
	};
};
