/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.analytics.settings.rest.internal.resource.v1_0;

import com.liferay.analytics.settings.configuration.AnalyticsConfiguration;
import com.liferay.analytics.settings.rest.dto.v1_0.Channel;
import com.liferay.analytics.settings.rest.dto.v1_0.DataSource;
import com.liferay.analytics.settings.rest.internal.client.AnalyticsCloudClient;
import com.liferay.analytics.settings.rest.internal.client.model.AnalyticsChannel;
import com.liferay.analytics.settings.rest.internal.client.model.AnalyticsDataSource;
import com.liferay.analytics.settings.rest.internal.dto.v1_0.converter.ChannelDTOConverter;
import com.liferay.analytics.settings.rest.internal.dto.v1_0.converter.ChannelDTOConverterContext;
import com.liferay.analytics.settings.rest.internal.manager.AnalyticsSettingsManager;
import com.liferay.analytics.settings.rest.resource.v1_0.ChannelResource;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.search.Sort;
import com.liferay.portal.kernel.service.GroupLocalService;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.HashMapBuilder;
import com.liferay.portal.kernel.util.Portal;
import com.liferay.portal.kernel.util.UnicodeProperties;
import com.liferay.portal.vulcan.dto.converter.DTOConverterRegistry;
import com.liferay.portal.vulcan.pagination.Page;
import com.liferay.portal.vulcan.pagination.Pagination;

import java.util.Map;
import java.util.function.Function;
import java.util.stream.Stream;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;

/**
 * @author Riccardo Ferrari
 */
@Component(
	properties = "OSGI-INF/liferay/rest/v1_0/channel.properties",
	scope = ServiceScope.PROTOTYPE, service = ChannelResource.class
)
public class ChannelResourceImpl extends BaseChannelResourceImpl {

	@Override
	public Page<Channel> getChannelsPage(
			String keywords, Pagination pagination, Sort[] sorts)
		throws Exception {

		com.liferay.analytics.settings.rest.internal.client.pagination.Page
			<AnalyticsChannel> analyticsChannelsPage =
				_analyticsCloudClient.getAnalyticsChannelsPage(
					contextCompany.getCompanyId(), keywords,
					pagination.getPage() - 1, pagination.getPageSize(), sorts);
		AnalyticsConfiguration analyticsConfiguration =
			_analyticsSettingsManager.getAnalyticsConfiguration(
				contextCompany.getCompanyId());

		return Page.of(
			transform(
				analyticsChannelsPage.getItems(),
				analyticsChannel -> _channelDTOConverter.toDTO(
					new ChannelDTOConverterContext(
						analyticsChannel.getId(),
						contextAcceptLanguage.getPreferredLocale(),
						analyticsConfiguration.commerceSyncEnabledChannelIds()),
					analyticsChannel)),
			pagination, analyticsChannelsPage.getTotalCount());
	}

	@Override
	public Channel patchChannel(Channel channel) throws Exception {
		AnalyticsConfiguration analyticsConfiguration =
			_analyticsSettingsManager.getAnalyticsConfiguration(
				contextCompany.getCompanyId());

		String[] commerceSyncEnabledChannelIds =
			analyticsConfiguration.commerceSyncEnabledChannelIds();

		if (channel.getCommerceSyncEnabled() != null) {
			boolean commerceSyncEnabled = ArrayUtil.contains(
				commerceSyncEnabledChannelIds, channel.getChannelId());

			if (channel.getCommerceSyncEnabled() && !commerceSyncEnabled) {
				commerceSyncEnabledChannelIds = ArrayUtil.append(
					commerceSyncEnabledChannelIds, channel.getChannelId());

				_analyticsSettingsManager.updateCompanyConfiguration(
					contextCompany.getCompanyId(),
					HashMapBuilder.<String, Object>put(
						"commerceSyncEnabledChannelIds",
						commerceSyncEnabledChannelIds
					).build());
			}

			if (!channel.getCommerceSyncEnabled() && commerceSyncEnabled) {
				commerceSyncEnabledChannelIds = ArrayUtil.remove(
					commerceSyncEnabledChannelIds, channel.getChannelId());

				_analyticsSettingsManager.updateCompanyConfiguration(
					contextCompany.getCompanyId(),
					HashMapBuilder.<String, Object>put(
						"commerceSyncEnabledChannelIds",
						commerceSyncEnabledChannelIds
					).build());
			}
		}

		DataSource[] dataSources = channel.getDataSources();

		if (ArrayUtil.isEmpty(dataSources)) {
			return _channelDTOConverter.toDTO(
				new ChannelDTOConverterContext(
					channel.getChannelId(),
					contextAcceptLanguage.getPreferredLocale(),
					commerceSyncEnabledChannelIds),
				_analyticsCloudClient.updateAnalyticsChannel(
					channel.getChannelId(),
					Stream.of(
						analyticsConfiguration.syncedCommerceChannelIds()
					).map(
						Long::valueOf
					).toArray(
						Long[]::new
					),
					contextUser.getCompanyId(),
					analyticsConfiguration.liferayAnalyticsDataSourceId(),
					contextAcceptLanguage.getPreferredLocale(),
					Stream.of(
						analyticsConfiguration.syncedGroupIds()
					).map(
						Long::valueOf
					).toArray(
						Long[]::new
					)));
		}

		if (dataSources.length > 1) {
			throw new PortalException("Unable to update multiple data sources");
		}

		DataSource dataSource = dataSources[0];

		AnalyticsChannel analyticsChannel =
			_analyticsCloudClient.updateAnalyticsChannel(
				channel.getChannelId(), dataSource.getCommerceChannelIds(),
				contextUser.getCompanyId(), dataSource.getDataSourceId(),
				contextAcceptLanguage.getPreferredLocale(),
				dataSource.getSiteIds());

		AnalyticsDataSource analyticsDataSource = _getAnalyticsDataSource(
			GetterUtil.getLong(dataSource.getDataSourceId()),
			analyticsChannel.getAnalyticsDataSources());

		_analyticsCloudClient.updateAnalyticsDataSourceDetails(
			null, contextCompany.getCompanyId(),
			ArrayUtil.isNotEmpty(analyticsDataSource.getCommerceChannelIds()),
			null, ArrayUtil.isNotEmpty(analyticsDataSource.getSiteIds()));

		_updateCommerceChannelGroups(
			analyticsConfiguration.syncedCommerceChannelIds(),
			channel.getChannelId(), contextCompany.getCompanyId(),
			analyticsDataSource.getCommerceChannelIds());

		_updateGroups(
			analyticsConfiguration.syncedGroupIds(), channel.getChannelId(),
			analyticsDataSource.getSiteIds());

		_analyticsSettingsManager.updateCompanyConfiguration(
			contextUser.getCompanyId(),
			HashMapBuilder.<String, Object>put(
				"syncedCommerceChannelIds",
				analyticsDataSource.getCommerceChannelIds()
			).put(
				"syncedGroupIds", analyticsDataSource.getSiteIds()
			).build());

		return _channelDTOConverter.toDTO(
			new ChannelDTOConverterContext(
				channel.getChannelId(),
				contextAcceptLanguage.getPreferredLocale(),
				commerceSyncEnabledChannelIds),
			analyticsChannel);
	}

	@Override
	public Channel postChannel(Channel channel) throws Exception {
		AnalyticsConfiguration analyticsConfiguration =
			_analyticsSettingsManager.getAnalyticsConfiguration(
				contextCompany.getCompanyId());

		return _channelDTOConverter.toDTO(
			new ChannelDTOConverterContext(
				channel.getChannelId(),
				contextAcceptLanguage.getPreferredLocale(),
				analyticsConfiguration.commerceSyncEnabledChannelIds()),
			_analyticsCloudClient.addAnalyticsChannel(
				contextCompany.getCompanyId(), channel.getName()));
	}

	@Activate
	protected void activate(Map<String, Object> properties) {
		_commerceChannelClassNameId = _portal.getClassNameId(
			"com.liferay.commerce.product.model.CommerceChannel");
	}

	@Reference
	protected DTOConverterRegistry dtoConverterRegistry;

	private AnalyticsDataSource _getAnalyticsDataSource(
		long analyticsDataSourceId,
		AnalyticsDataSource[] analyticsDataSources) {

		for (AnalyticsDataSource analyticsDataSource : analyticsDataSources) {
			if (analyticsDataSource.getId() == analyticsDataSourceId) {
				return analyticsDataSource;
			}
		}

		throw new RuntimeException("Unable to get analytics data source");
	}

	private void _updateCommerceChannelGroups(
			String[] analyticsConfigurationCommerceChannelIds, String channelId,
			long companyId, Long[] dataSourceCommerceChannelIds)
		throws Exception {

		_updateTypeSetting(
			channelId,
			commerceChannelId -> _groupLocalService.fetchGroup(
				companyId, _commerceChannelClassNameId, commerceChannelId),
			ArrayUtil.filter(
				dataSourceCommerceChannelIds,
				commerceChannelId -> !ArrayUtil.contains(
					analyticsConfigurationCommerceChannelIds,
					String.valueOf(commerceChannelId))),
			false);

		_updateTypeSetting(
			channelId,
			commerceChannelId -> _groupLocalService.fetchGroup(
				companyId, _commerceChannelClassNameId, commerceChannelId),
			ArrayUtil.filter(
				analyticsConfigurationCommerceChannelIds,
				commerceChannelId -> !ArrayUtil.contains(
					dataSourceCommerceChannelIds,
					Long.valueOf(commerceChannelId))),
			true);
	}

	private void _updateGroups(
			String[] analyticsConfigurationGroupIds, String channelId,
			Long[] dataSourceGroupIds)
		throws Exception {

		_updateTypeSetting(
			channelId, groupId -> _groupLocalService.fetchGroup(groupId),
			ArrayUtil.filter(
				dataSourceGroupIds,
				groupId -> !ArrayUtil.contains(
					analyticsConfigurationGroupIds, String.valueOf(groupId))),
			false);

		_updateTypeSetting(
			channelId, groupId -> _groupLocalService.fetchGroup(groupId),
			ArrayUtil.filter(
				analyticsConfigurationGroupIds,
				groupId -> !ArrayUtil.contains(
					dataSourceGroupIds, Long.valueOf(groupId))),
			true);
	}

	private <T> void _updateTypeSetting(
			String channelId, Function<Long, Group> fetchGroupFunction,
			T[] groupIds, boolean remove)
		throws Exception {

		for (T groupId : groupIds) {
			Group group = fetchGroupFunction.apply(GetterUtil.getLong(groupId));

			if (group == null) {
				continue;
			}

			UnicodeProperties typeSettingsUnicodeProperties =
				group.getTypeSettingsProperties();

			if (remove) {
				String analyticsChannelId =
					typeSettingsUnicodeProperties.remove("analyticsChannelId");

				if ((analyticsChannelId != null) &&
					!channelId.equals(analyticsChannelId)) {

					throw new IllegalArgumentException("Invalid channel ID");
				}
			}
			else {
				typeSettingsUnicodeProperties.setProperty(
					"analyticsChannelId", channelId);
			}

			_groupLocalService.updateGroup(group);
		}
	}

	@Reference
	private AnalyticsCloudClient _analyticsCloudClient;

	@Reference
	private AnalyticsSettingsManager _analyticsSettingsManager;

	@Reference
	private ChannelDTOConverter _channelDTOConverter;

	private long _commerceChannelClassNameId;

	@Reference
	private GroupLocalService _groupLocalService;

	@Reference
	private Portal _portal;

}