package com.kontakt.sdk.android.http.interfaces;


import com.kontakt.sdk.android.common.model.IManager;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ETag;
import com.kontakt.sdk.android.http.HttpResult;
import com.kontakt.sdk.android.http.RequestDescription;
import com.kontakt.sdk.android.http.data.ManagerData;
import com.kontakt.sdk.android.http.exception.ClientException;

import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * Managers API Accessor provides interface for Managers hierarchy management.
 */
public interface ManagersApiAccessor extends ApiAccessor {

    /**
     * Gets subordinates for manager by specified Manager Id.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     *
     * @param managerId          the manager id
     * @param requestDescription the request description
     * @return the subordinates for manager
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<List<IManager>> listSubordinatesForManager(UUID managerId, RequestDescription requestDescription) throws ClientException;

    /**
     * List subordinates for manager.
     *
     * @param managerId the manager id
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IManager>> listSubordinatesForManager(UUID managerId) throws ClientException;

    /**
     * List subordinates for manager.
     *
     * @param managerId          the manager id
     * @param requestDescription the request description
     * @param resultApiCallback  the result api callback
     */
    void listSubordinatesForManager(UUID managerId, RequestDescription requestDescription, ResultApiCallback<List<IManager>> resultApiCallback);

    /**
     * List subordinates for manager.
     *
     * @param managerId         the manager id
     * @param resultApiCallback the result api callback
     */
    void listSubordinatesForManager(UUID managerId, ResultApiCallback<List<IManager>> resultApiCallback);

    /**
     * Deletes manager.
     *
     * @param managerId the manager id
     * @return the Http Status. Successful deletion should return Http Status: 200.
     * @throws ClientException is thrown if unexpected error has occured
     */
    int deleteManager(UUID managerId) throws ClientException;

    /**
     * Delete manager.
     *
     * @param managerId   the manager id
     * @param apiCallback the api callback
     */
    void deleteManager(UUID managerId, UpdateApiCallback apiCallback);

    /**
     * Updates manager.
     * <p>
     * Required parameter: manager Id. The optional parameters: firstName, lastName, role(one of: ADMIN, OPERATOR),
     * e-mail.If any of the optional parameters is null, it will not be updated.
     *
     * @param managerData the manager update data
     * @return the Http Status. Successful update should return HttpStauts: 200.
     * @throws ClientException is thrown if unexpected error has occured
     */
    int updateManager(ManagerData managerData) throws ClientException;

    /**
     * Update manager.
     *
     * @param managerData       the manager data
     * @param updateApiCallback the update api callback
     */
    void updateManager(ManagerData managerData, UpdateApiCallback updateApiCallback);

    /**
     * Creates manager.
     * <p>
     * Required parameters: firstName, lastName, e-mail, role(one of: SUPERUSER, ADMIN, OPERATOR).
     *
     * @param manager the manager data
     * @return the result
     * @throws ClientException is thrown if unexpected error has occured
     */
    HttpResult<IManager> createManager(ManagerData manager) throws ClientException;

    /**
     * Create manager.
     *
     * @param managerData the manager data
     * @param apiCallback the api callback
     */
    void createManager(ManagerData managerData, ResultApiCallback<IManager> apiCallback);

    /**
     * Gets manager.
     *
     * @param managerId    the manager id
     * @param eTagOptional the e tag optional
     * @return the manager
     * @throws ClientException the client exception
     */
    HttpResult<IManager> getManager(UUID managerId, SDKOptional<ETag> eTagOptional) throws ClientException;

    /**
     * Gets manager.
     *
     * @param managerId the manager id
     * @return the manager
     * @throws ClientException the client exception
     */
    HttpResult<IManager> getManager(UUID managerId) throws ClientException;

    /**
     * Gets manager.
     *
     * @param managerId   the manager id
     * @param etag        the etag
     * @param apiCallback the api callback
     */
    void getManager(UUID managerId, SDKOptional<ETag> etag, ResultApiCallback<IManager> apiCallback);

    /**
     * Gets manager.
     *
     * @param managerId   the manager id
     * @param apiCallback the api callback
     */
    void getManager(UUID managerId, ResultApiCallback<IManager> apiCallback);

    /**
     * Assigns managers to supervisor.
     *
     * @param supervisorId the supervisor id
     * @param managerIdSet the manager ids set
     * @return the Http Status. Successful assign returns Http Status: 200.
     * @throws ClientException is thrown if unexpected error has occured
     */
    int assignManagersToSupervisor(UUID supervisorId, Set<UUID> managerIdSet) throws ClientException;

    /**
     * Assign managers to supervisor.
     *
     * @param supervisorId the supervisor id
     * @param managerIdSet the manager id set
     * @param apiCallback  the api callback
     */
    void assignManagersToSupervisor(UUID supervisorId, Set<UUID> managerIdSet, UpdateApiCallback apiCallback);

    /**
     * Provides the list of managers. According to request description
     * the response:
     * <ul>
     * <li>may contain sorted list items</li>
     * <li>may contain empty body for conditional requests with ETag value included in headers</li>
     * <li>may contain list items in ascending or descending order</li>
     * <li>may contain ranged list items by specifying start index and offset</li>
     * </ul>
     *
     * @param requestDescription the request description
     * @return the devices
     * @throws ClientException the client exception
     */
    HttpResult<List<IManager>> listManagers(final RequestDescription requestDescription) throws ClientException;

    /**
     * List managers.
     *
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IManager>> listManagers() throws ClientException;

    /**
     * List managers.
     *
     * @param requestDescription the request description
     * @param apiCallback        the api callback
     */
    void listManagers(RequestDescription requestDescription, ResultApiCallback<List<IManager>> apiCallback);

    /**
     * List managers.
     *
     * @param apiCallback the api callback
     */
    void listManagers(ResultApiCallback<List<IManager>> apiCallback);
}
