package com.kontakt.sdk.android.http.interfaces;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.IFirmware;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ETag;
import com.kontakt.sdk.android.http.HttpResult;
import com.kontakt.sdk.android.http.exception.ClientException;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Firmware API accessor provides interface for Firmware retrieval.
 */
public interface FirmwareApiAccessor extends ApiAccessor {

    /**
     * Requests latest firmware for beacons.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     *
     * @param beaconUniqueIds the beacon unique ids
     * @param eTag            the e tag
     * @return the latest firmware for beacons
     * @throws ClientException the client exception
     */
    @Deprecated
    HttpResult<Map<String, IFirmware>> getLatestFirmwareForBeacons(Set<String> beaconUniqueIds, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets latest firmware for beacons.
     *
     * @param beaconUniqueIds the beacon unique ids
     * @return the latest firmware for beacons
     * @throws ClientException the client exception
     */
    @Deprecated
    HttpResult<Map<String, IFirmware>> getLatestFirmwareForBeacons(Set<String> beaconUniqueIds) throws ClientException;

    /**
     * Gets latest firmware for beacons.
     *
     * @param deviceUniqueIds the device unique ids
     * @param eTag            the e tag
     * @param callback        the callback
     */
    @Deprecated
    void getLatestFirmwareForBeacons(Set<String> deviceUniqueIds, SDKOptional<ETag> eTag, ResultApiCallback<Map<String, IFirmware>> callback);

    /**
     * Gets latest firmware for beacons.
     *
     * @param deviceUniqueIds the device unique ids
     * @param callback        the callback
     */
    @Deprecated
    void getLatestFirmwareForBeacons(Set<String> deviceUniqueIds, ResultApiCallback<Map<String, IFirmware>> callback);

    /**
     * Gets List of firmware for given list of device unique ids
     *
     * @param deviceUniqueIds List of device unique id
     * @return List of {@link IFirmware}
     * @throws ClientException
     */
    HttpResult<List<IFirmware>> getFirmwares(Set<String> deviceUniqueIds) throws ClientException;

    /**
     * Gets List of firmware for given list of device unique ids
     *
     * @param deviceUniqueIds List of device unique id
     * @param apiCallback     the callback
     */
    void getFirmwares(Set<String> deviceUniqueIds, ResultApiCallback<List<IFirmware>> apiCallback);

    /**
     * Fetch firmware file data.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     *
     * @param firmwareName the firmware name
     * @param eTag         the e tag
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<FileData> fetchFirmwareFileData(String firmwareName, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Fetch firmware file data.
     *
     * @param firmwareName the firmware name
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<FileData> fetchFirmwareFileData(String firmwareName) throws ClientException;

    /**
     * Fetch firmware file data.
     *
     * @param firmwareName      the firmware name
     * @param etag              the etag
     * @param resultApiCallback the result api callback
     */
    void fetchFirmwareFileData(String firmwareName, SDKOptional<ETag> etag, ResultApiCallback<FileData> resultApiCallback);

    /**
     * Fetch firmware file data.
     *
     * @param firmwareName      the firmware name
     * @param resultApiCallback the result api callback
     */
    void fetchFirmwareFileData(String firmwareName, ResultApiCallback<FileData> resultApiCallback);

    /**
     * Requests firmware by its name.
     * If the ETag's value matches the latest one
     * the code 304 (NOT MODIFIED) is returned.
     *
     * @param firmwareName the firmware name
     * @param deviceType   the device type
     * @param eTag         the e tag
     * @return the firmware
     * @throws ClientException the client exception
     */
    HttpResult<IFirmware> getFirmware(String firmwareName, DeviceType deviceType, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets firmware.
     *
     * @param firmwareName the firmware name
     * @param deviceType   the device type
     * @return the firmware
     * @throws ClientException the client exception
     */
    HttpResult<IFirmware> getFirmware(String firmwareName, DeviceType deviceType) throws ClientException;

    /**
     * Gets firmware.
     *
     * @param firmwareName the firmware name
     * @param deviceType   the device type
     * @param eTag         the e tag
     * @param apiCallback  the api callback
     */
    void getFirmware(String firmwareName, DeviceType deviceType, SDKOptional<ETag> eTag, ResultApiCallback<IFirmware> apiCallback);

    /**
     * Gets firmware.
     *
     * @param firmwareName the firmware name
     * @param deviceType   the device type
     * @param apiCallback  the api callback
     */
    void getFirmware(String firmwareName, DeviceType deviceType, ResultApiCallback<IFirmware> apiCallback);
}
