package com.kontakt.sdk.android.http.interfaces;

import com.kontakt.sdk.android.common.model.BeaconId;
import com.kontakt.sdk.android.common.model.EddystoneFutureUID;
import com.kontakt.sdk.android.common.model.EddystoneUID;
import com.kontakt.sdk.android.common.model.IBeaconFutureId;
import com.kontakt.sdk.android.common.model.ICloudConfig;
import com.kontakt.sdk.android.common.model.IConfig;
import com.kontakt.sdk.android.common.model.ICredentials;
import com.kontakt.sdk.android.common.model.IDevice;
import com.kontakt.sdk.android.common.model.SecureSingleConfig;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ETag;
import com.kontakt.sdk.android.http.HttpResult;
import com.kontakt.sdk.android.http.RequestDescription;
import com.kontakt.sdk.android.http.data.DeviceData;
import com.kontakt.sdk.android.http.exception.ClientException;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * The interface Devices api accessor.
 */
public interface DevicesApiAccessor extends ApiAccessor {

    /**
     * Gets device credentials.
     *
     * @param deviceUniqueId the device unique id
     * @param eTag           the e tag
     * @return the device credentials
     * @throws ClientException the client exception
     */
    HttpResult<ICredentials> getDeviceCredentials(String deviceUniqueId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Provides credentials list for devices unique identifiers.
     *
     * @param uniqueIds the unique ids
     * @param eTag      the e tag
     * @return the devices credentials
     * @throws ClientException the client exception
     */
    HttpResult<List<ICredentials>> listDevicesCredentials(Collection<String> uniqueIds,
                                                          SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Provides credentials list for devices unique identifiers asynchronously.
     *
     * @param uniqueIds the unique ids
     * @param eTag      the e tag
     * @throws ClientException the client exception
     */
    void listDevicesCredentials(Collection<String> uniqueIds,
                                SDKOptional<ETag> eTag,
                                ResultApiCallback<List<ICredentials>> resultApiCallback);

    /**
     * Gets device credentials.
     *
     * @param deviceUniqueId the device unique id
     * @param etag           the etag
     * @param apiCallback    the api callback
     */
    void getDeviceCredentials(String deviceUniqueId, SDKOptional<ETag> etag, ResultApiCallback<ICredentials> apiCallback);

    /**
     * Gets devices by proximity.
     *
     * @param proximityUUID the proximity uUID
     * @param major         the major
     * @param minor         the minor
     * @param eTag          the e tag
     * @return the devices by proximity
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> getDevicesByProximity(UUID proximityUUID,
                                                    int major,
                                                    int minor,
                                                    SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets devices by proximity.
     *
     * @param proximityUUID the proximity uUID
     * @param major         the major
     * @param minor         the minor
     * @return the devices by proximity
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> getDevicesByProximity(UUID proximityUUID,
                                                    int major,
                                                    int minor) throws ClientException;

    /**
     * Gets devices by proximity.
     *
     * @param proximityUUID the proximity uUID
     * @param major         the major
     * @param minor         the minor
     * @param eTag          the e tag
     * @param apiCallback   the api callback
     */
    void getDevicesByProximity(UUID proximityUUID,
                               int major,
                               int minor,
                               SDKOptional<ETag> eTag,
                               ResultApiCallback<List<IDevice>> apiCallback);

    /**
     * Gets devices by proximity.
     *
     * @param proximityUUID the proximity uUID
     * @param major         the major
     * @param minor         the minor
     * @param apiCallback   the api callback
     */
    void getDevicesByProximity(UUID proximityUUID,
                               int major,
                               int minor,
                               ResultApiCallback<List<IDevice>> apiCallback);

    /**
     * Gets device by namespace and instance id.
     *
     * @param namespace  the namespace
     * @param instanceId the instance id
     * @param eTag       the e tag
     * @return the device by namespace and instance id
     * @throws ClientException the client exception
     */
    HttpResult<IDevice> getDeviceByNamespaceAndInstanceId(String namespace,
                                                          String instanceId,
                                                          SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets device by namespace and instance id.
     *
     * @param namespace  the namespace
     * @param instanceId the instance id
     * @return the device by namespace and instance id
     * @throws ClientException the client exception
     */
    HttpResult<IDevice> getDeviceByNamespaceAndInstanceId(String namespace,
                                                          String instanceId) throws ClientException;


    /**
     * Gets device by namespace and instance id.
     *
     * @param namespace   the namespace
     * @param instanceId  the instance id
     * @param eTag        the e tag
     * @param apiCallback the api callback
     */
    void getDeviceByNamespaceAndInstanceId(String namespace,
                                           String instanceId,
                                           SDKOptional<ETag> eTag,
                                           ResultApiCallback<IDevice> apiCallback);

    /**
     * Gets device by namespace and instance id.
     *
     * @param namespace   the namespace
     * @param instanceId  the instance id
     * @param apiCallback the api callback
     */
    void getDeviceByNamespaceAndInstanceId(String namespace,
                                           String instanceId,
                                           ResultApiCallback<IDevice> apiCallback);

    /**
     * Gets device.
     *
     * @param deviceUniqueId the device unique id
     * @param eTag           the e tag
     * @return the device
     * @throws ClientException the client exception
     */
    HttpResult<IDevice> getDevice(String deviceUniqueId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets device.
     *
     * @param deviceUniqueId the device unique id
     * @return the device
     * @throws ClientException the client exception
     */
    HttpResult<IDevice> getDevice(String deviceUniqueId) throws ClientException;

    /**
     * Gets device.
     *
     * @param deviceUniqueId the device unique id
     * @param etag           the etag
     * @param apiCallback    the api callback
     */
    void getDevice(String deviceUniqueId, SDKOptional<ETag> etag, ResultApiCallback<IDevice> apiCallback);

    /**
     * Gets device.
     *
     * @param deviceUniqueId the device unique id
     * @param apiCallback    the api callback
     */
    void getDevice(String deviceUniqueId, ResultApiCallback<IDevice> apiCallback);


    /**
     * List devices.
     *
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> listDevices() throws ClientException;

    /**
     * List devices.
     *
     * @param requestDescription the request description
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> listDevices(RequestDescription requestDescription) throws ClientException;

    /**
     * List devices.
     *
     * @param requestDescription the request description
     * @param apiCallback        the api callback
     */
    void listDevices(RequestDescription requestDescription, ResultApiCallback<List<IDevice>> apiCallback);

    /**
     * List devices.
     *
     * @param apiCallback the api callback
     */
    void listDevices(ResultApiCallback<List<IDevice>> apiCallback);


    /**
     * Update device password.
     *
     * @param beaconUniqueId the beacon unique id
     * @param password       the password
     * @return the int
     * @throws ClientException the client exception
     */
    int updateDevicePassword(String beaconUniqueId, String password) throws ClientException;

    /**
     * Update device password.
     *
     * @param beaconUniqueId the beacon unique id
     * @param password       the password
     * @param callback       the callback
     */
    void updateDevicePassword(String beaconUniqueId, String password, UpdateApiCallback callback);

    /**
     * List devices for managers.
     *
     * @param managerIds the manager ids
     * @param eTag       the e tag
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> listDevicesForManagers(Set<UUID> managerIds, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * List devices for managers.
     *
     * @param managerIds the manager ids
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> listDevicesForManagers(Set<UUID> managerIds) throws ClientException;

    /**
     * List devices for managers.
     *
     * @param managerIds  the manager ids
     * @param eTag        the e tag
     * @param apiCallback the api callback
     */
    void listDevicesForManagers(Set<UUID> managerIds, SDKOptional<ETag> eTag, ResultApiCallback<List<IDevice>> apiCallback);

    /**
     * List devices for managers.
     *
     * @param managerIds  the manager ids
     * @param apiCallback the api callback
     */
    void listDevicesForManagers(Set<UUID> managerIds, ResultApiCallback<List<IDevice>> apiCallback);


    /**
     * Update device.
     *
     * @param deviceData the device data
     * @return the int
     * @throws ClientException the client exception
     */
    int updateDevice(DeviceData deviceData) throws ClientException;

    /**
     * Update device.
     *
     * @param deviceData the device data
     * @param callback   the callback
     */
    void updateDevice(DeviceData deviceData, UpdateApiCallback callback);


    /**
     * Move devices to manager.
     *
     * @param deviceUniqueIds   the device unique ids
     * @param receiverId        the receiver id
     * @param receiverCompanyId the receiver company id
     * @return the int
     * @throws ClientException the client exception
     */
    int moveDevicesToManager(Set<String> deviceUniqueIds, UUID receiverId, UUID receiverCompanyId) throws ClientException;

    /**
     * Move devices to manager.
     *
     * @param deviceUniqueIds   the device unique ids
     * @param receiverID        the receiver iD
     * @param receiverCompanyId the receiver company id
     * @param apiUpdateCallback the api update callback
     */
    void moveDevicesToManager(Set<String> deviceUniqueIds, UUID receiverID, UUID receiverCompanyId, UpdateApiCallback apiUpdateCallback);

    /**
     * Assign devices to venue.
     *
     * @param venueId     the venue id
     * @param deviceIdSet the device id set
     * @return the int
     * @throws ClientException the client exception
     */
    int assignDevicesToVenue(UUID venueId, Set<UUID> deviceIdSet) throws ClientException;

    /**
     * Assign devices to venue.
     *
     * @param venueId           the venue id
     * @param deviceIdSet       the device id set
     * @param apiUpdateCallback the api update callback
     */
    void assignDevicesToVenue(UUID venueId, Set<UUID> deviceIdSet, UpdateApiCallback apiUpdateCallback);


    /**
     * Assign devices to manager.
     *
     * @param managerId   the manager id
     * @param beaconIdSet the beacon id set
     * @return the int
     * @throws ClientException the client exception
     */
    int assignDevicesToManager(UUID managerId, Set<UUID> beaconIdSet) throws ClientException;

    /**
     * Assign devices to manager.
     *
     * @param managerId   the manager id
     * @param beaconIdSet the beacon id set
     * @param apiCallback the api callback
     */
    void assignDevicesToManager(UUID managerId, Set<UUID> beaconIdSet, UpdateApiCallback apiCallback);

    /**
     * Apply config.
     *
     * @param config the config
     * @return the int
     * @throws ClientException the client exception
     */
    int applyConfig(IConfig config) throws ClientException;

    /**
     * Apply config.
     *
     * @param config      the config
     * @param apiCallback the api callback
     */
    void applyConfig(IConfig config, UpdateApiCallback apiCallback);


    /**
     * Apply cloud config.
     *
     * @param cloudConfig the cloud config
     * @return the int
     * @throws ClientException the client exception
     */
    int applyCloudConfig(ICloudConfig cloudConfig) throws ClientException;

    /**
     * Apply config.
     *
     * @param cloudConfig the cloud config
     * @param apiCallback the api callback
     */
    void applyConfig(ICloudConfig cloudConfig, UpdateApiCallback apiCallback);

    /**
     * List unassigned devices for manager.
     *
     * @param managerId          the manager id
     * @param requestDescription the request description
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> listUnassignedDevicesForManager(UUID managerId, RequestDescription requestDescription) throws ClientException;

    /**
     * List unassigned devices for manager.
     *
     * @param managerId the manager id
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IDevice>> listUnassignedDevicesForManager(UUID managerId) throws ClientException;

    /**
     * List unassigned devices for manager.
     *
     * @param managerId          the manager id
     * @param requestDescription the request description
     * @param apiCallback        the api callback
     */
    void listUnassignedDevicesForManager(UUID managerId, RequestDescription requestDescription, ResultApiCallback<List<IDevice>> apiCallback);

    /**
     * List unassigned devices for manager.
     *
     * @param managerId   the manager id
     * @param apiCallback the api callback
     */
    void listUnassignedDevicesForManager(UUID managerId, ResultApiCallback<List<IDevice>> apiCallback);

    /**
     * Resolves shuffled to true beacon ids (proximity, major, minor)
     *
     * @param beaconIds   the beacon ids
     * @param eTag        the e tag
     * @param apiCallback the api callback
     */
    void resolveIBeacon(Collection<BeaconId> beaconIds, SDKOptional<ETag> eTag, ResultApiCallback<List<IBeaconFutureId>> apiCallback);

    /**
     * Resolves shuffled to true beacon ids (proximity, major, minor)
     *
     * @param beaconIds the beacon ids
     * @param eTag      the e tag
     * @return list of resolved beacon ids with future shuffles
     * @throws ClientException
     */
    HttpResult<List<IBeaconFutureId>> resolveIBeacon(Collection<BeaconId> beaconIds, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Resolves shuffled to true eddystone uids (namespace, instance id)
     *
     * @param eddystoneUids the eddystone uids
     * @param eTag          the e tag
     * @param apiCallback   the api callback
     */
    void resolveEddystone(Collection<EddystoneUID> eddystoneUids, SDKOptional<ETag> eTag, ResultApiCallback<List<EddystoneFutureUID>> apiCallback);

    /**
     * Resolves shuffled to true eddystone uids (namespace, instance id)
     *
     * @param eddystoneUids the eddystoen uids
     * @param eTag          the e tag
     * @return list of resolved eddystone uids with future shuffles
     * @throws ClientException
     */
    HttpResult<List<EddystoneFutureUID>> resolveEddystone(Collection<EddystoneUID> eddystoneUids, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Applies secure config to cloud.
     *
     * @param secureConfigApplies collection with secure configs created from beacon response
     * @return Collection with next secure configs to apply on device, in most cases it will be empty
     * @throws ClientException
     */
    HttpResult<List<SecureSingleConfig>> applySecureConfig(Collection<SecureSingleConfig> secureConfigApplies) throws ClientException;

    /**
     * Applies secure config to cloud.
     *
     * @param secureConfigApplies collection with secure configs created from beacon response
     * @param apiCallback         the api callback with collection with next secure configs to apply on device, in most cases it will be empty
     */
    void applySecureConfig(Collection<SecureSingleConfig> secureConfigApplies, ResultApiCallback<List<SecureSingleConfig>> apiCallback);
}
