package com.kontakt.sdk.android.http.interfaces;

import com.kontakt.sdk.android.common.model.ICloudConfig;
import com.kontakt.sdk.android.common.model.IConfig;
import com.kontakt.sdk.android.common.model.IPreset;
import com.kontakt.sdk.android.common.model.SecureSingleConfig;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ETag;
import com.kontakt.sdk.android.http.HttpResult;
import com.kontakt.sdk.android.http.RequestDescription;
import com.kontakt.sdk.android.http.data.ConfigData;
import com.kontakt.sdk.android.http.exception.ClientException;

import java.util.Collection;
import java.util.List;

/**
 * Configuration API accessor provides interface for Configs Profiles management.
 */
public interface ConfigurationApiAccessor extends ApiAccessor {
    /**
     * Gets config sending ETag's value in request newHeader additionally.
     * If the ETag's value matches the latest one the code 304 is returned.
     *
     * @param requestDescription the request description
     * @return the config
     * @throws ClientException the client exception
     */
    HttpResult<List<IConfig>> listConfigs(RequestDescription requestDescription) throws ClientException;

    /**
     * List configs.
     *
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<IConfig>> listConfigs() throws ClientException;

    /**
     * List configs.
     *
     * @param requestDescription the request description
     * @param apiCallback        the api callback
     */
    void listConfigs(RequestDescription requestDescription, ResultApiCallback<List<IConfig>> apiCallback);

    /**
     * List configs.
     *
     * @param apiCallback the api callback
     */
    void listConfigs(ResultApiCallback<List<IConfig>> apiCallback);

    /**
     * List cloud configs.
     *
     * @param requestDescription the request description
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<ICloudConfig>> listCloudConfigs(RequestDescription requestDescription) throws ClientException;

    /**
     * List cloud configs.
     *
     * @return the http result
     * @throws ClientException the client exception
     */
    HttpResult<List<ICloudConfig>> listCloudConfigs() throws ClientException;

    /**
     * List cloud configs.
     *
     * @param requestDescription the request description
     * @param apiCallback        the api callback
     */
    void listCloudConfigs(RequestDescription requestDescription, ResultApiCallback<List<ICloudConfig>> apiCallback);

    /**
     * List cloud configs.
     *
     * @param apiCallback the api callback
     */
    void listCloudConfigs(ResultApiCallback<List<ICloudConfig>> apiCallback);

    /**
     * Gets config for beacon by beacon unique Id sending ETag's value in request newHeader additionally.
     * If the ETag's value matches the latest one
     * the code 304 is returned.
     *
     * @param beaconUniqueId the beacon unique id
     * @param eTag           the ETag optional
     * @return the config for beacon
     * @throws ClientException the client exception
     */
    HttpResult<IConfig> getConfigForDevice(String beaconUniqueId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets config for device.
     *
     * @param beaconUniqueId the beacon unique id
     * @return the config for device
     * @throws ClientException the client exception
     */
    HttpResult<IConfig> getConfigForDevice(String beaconUniqueId) throws ClientException;

    /**
     * Gets config for device.
     *
     * @param beaconUniqueId the beacon unique id
     * @param etag           the etag
     * @param apiCallback    the api callback
     */
    void getConfigForDevice(String beaconUniqueId, SDKOptional<ETag> etag, ResultApiCallback<IConfig> apiCallback);

    /**
     * Gets config for device.
     *
     * @param beaconUniqueId the beacon unique id
     * @param apiCallback    the api callback
     */
    void getConfigForDevice(String beaconUniqueId, ResultApiCallback<IConfig> apiCallback);

    /**
     * Gets cloud config for beacon.
     *
     * @param beaconUniqueId the beacon unique id
     * @param eTag           the e tag
     * @return the cloud config for beacon
     * @throws ClientException the client exception
     */
    HttpResult<ICloudConfig> getCloudConfigForDevice(String beaconUniqueId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets cloud config for device.
     *
     * @param beaconUniqueId the beacon unique id
     * @return the cloud config for device
     * @throws ClientException the client exception
     */
    HttpResult<ICloudConfig> getCloudConfigForDevice(String beaconUniqueId) throws ClientException;

    /**
     * Gets cloud config for device.
     *
     * @param beaconUniqueId the beacon unique id
     * @param eTag           the e tag
     * @param apiCallback    the api callback
     */
    void getCloudConfigForDevice(String beaconUniqueId, SDKOptional<ETag> eTag, ResultApiCallback<ICloudConfig> apiCallback);

    /**
     * Gets cloud config for device.
     *
     * @param beaconUniqueId the beacon unique id
     * @param apiCallback    the api callback
     */
    void getCloudConfigForDevice(String beaconUniqueId, ResultApiCallback<ICloudConfig> apiCallback);

    /**
     * Creates config.
     *
     * @param configData the config data
     * @return the result
     * @throws ClientException is thrown if unexpected error occurs
     */
    HttpResult<IConfig> createConfig(ConfigData configData) throws ClientException;

    /**
     * Create config.
     *
     * @param configData  the config data
     * @param apiCallback the api callback
     */
    void createConfig(ConfigData configData, ResultApiCallback<IConfig> apiCallback);

    /**
     * Creates cloud config.
     *
     * @param configData the config data
     * @return the Http Status code
     * @throws ClientException thrown if unexpected error occurs
     */
    HttpResult<ICloudConfig> createCloudConfig(ConfigData configData) throws ClientException;

    /**
     * Create cloud config.
     *
     * @param configData  the config data
     * @param apiCallback the api callback
     */
    void createCloudConfig(ConfigData configData, ResultApiCallback<ICloudConfig> apiCallback);

    /**
     * Requests profiles sending ETag's value in request newHeader additionally.
     * If the ETag's value matches the latest one
     * the code 304 is returned.
     *
     * @param eTag the e tag
     * @return the profiles
     * @throws ClientException the client exception
     */
    HttpResult<List<IPreset>> getPresets(SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets presets.
     *
     * @return the presets
     * @throws ClientException the client exception
     */
    HttpResult<List<IPreset>> getPresets() throws ClientException;

    /**
     * Gets presets.
     *
     * @param eTag        the e tag
     * @param apiCallback the api callback
     */
    void getPresets(SDKOptional<ETag> eTag, ResultApiCallback<List<IPreset>> apiCallback);

    /**
     * Gets presets.
     *
     * @param apiCallback the api callback
     */
    void getPresets(ResultApiCallback<List<IPreset>> apiCallback);

    /**
     * Requests profile by its name.
     * If the ETag's value matches the latest one
     * the code 304 is returned.
     *
     * @param profileName the profile name
     * @param eTag        the e tag
     * @return the profile
     * @throws ClientException the client exception
     */
    HttpResult<IPreset> getPreset(String profileName, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Gets preset.
     *
     * @param profileName the profile name
     * @return the preset
     * @throws ClientException the client exception
     */
    HttpResult<IPreset> getPreset(String profileName) throws ClientException;

    /**
     * Gets preset.
     *
     * @param profileName the profile name
     * @param etag        the etag
     * @param apiCallback the api callback
     */
    void getPreset(String profileName, SDKOptional<ETag> etag, ResultApiCallback<IPreset> apiCallback);

    /**
     * Gets preset.
     *
     * @param profileName the profile name
     * @param apiCallback the api callback
     */
    void getPreset(String profileName, ResultApiCallback<IPreset> apiCallback);

    /**
     * Lists secure configs to apply on beacon
     *
     * @param requestDescription - request description
     * @return List of configs to be applied
     * @throws ClientException
     */
    HttpResult<List<SecureSingleConfig>> listSecureConfigs(RequestDescription requestDescription) throws ClientException;

    /**
     * Lists secure configs to apply on beacon
     *
     * @param requestDescription request description
     * @param apiCallback        api callback to receive list of configs to be applied
     */
    void listSecureConfigs(RequestDescription requestDescription, ResultApiCallback<List<SecureSingleConfig>> apiCallback);

    /**
     * Lists secure configs to apply on beacon
     *
     * @param etag Optional param, use {@link SDKOptional#absent()} instead of null value
     * @return List of configs to be applied
     * @throws ClientException
     */
    HttpResult<List<SecureSingleConfig>> listSecureConfigs(SDKOptional<ETag> etag) throws ClientException;

    /**
     * Lists secure configs to apply on beacon
     *
     * @param eTag        Optional param, use {@link SDKOptional#absent()} instead of null value
     * @param apiCallback api callback to receive list of secure configs
     */
    void listSecureConfigs(SDKOptional<ETag> eTag, ResultApiCallback<List<SecureSingleConfig>> apiCallback);

    /**
     * Lists secure configs to apply on beacons with provided unique ids
     *
     * @param uniqueId collection of unique ids
     * @param eTag     Optional param, use {@link SDKOptional#absent()} instead of null value
     * @return List of configs to be applied
     * @throws ClientException
     */
    HttpResult<List<SecureSingleConfig>> listSecureConfigs(Collection<String> uniqueId, SDKOptional<ETag> eTag) throws ClientException;

    /**
     * Lists secure configs to apply on beacons with provided unique ids
     *
     * @param uniqueId    collection of unique ids
     * @param eTag        Optional param, use {@link SDKOptional#absent()} instead of null value
     * @param apiCallback api callback to receive list of secure configs
     */
    void listSecureConfigs(Collection<String> uniqueId, SDKOptional<ETag> eTag, ResultApiCallback<List<SecureSingleConfig>> apiCallback);
}
