package com.kontakt.sdk.android.http.data;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.File;
import java.io.IOException;
import java.util.UUID;

/**
 * Action data provides dataset that may be included to
 * perform CRUD operations with Venue resource.
 */
public class VenueData extends AbstractEntityData {

    private static final String VENUE_ID_PARAMETER = "venueId";
    private static final String NAME_PARAMETER = "name";
    private static final String DESCRIPTION_PARAMETER = "description";
    private static final String LAT_PARAMETER = "lat";
    private static final String LNG_PARAMETER = "lng";

    private VenueData(EntityData entityData) {
        super(entityData);
    }

    /**
     * Creates venue builder for Venue instance creation operation.
     *
     * @param name        the name
     * @param description the description
     * @return the builder
     */
    public static Builder create(final String name, final String description) {
        SDKPreconditions.checkNotNullOrEmpty(name, "Venue name is null");
        SDKPreconditions.checkNotNullOrEmpty(description, "Venue description is null");

        return new Builder(name, description);
    }

    /**
     * Creates venue builder for Venue instance update operation.
     *
     * @param venueId the venue id
     * @return the builder
     */
    public static Builder update(final UUID venueId) {
        SDKPreconditions.checkNotNull(venueId, "Venue Id is null.");
        return new Builder(venueId);
    }

    /**
     * Venue Data Builder.
     */
    public static final class Builder {

        private final EntityData.Builder builder = new EntityData.Builder();

        private Builder(final UUID venueId) {
            builder.addParameter(VENUE_ID_PARAMETER, venueId.toString());
        }

        private Builder(String name, String description) {
            builder.addParameter(NAME_PARAMETER, name)
                    .addParameter(DESCRIPTION_PARAMETER, description);
        }

        /**
         * Adds name.
         *
         * @param name the name
         * @return the builder
         */
        public Builder withName(final String name) {
            builder.addParameter(NAME_PARAMETER, name);
            return this;
        }

        /**
         * Adds description.
         *
         * @param description the description
         * @return the builder
         */
        public Builder withDescription(final String description) {
            builder.addParameter(DESCRIPTION_PARAMETER, description);
            return this;
        }

        /**
         * Adds cover file.
         *
         * @param file            the file
         * @param convertFunction the convert function
         * @return the builder
         * @throws IOException the iO exception
         */
        public Builder withCoverFile(final File file, final SDKFunction<byte[], String> convertFunction) throws IOException {
            SDKPreconditions.checkNotNull(file, "File is null.");
            SDKPreconditions.checkNotNull(file.exists(), "File does not exist");
            SDKPreconditions.checkNotNull(convertFunction, "Convert Function is null.");

            final byte[] bytes = ConversionUtils.convert(file);

            builder.addParameter(Constants.FILE, convertFunction.apply(bytes));
            return this;
        }

        /**
         * Sets latitude
         *
         * @param lat latitude value
         * @return the builder instance
         */
        public Builder withLatitude(final String lat) {
            builder.addParameter(LAT_PARAMETER, lat);
            return this;
        }

        /**
         * Sets longitude
         *
         * @param lng longitude value
         * @return the builder instance
         */
        public Builder withLongitude(final String lng) {
            builder.addParameter(LNG_PARAMETER, lng);
            return this;
        }

        /**
         * Adds venue data.
         *
         * @return the venue data
         */
        public VenueData build() {
            return new VenueData(builder.build());
        }

    }
}
