package com.kontakt.sdk.android.http.data;

import com.kontakt.sdk.android.common.model.IManager;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.UUID;

/**
 * Action data provides data set that may be included to
 * perform CRUD operations with Manager resource.
 */
public class ManagerData extends AbstractEntityData {

    private static final String MANAGER_ID_PARAMETER = "managerId";
    private static final String EMAIL_PARAMETER = "email";
    private static final String FIRST_NAME_PARAMETER = "firstName";
    private static final String LAST_NAME_PARAMETER = "lastName";
    private static final String ROLE_PARAMETER = "role";

    private ManagerData(EntityData entityData) {
        super(entityData);
    }

    /**
     * Create builder with parameters required to create Manager instance.
     *
     * @param firstName the first name
     * @param lastName the last name
     * @param email the email
     * @param role the role
     * @return the builder
     */
    public static Builder create(final String firstName,
                                 final String lastName,
                                 final String email,
                                 final IManager.Role role) {
        SDKPreconditions.checkNotNullOrEmpty(firstName, "First Name is null or empty");
        SDKPreconditions.checkNotNullOrEmpty(lastName, "Last Name is null or empty");
        SDKPreconditions.checkNotNullOrEmpty(email, "Email is null or empty");
        SDKPreconditions.checkNotNull(role, "Role is null");

        return new Builder(firstName, lastName, email, role);
    }

    /**
     * Creates builder for update operation with Manager id.
     *
     * @param managerId the manager id
     * @return the builder instance
     */
    public static Builder update(final UUID managerId) {
        SDKPreconditions.checkNotNull(managerId, "Manager id is null");
        return new Builder(managerId);
    }

    /**
     * The type Builder.
     */
    public static final class Builder {

        private final EntityData.Builder builder = new EntityData.Builder();

        private Builder(final UUID managerId) {
            builder.addParameter(MANAGER_ID_PARAMETER, managerId.toString());
        }

        private Builder(String firstName, String lastName, String email, IManager.Role role) {
            withFirstName(firstName)
                    .withLastName(lastName)
                    .withRole(role);
            builder.addParameter(EMAIL_PARAMETER, email);
        }

        /**
         * Adds first name.
         *
         * @param firstName the first name
         * @return the builder
         */
        public Builder withFirstName(final String firstName) {
            builder.addParameter(FIRST_NAME_PARAMETER, firstName);
            return this;
        }

        /**
         * Adds last name.
         *
         * @param lastName the last name
         * @return the builder
         */
        public Builder withLastName(final String lastName) {
            builder.addParameter(LAST_NAME_PARAMETER, lastName);
            return this;
        }

        /**
         * Adds role.
         *
         * @param role the role
         * @return the builder
         */
        public Builder withRole(final IManager.Role role) {
            SDKPreconditions.checkNotNull(role, "Role is null");
            builder.addParameter(ROLE_PARAMETER, role.name());
            return this;
        }

        /**
         * Builds manager data.
         *
         * @return the manager data
         */
        public ManagerData build() {
            return new ManagerData(builder.build());
        }
    }
}
