package com.kontakt.sdk.android.http.data;

import android.text.TextUtils;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.HttpUtils.toUrlParameter;

/**
 * Entity Data is a container for url parameters included
 * to Http Request.
 */
public final class EntityData {

    private final List<Map.Entry<String, String>> parameterList;

    private EntityData(Builder builder) {
        parameterList = Collections.unmodifiableList(builder.argumentList);
    }

    /**
     * Provides parameter list.
     *
     * @return the parameter list
     */
    List<Map.Entry<String, String>> getParameters() {
        return parameterList;
    }

    /**
     * Entity data Builder.
     */
    public static final class Builder {
        private List<Map.Entry<String, String>> argumentList = new ArrayList<Map.Entry<String, String>>();

        /**
         * Add parameter.
         *
         * @param name  the name
         * @param value the value
         * @return the builder
         */
        public Builder addParameter(final String name, final String value) {
            argumentList.add(toUrlParameter(name, value));
            return this;
        }

        /**
         * Add parameter if not null.
         *
         * @param name  the name
         * @param value the value
         * @return the builder
         */
        public Builder addParameterIfNotNull(final String name, final String value) {
            if (name != null && value != null) {
                return addParameter(name, value);
            }

            return this;
        }

        /**
         * Add parameter.
         *
         * @param name  the name
         * @param value the value
         * @return the builder
         */
        public Builder addParameter(final String name, final int value) {
            return addParameter(name, String.valueOf(value));
        }

        /**
         * Add parameter.
         *
         * @param name  the name
         * @param value the value
         * @return the builder
         */
        public Builder addParameter(final String name, final UUID value) {
            if (value != null) {
                addParameter(name, value.toString());
            }
            return this;
        }

        /**
         * Add parameter.
         *
         * @param name  the name
         * @param value the value
         * @return the builder
         */
        public Builder addParameter(final String name, final double value) {
            return addParameter(name, String.valueOf(value));
        }

        /**
         * Add parameter
         *
         * @param name  parameter name
         * @param value boolean value
         * @return the builder
         */
        public Builder addParameter(final String name, final boolean value) {
            return addParameter(name, String.valueOf(value));
        }

        /**
         * Add parameter.
         *
         * @param name            the name
         * @param file            the file
         * @param convertFunction the convert function
         * @return the builder
         * @throws java.io.IOException the iO exception
         */
        public Builder addParameter(final String name, final File file, final SDKFunction<byte[], String> convertFunction) throws IOException {
            SDKPreconditions.checkNotNull(file, "File is null.");
            SDKPreconditions.checkState(file.exists(), "File does not exist");
            SDKPreconditions.checkNotNull(convertFunction, "Convert function is null.");

            final byte[] fileData = ConversionUtils.convert(file);

            return addParameter(name, convertFunction.apply(fileData));
        }

        /**
         * Add parameter. Values of collection will be separated by "," char
         *
         * @param name       parameter name
         * @param collection collection of strings
         * @return the builder
         */
        public Builder addParameter(final String name, final Collection<String> collection) {
            SDKPreconditions.checkNotNull(collection, "Collection is null.");
            String value = TextUtils.join(",", collection);
            return addParameter(name, value);
        }

        /**
         * Build update data.
         *
         * @return the update data
         */
        public EntityData build() {
            return new EntityData(this);
        }
    }

}
