package com.kontakt.sdk.android.http.data;


import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.ICloudBeacon;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

/**
 * Action data provides dataset that may be included to
 * perform CRUD operations with Device resource.
 */
public final class DeviceData extends AbstractEntityData {

    private static final String DEVICE_TYPE_PARAMETER = "deviceType";
    private static final String UNIQUE_ID_PARAMETER = "uniqueId";
    private static final String PROXIMITY_PARAMETER = "proximity";
    private static final String MAJOR_PARAMETER = "major";
    private static final String MINOR_PARAMETER = "minor";
    private static final String PASSWORD_PARAMETER = "password";
    private static final String TX_POWER_PARAMETER = "txPower";
    private static final String INTERVAL_PARAMETER = "interval";
    private static final String ALIAS_PARAMETER = "alias";
    private static final String NAME_PARAMETER = "name";
    private static final String FIRMWARE_PARAMETER = "firmware";
    private static final String DEFAULT_SSID_AUTH_PARAMETER = "defaultSSIDAuth";
    private static final String DEFAULT_SSID_CRYPT_PARAMETER = "defaultSSIDCrypt";
    private static final String DEFAULT_SSID_NAME_PARAMETER = "defaultSSIDName";
    private static final String DEFAULT_SSID_KEY_PARAMETER = "defaultSSIDKey";
    private static final String WORKING_MODE_PARAMETER = "workingMode";
    private static final String WIFI_SCAN_INTERVAL_PARAMETER = "wifiScanInterval";
    private static final String DATA_SEND_INTERVAL_PARAMETER = "dataSendInterval";
    private static final String BLE_SCAN_INTERVAL_PARAMETER = "bleScanInterval";
    private static final String BLE_SCAN_DURATION_PARAMETER = "bleScanDuration";

    private DeviceData(EntityData entityData) {
        super(entityData);
    }

    /**
     * Builds data set updating Device resource.
     *
     * @param deviceType the device type
     * @param uniqueId   the unique id
     * @return the builder
     */
    public static Builder update(final DeviceType deviceType, final String uniqueId) {
        SDKPreconditions.checkNotNull(deviceType, "Device type is null");
        SDKPreconditions.checkNotNullOrEmpty(uniqueId, "Unique Id is null or empty");
        return new Builder(deviceType, uniqueId);
    }

    /**
     * Device Data Builder.
     */
    public static final class Builder {

        private final EntityData.Builder builder;

        private Builder(final DeviceType deviceType, final String uniqueId) {
            builder = new EntityData.Builder();
            builder.addParameter(DEVICE_TYPE_PARAMETER, deviceType.name());
            builder.addParameter(UNIQUE_ID_PARAMETER, uniqueId);
        }

        /**
         * Adds proximity uUID.
         *
         * @param proximityUUID the proximity uUID
         * @return the builder
         */
        public Builder withProximityUUID(final UUID proximityUUID) {
            if (proximityUUID != null) {
                builder.addParameter(PROXIMITY_PARAMETER, proximityUUID.toString());
            }
            return this;
        }

        /**
         * Adds major.
         *
         * @param major the major
         * @return the builder
         */
        public Builder withMajor(final int major) {
            builder.addParameter(MAJOR_PARAMETER, major);
            return this;
        }

        /**
         * Adds minor.
         *
         * @param minor the minor
         * @return the builder
         */
        public Builder withMinor(final int minor) {
            builder.addParameter(MINOR_PARAMETER, minor);
            return this;
        }

        /**
         * Adds password.
         *
         * @param password the password
         * @return the builder
         */
        public Builder withPassword(final String password) {
            builder.addParameter(PASSWORD_PARAMETER, password);
            return this;
        }

        /**
         * Adds tx power.
         *
         * @param txPower the tx power
         * @return the builder
         */
        public Builder withTxPower(final int txPower) {
            builder.addParameter(TX_POWER_PARAMETER, txPower);
            return this;
        }

        /**
         * Adds interval.
         *
         * @param interval the interval
         * @return the builder
         */
        public Builder withInterval(final int interval) {
            builder.addParameter(INTERVAL_PARAMETER, interval);
            return this;
        }

        /**
         * Adds alias.
         *
         * @param alias the alias
         * @return the builder
         */
        public Builder withAlias(final String alias) {
            builder.addParameter(ALIAS_PARAMETER, alias);
            return this;
        }

        /**
         * Adds name.
         *
         * @param name the name
         * @return the builder
         */
        public Builder withName(final String name) {
            builder.addParameter(NAME_PARAMETER, name);
            return this;
        }

        /**
         * Adds firmware.
         *
         * @param firmware the firmware
         * @return the builder
         */
        public Builder withFirmware(final String firmware) {
            builder.addParameter(FIRMWARE_PARAMETER, firmware);
            return this;
        }

        /**
         * Adds default SSID auth. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDAuth the default sSID auth
         * @return the builder
         */
        public Builder withDefaultSSIDAuth(final String defaultSSIDAuth) {
            builder.addParameter(DEFAULT_SSID_AUTH_PARAMETER, defaultSSIDAuth);
            return this;
        }

        /**
         * Adds default SSID crypt. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDCrypt the default SSID crypt
         * @return the builder
         */
        public Builder withDefaultSSIDCrypt(final String defaultSSIDCrypt) {
            builder.addParameter(DEFAULT_SSID_CRYPT_PARAMETER, defaultSSIDCrypt);
            return this;
        }

        /**
         * Adds default sSID name. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDName the default sSID name
         * @return the builder
         */
        public Builder withDefaultSSIDName(String defaultSSIDName) {
            builder.addParameter(DEFAULT_SSID_NAME_PARAMETER, defaultSSIDName);
            return this;
        }

        /**
         * Adds default SSID key. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDKey the default sSID key
         * @return the builder
         */
        public Builder withDefaultSSIDKey(String defaultSSIDKey) {
            builder.addParameter(DEFAULT_SSID_KEY_PARAMETER, defaultSSIDKey);
            return this;
        }

        /**
         * Adds working mode. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param workingMode the working mode
         * @return the builder
         */
        public Builder withWorkingMode(final ICloudBeacon.WorkingMode workingMode) {
            if (workingMode != null) {
                builder.addParameter(WORKING_MODE_PARAMETER, workingMode.name());
            }
            return this;
        }

        /**
         * Adds wifi scan interval. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param wifiScanInterval the wifi scan interval
         * @return the builder
         */
        public Builder withWifiScanInterval(final int wifiScanInterval) {
            builder.addParameter(WIFI_SCAN_INTERVAL_PARAMETER, wifiScanInterval);
            return this;
        }

        /**
         * Adds data send interval. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param dataSendInterval the data send interval
         * @return the builder
         */
        public Builder withDataSendInterval(final int dataSendInterval) {
            builder.addParameter(DATA_SEND_INTERVAL_PARAMETER, dataSendInterval);
            return this;
        }

        /**
         * With ble scan interval. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param bleScanInterval the ble scan interval
         * @return the builder
         */
        public Builder withBleScanInterval(final int bleScanInterval) {
            builder.addParameter(BLE_SCAN_INTERVAL_PARAMETER, bleScanInterval);
            return this;
        }

        /**
         * Adds ble scan duration. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param bleScanDuration the ble scan duration
         * @return the builder
         */
        public Builder withBleScanDuration(final int bleScanDuration) {
            builder.addParameter(BLE_SCAN_DURATION_PARAMETER, bleScanDuration);
            return this;
        }

        /**
         * With instance id.
         *
         * @param instanceId the instance id
         * @return the builder
         */
        public Builder withInstanceId(final String instanceId) {
            builder.addParameter(Constants.Eddystone.INSTANCE_ID, instanceId);
            return this;
        }

        /**
         * Sets latitude
         *
         * @param latitude latitude value
         * @return the builder instance
         */
        public Builder withLatitude(final String latitude) {
            builder.addParameter(Constants.Devices.LATITUDE, latitude);
            return this;
        }

        /**
         * Sets longitude
         *
         * @param longitude longitude value
         * @return the builder instance
         */
        public Builder withLongitude(final String longitude) {
            builder.addParameter(Constants.Devices.LONGITUDE, longitude);
            return this;
        }

        public Builder force(boolean force){
            builder.addParameter("force", String.valueOf(force));
            return this;
        }

        /**
         * With url.
         *
         * @param url the url
         * @return the builder
         */
        public Builder withUrl(final String url) {
            builder.addParameter(Constants.Eddystone.URL, url);
            return this;
        }

        /**
         * With namespace.
         *
         * @param namespace the namespace
         * @return the builder
         */
        public Builder withNamespace(final String namespace) {
            builder.addParameter(Constants.Eddystone.NAMESPACE, namespace);
            return this;
        }

        /**
         * Sets {@link DeviceProfile} for device
         *
         * @param deviceProfiles Collection of DeviceProfile
         * @return the builder
         */
        public Builder withDeviceProfiles(final Collection<DeviceProfile> deviceProfiles) {
            List<String> deviceProfilesList = new ArrayList<String>();
            for (DeviceProfile deviceProfile : deviceProfiles) {
                deviceProfilesList.add(deviceProfile.name());
            }
            builder.addParameter(Constants.Devices.PROFILES, deviceProfilesList);
            return this;
        }

        /**
         * Adds device data. Reserved only for {@link DeviceType#CLOUD_BEACON}.
         *
         * @return the device data
         */
        public DeviceData build() {
            return new DeviceData(builder.build());
        }
    }
}
