package com.kontakt.sdk.android.http.data;

import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.ICloudBeacon;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

/**
 * Action data provides data set that may be included to
 * perform CRUD operations with Config resource.
 */
public final class ConfigData extends AbstractEntityData {

    private static final String UNIQUE_ID_PARAMETER = "uniqueId";
    private static final String DEVICE_TYPE_PARAMETER = "deviceType";
    private static final String PROXIMITY_PARAMETER = "proximity";
    private static final String MAJOR_PARAMETER = "major";
    private static final String MINOR_PARAMETER = "minor";
    private static final String TX_POWER_PARAMETER = "txPower";
    private static final String INTERVAL_PARAMETER = "interval";
    private static final String PASSWORD_PARAMETER = "password";
    private static final String DEFAULT_SSID_NAME_PARAMETER = "defaultSSIDName";
    private static final String DEFAULT_SSID_KEY_PARAMETER = "defaultSSIDKey";
    private static final String DEFAULT_SSID_AUTH_PARAMETER = "defaultSSIDAuth";
    private static final String DEFAULT_SSID_CRYPT_PARAMETER = "defaultSSIDCrypt";
    private static final String WORKING_MODE_PARAMETER = "workingMode";
    private static final String WIFI_SCAN_INTERVAL_PARAMETER = "wifiScanInterval";
    private static final String DATA_SEND_INTERVAL_PARAMETER = "dataSendInterval";
    private static final String BLE_SCAN_INTERVAL_PARAMETER = "bleScanInterval";
    private static final String BLE_SCAN_DURATION_PARAMETER = "bleScanDuration";
    private static final String NAME_PARAMETER = "name";

    private ConfigData(EntityData entityData) {
        super(entityData);
    }

    /**
     * Creates builder including necessary parameters needed to create the Config for many devices.
     *
     * @param uniqueIds collection of device unique ids
     * @param deviceType one of {@link DeviceType#BEACON} or
     *                   {@link DeviceType#CLOUD_BEACON}
     * @return the builder
     */
    public static Builder create(final Collection<String> uniqueIds, final DeviceType deviceType) {

        return new Builder(uniqueIds, deviceType);
    }

    /**
     * Create builder including necessary parameters needed to create the Config for single device.
     *
     * @param uniqueId device unique id
     * @param deviceType the device type
     * @return the builder
     */
    public static Builder create(final String uniqueId, final DeviceType deviceType) {
        return create(Arrays.asList(uniqueId), deviceType);
    }

    /**
     * Creates builder with required parameters .
     *
     * @param uniqueId the unique id
     * @param deviceType the device type
     * @return the builder
     */
    public static Builder update(final String uniqueId, final DeviceType deviceType) {
         return update(Arrays.asList(uniqueId), deviceType);
    }

    /**
     * Update builder.
     *
     * @param uniqueIds the unique ids
     * @param deviceType the device type
     * @return the builder
     */
    public static Builder update(final Collection<String> uniqueIds, final DeviceType deviceType) {
        return new Builder(uniqueIds, deviceType);
    }

    /**
     * Config data Builder.
     */
    public static final class Builder {

        private final EntityData.Builder builder = new EntityData.Builder();

        private Builder(Collection<String> uniqueIds, DeviceType deviceType) {
            SDKPreconditions.checkNotNullOrEmpty(uniqueIds, "Unique Ids collection is empty");
            SDKPreconditions.checkNotNull(deviceType, "Device type is null");

            for(final String uniqueId : uniqueIds) {
                builder.addParameter(UNIQUE_ID_PARAMETER, uniqueId);
            }

            builder.addParameter(DEVICE_TYPE_PARAMETER, deviceType.name());
        }

        /**
         * Adds proximity UUID.
         *
         * @param proximityUUID the proximity uUID
         * @return the builder
         */
        public Builder withProximityUUID(final UUID proximityUUID) {
            SDKPreconditions.checkNotNull(proximityUUID, "Proximity UUID is null.");
            builder.addParameter(PROXIMITY_PARAMETER, proximityUUID.toString());
            return this;
        }

        /**
         * Adds major to update/create action.
         *
         * @param major the major
         * @return the builder
         */
        public Builder withMajor(final int major) {
            builder.addParameter(MAJOR_PARAMETER, major);
            return this;
        }

        /**
         * Adds minor to update/create action.
         *
         * @param minor the minor
         * @return the builder
         */
        public Builder withMinor(final int minor) {
            builder.addParameter(MINOR_PARAMETER, minor);
            return this;
        }

        /**
         * Adds tx power to update/create action.
         *
         * @param txPower the tx power
         * @return the builder
         */
        public Builder withTxPower(final int txPower) {
            builder.addParameter(TX_POWER_PARAMETER, txPower);
            return this;
        }

        /**
         * With interval to update/create action.
         *
         * @param interval the interval
         * @return the builder
         */
        public Builder withInterval(final int interval) {
            builder.addParameter(INTERVAL_PARAMETER, interval);
            return this;
        }

        /**
         * Adds password to update/create action.
         *
         * @param password the password
         * @return the builder
         */
        public Builder withPassword(final String password) {
            builder.addParameter(PASSWORD_PARAMETER, password);
            return this;
        }

        /**
         * Adds default SSID name. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDName the default SSID name
         * @return the builder
         */
        public Builder withDefaultSSIDName(final String defaultSSIDName) {
            builder.addParameter(DEFAULT_SSID_NAME_PARAMETER, defaultSSIDName);
            return this;
        }

        /**
         * Adds default sSID key. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDKey the default sSID key
         * @return the builder
         */
        public Builder withDefaultSSIDKey(final String defaultSSIDKey) {
            builder.addParameter(DEFAULT_SSID_KEY_PARAMETER, defaultSSIDKey);
            return this;
        }

        /**
         * Adds default SSID auth. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDAuth the default sSID auth
         * @return the builder
         */
        public Builder withDefaultSSIDAuth(final String defaultSSIDAuth) {
            builder.addParameter(DEFAULT_SSID_AUTH_PARAMETER, defaultSSIDAuth);
            return this;
        }

        /**
         * Adds default SSID crypt. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param defaultSSIDCrypt the default SSID crypt
         * @return the builder
         */
        public Builder withDefaultSSIDCrypt(final String defaultSSIDCrypt) {
            builder.addParameter(DEFAULT_SSID_CRYPT_PARAMETER, defaultSSIDCrypt);
            return this;
        }

        /**
         * Adds working mode. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param workingMode the working mode
         * @return the builder
         */
        public Builder withWorkingMode(final ICloudBeacon.WorkingMode workingMode) {
            builder.addParameterIfNotNull(WORKING_MODE_PARAMETER, workingMode == null ? null : workingMode.name());
            return this;
        }

        /**
         * Adds wifi scan interval. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param wifiScanInterval the wifi scan interval
         * @return the builder
         */
        public Builder withWifiScanInterval(final int wifiScanInterval) {
            builder.addParameter(WIFI_SCAN_INTERVAL_PARAMETER, wifiScanInterval);
            return this;
        }

        /**
         * Adds ith data send interval. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param dataSendInterval the data send interval
         * @return the builder
         */
        public Builder withDataSendInterval(final int dataSendInterval) {
            builder.addParameter(DATA_SEND_INTERVAL_PARAMETER, dataSendInterval);
            return this;
        }

        /**
         * Adds ble scan interval. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param bleScanInterval the ble scan interval
         * @return the builder
         */
        public Builder withBleScanInterval(final int bleScanInterval) {
            builder.addParameter(BLE_SCAN_INTERVAL_PARAMETER, bleScanInterval);
            return this;
        }

        /**
         * With ble scan duration. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param bleScanDuration the ble scan duration
         * @return the builder
         */
        public Builder withBleScanDuration(final int bleScanDuration) {
            builder.addParameter(BLE_SCAN_DURATION_PARAMETER, bleScanDuration);
            return this;
        }

        /**
         * Adds name. Reserved for {@link DeviceType#CLOUD_BEACON}.
         *
         * @param name the name
         * @return the builder
         */
        public Builder withName(final String name) {
            builder.addParameter(NAME_PARAMETER, name);
            return this;
        }

        /**
         * With url.
         *
         * @param url the url
         * @return the builder
         */
        public Builder withUrl(final String url) {
            builder.addParameter(Constants.Eddystone.URL, EddystoneUtils.toHexString(EddystoneUtils.serializeUrl(url)));
            return this;
        }


        /**
         * With namespace.
         *
         * @param namespace the namespace
         * @return the builder
         */
        public Builder withNamespace(final String namespace) {
            builder.addParameter(Constants.Eddystone.NAMESPACE, namespace);
            return this;
        }

        /**
         * Sets {@link DeviceProfile} for device
         * @param deviceProfiles Collection of DeviceProfile
         * @return the builder
         */
        public Builder withDeviceProfiles(final Collection<DeviceProfile> deviceProfiles) {
            List<String> deviceProfilesList = new ArrayList<String>();
            for (DeviceProfile deviceProfile : deviceProfiles) {
                deviceProfilesList.add(deviceProfile.name());
            }
            builder.addParameter(Constants.Devices.PROFILES, deviceProfilesList);
            return this;
        }


        /**
         * With instance id.
         *
         * @param instanceId the instance id
         * @return the builder
         */
        public Builder withInstanceId(final String instanceId) {
            builder.addParameter(Constants.Eddystone.INSTANCE_ID, instanceId);
            return this;
        }


        /**
         * With shufled.
         *
         * @param shuffled the shuffled
         * @return the builder
         */
        public Builder withShufled(final boolean shuffled) {
            builder.addParameter(Constants.Config.SHUFFLED, shuffled);
            return this;
        }

        /**
         * Build config data.
         *
         * @return the config data
         */
        public ConfigData build() {
            return new ConfigData(builder.build());
        }
    }

}
