package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.model.IVenue;
import com.kontakt.sdk.android.common.model.Venue;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ApiConstants.Common;
import com.kontakt.sdk.android.http.data.VenueData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.kontakt.sdk.android.http.interfaces.UpdateApiCallback;
import com.kontakt.sdk.android.http.interfaces.VenuesApiAccessor;

import org.json.JSONObject;

import java.util.List;
import java.util.UUID;

import static com.kontakt.sdk.android.http.ApiConstants.Venues.VENUES_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Venues.VENUE_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiMethods.Venues.VENUES_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Venues.VENUE_CREATE;
import static com.kontakt.sdk.android.http.ApiMethods.Venues.VENUE_DELETE;
import static com.kontakt.sdk.android.http.ApiMethods.Venues.VENUE_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Venues.VENUE_GET_IMAGE;
import static com.kontakt.sdk.android.http.ApiMethods.Venues.VENUE_UPDATE;

/**
 * {@link VenuesApiAccessor} implementation.
 */
final class VenuesApiAccessorImpl extends AbstractApiAccessor implements VenuesApiAccessor {

    /**
     * Instantiates a new VenuesApiClientDelegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    VenuesApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<IVenue> getVenue(UUID venueId, SDKOptional<ETag> eTag) throws ClientException {
        final String uri = String.format(VENUE_GET, venueId.toString());

        return getAndTransform(uri,
                RequestDescription.start()
                        .setETag(eTag.isPresent() ? eTag.get() : null)
                        .build(),
                new SDKFunction<JSONObject, IVenue>() {
                    @Override
                    public Venue apply(JSONObject object) {
                        return Venue.from(object);
                    }
                });
    }

    @Override
    public void getVenue(UUID venueId, SDKOptional<ETag> etag, ResultApiCallback<IVenue> apiCallback) {
        final String uri = String.format(VENUE_GET, venueId.toString());

        getAsyncAndRetrieveFromJSONObject(uri,
                DEFAULT_REQUEST_DESCRIPTION,
                HttpUtils.SC_OK,
                apiCallback,
                new SDKFunction<JSONObject, IVenue>() {
                    @Override
                    public Venue apply(JSONObject object) {
                        return Venue.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IVenue> createVenue(final VenueData venue) throws ClientException {
        return createAndTransform(VENUE_CREATE,
                venue,
                new SDKFunction<JSONObject, IVenue>() {
                    @Override
                    public Venue apply(JSONObject object) {
                        return Venue.from(object);
                    }
                });
    }

    @Override
    public void createVenue(VenueData venueData, ResultApiCallback<IVenue> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameters(venueData.getParameters())
                .build();

        postAsyncAndBuildFromJSONObject(VENUE_CREATE,
                requestDescription,
                HttpUtils.SC_CREATED,
                apiCallback,
                new SDKFunction<JSONObject, IVenue>() {
                    @Override
                    public IVenue apply(JSONObject object) {
                        return Venue.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IVenue>> listVenues(RequestDescription requestDescription) throws ClientException {

        return getAndTransformToList(VENUES_GET,
                requestDescription,
                VENUES_PARAMETER,
                new SDKFunction<JSONObject, IVenue>() {
                    @Override
                    public Venue apply(JSONObject object) {
                        return Venue.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IVenue>> listVenues() throws ClientException {
        return listVenues(DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public void listVenues(RequestDescription requestDescription, ResultApiCallback<List<IVenue>> apiCallback) {
        transformToListAsynchronously(VENUES_GET,
                requestDescription,
                HttpUtils.SC_OK,
                VENUES_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IVenue>() {
                    @Override
                    public IVenue apply(JSONObject object) {
                        return Venue.from(object);
                    }
                });
    }

    @Override
    public void listVenues(ResultApiCallback<List<IVenue>> apiCallback) {
        listVenues(DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public HttpResult<FileData> getVenueImage(final UUID venueId) throws ClientException {
        final String venueUri = String.format(VENUE_GET_IMAGE, venueId.toString());
        return getAndTransformByteArray(venueUri,
                DEFAULT_REQUEST_DESCRIPTION,
                new SDKFunction<byte[], FileData>() {
                    @Override
                    public FileData apply(byte[] object) {
                        return FileData.of(object);
                    }
                });
    }

    @Override
    public void getVenueImage(UUID venueId, ResultApiCallback<FileData> apiCallback) {
        final String venueUri = String.format(VENUE_GET_IMAGE, venueId.toString());
        getAsync(venueUri,
                DEFAULT_REQUEST_DESCRIPTION,
                HttpUtils.SC_OK,
                apiCallback,
                BYTE_ARRAY_EXTRACT_FUNCTION,
                new SDKFunction<byte[], FileData>() {
                    @Override
                    public FileData apply(byte[] object) {
                        return FileData.of(object);
                    }
                });
    }

    @Override
    public int updateVenue(VenueData venueData) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addHeader(Common.CONTENT_TYPE, Common.CONTENT_TYPE)
                .addParameters(venueData.getParameters())
                .build();

        return postAndReturnHttpStatus(VENUE_UPDATE,
                requestDescription);
    }

    @Override
    public void updateVenue(VenueData venueData, UpdateApiCallback apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addHeader(Common.CONTENT_TYPE, Common.CONTENT_TYPE)
                .addParameters(venueData.getParameters())
                .build();

        postAsyncAndReturnHttpStatus(VENUE_UPDATE, requestDescription, apiCallback);
    }

    @Override
    public int deleteVenue(UUID venueId) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(VENUE_ID_PARAMETER, venueId.toString())
                .build();

        return postAndReturnHttpStatus(VENUE_DELETE, requestDescription);
    }

    @Override
    public void deleteVenue(UUID venueId, UpdateApiCallback apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(VENUE_ID_PARAMETER, venueId.toString())
                .build();

        postAsyncAndReturnHttpStatus(VENUE_DELETE, requestDescription, apiCallback);
    }
}
