package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;

import org.json.JSONObject;

import java.net.URI;

/**
 * REST API 5 brought the functionality of paging results including:
 * <ul>
 * <li>Devices</li>
 * <li>Venues</li>
 * <li>Configs</li>
 * <li>Devices</li>
 * </ul>
 * Once you request one of the resources above the response contains meta data
 * telling you of what search type your request was.
 */
public final class SearchMeta {

    /**
     * The constant JSON_ENTRY.
     */
    public static final String JSON_ENTRY = "searchMeta";

    private final int startIndex;
    private final int offset;
    private final SDKOptional<URI> previousResultsURI;
    private final SDKOptional<URI> nextResultsURI;
    private final Order order;
    private final OrderBy orderBy;

    private SearchMeta(final Builder builder) {
        this.startIndex = builder.startIndex;
        this.offset = builder.offset;
        this.order = builder.order;
        this.orderBy = builder.orderBy;

        final String previousResults = builder.previousResults;
        this.previousResultsURI = previousResults == null || previousResults.isEmpty() ? SDKOptional.<URI>absent() : SDKOptional.<URI>of(URI.create(builder.previousResults));

        final String nextResults = builder.nextResults;
        this.nextResultsURI = nextResults == null || nextResults.isEmpty() ? SDKOptional.<URI>absent() : SDKOptional.<URI>of(URI.create(builder.nextResults));
    }

    /**
     * Gets start index.
     *
     * @return the start index
     */
    public int getStartIndex() {
        return startIndex;
    }

    /**
     * Gets max result.
     *
     * @return the max result
     */
    public int getOffset() {
        return offset;
    }

    /**
     * Provides answer if request descritpion has previous results.
     *
     * @return the boolean
     */
    public boolean hasPreviousResultsURI() {
        return previousResultsURI.isPresent();
    }

    /**
     * Gets previous results uRI.
     *
     * @return the previous results uRI
     */
    public URI getPreviousResultsURI() {
        return previousResultsURI.get();
    }

    /**
     * Gets next results uRI.
     *
     * @return the next results uRI
     */
    public URI getNextResultsURI() {
        return nextResultsURI.get();
    }

    /**
     * Gets order.
     *
     * @return the order
     */
    public Order getOrder() {
        return order;
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.init()
                .append(startIndex)
                .append(offset)
                .append(previousResultsURI)
                .append(nextResultsURI)
                .append(order.name())
                .append(orderBy.name())
                .build();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return false;
        }

        if (obj == null || !(obj instanceof SearchMeta)) {
            return false;
        }

        final SearchMeta searchMeta = (SearchMeta) obj;

        return this.offset == searchMeta.offset &&
                this.startIndex == searchMeta.startIndex &&
                this.nextResultsURI.equals(searchMeta.nextResultsURI) &&
                this.previousResultsURI.equals(searchMeta.previousResultsURI);
    }

    @Override
    public String toString() {
        return new StringBuilder("SearchMeta[").append("offset: ").append(offset).append(" ")
                .append("startIndex: ").append(startIndex).append(" ")
                .append("nextResultsURI: ").append(nextResultsURI).append(" ")
                .append("previousResultsURI: ").append(previousResultsURI).append("]")
                .toString();
    }

    /**
     * From search meta.
     *
     * @param jsonObject the json object
     * @return the search meta
     */
    public static SearchMeta from(final JSONObject jsonObject) {
        return new Builder()
                .setStartIndex(JSONUtils.getInt(jsonObject, Constants.SearchMeta.START_INDEX, 0))
                .setOffset(JSONUtils.getInt(jsonObject, Constants.SearchMeta.MAX_RESULT, 0))
                .setNextResults(JSONUtils.getString(jsonObject, Constants.SearchMeta.NEXT_RESULTS, ""))
                .setPreviousResults(JSONUtils.getString(jsonObject, Constants.SearchMeta.PREV_RESULTS, ""))
                .setOrder(Order.valueOf(JSONUtils.getString(jsonObject, Constants.SearchMeta.ORDER, Order.ASC.name())))
                .setOrderBy(OrderBy.fromJSON(JSONUtils.getString(jsonObject, Constants.SearchMeta.ORDER_BY, OrderBy.CREATION.name())))
                .build();
    }

    /**
     * Gets order by.
     *
     * @return the order by
     */
    public OrderBy getOrderBy() {
        return orderBy;
    }

    /**
     * Has next results uRI.
     *
     * @return the boolean
     */
    public boolean hasNextResultsURI() {
        return nextResultsURI.isPresent();
    }

    /**
     * The type Builder.
     */
    public static class Builder {
        private int startIndex = 0;
        private int offset = 0;
        private String previousResults = "";
        private String nextResults = "";
        private OrderBy orderBy = OrderBy.CREATION;
        private Order order = Order.ASC;

        /**
         * Sets start index.
         *
         * @param startIndex the start index
         * @return the start index
         */
        public Builder setStartIndex(int startIndex) {
            this.startIndex = startIndex;
            return this;
        }

        /**
         * Sets max result.
         *
         * @param offset the max result
         * @return the max result
         */
        public Builder setOffset(int offset) {
            this.offset = offset;
            return this;
        }

        /**
         * Sets previous results.
         *
         * @param previousResults the previous results
         * @return the previous results
         */
        public Builder setPreviousResults(String previousResults) {
            this.previousResults = previousResults;
            return this;
        }

        /**
         * Sets next results.
         *
         * @param nextResults the next results
         * @return the next results
         */
        public Builder setNextResults(String nextResults) {
            this.nextResults = nextResults;
            return this;
        }

        /**
         * Sets order by.
         *
         * @param orderBy the order by
         * @return the order by
         */
        public Builder setOrderBy(OrderBy orderBy) {
            this.orderBy = orderBy;
            return this;
        }

        /**
         * Sets order.
         *
         * @param order the order
         * @return the order
         */
        public Builder setOrder(Order order) {
            this.order = order;
            return this;
        }

        /**
         * Build search meta.
         *
         * @return the search meta
         */
        public SearchMeta build() {

            return new SearchMeta(this);
        }
    }

}
