package com.kontakt.sdk.android.http;

import android.text.TextUtils;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.model.IFirmware;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.FirmwareApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.squareup.okhttp.Headers;
import com.squareup.okhttp.Response;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.kontakt.sdk.android.common.util.HttpUtils.toUrlParameterList;
import static com.kontakt.sdk.android.http.ApiConstants.Firmware.UNIQUE_ID_PARAMETER;
import static com.kontakt.sdk.android.http.ApiMethods.Firmware.FIRMWARE;
import static com.kontakt.sdk.android.http.ApiMethods.Firmware.FIRMWARE_FILE_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Firmware.FIRMWARE_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Firmware.LAST_FIRMWARE_GET;

/**
 * {@link FirmwareApiAccessor} implementation.
 */
final class FirmwareApiAccessorImpl extends AbstractApiAccessor implements FirmwareApiAccessor {

    /**
     * Instantiates a new FirmwareApiClientDelegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    FirmwareApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<Map<String, IFirmware>> getLatestFirmwareForBeacons(final Set<String> beaconUniqueIds, final SDKOptional<ETag> eTagOptional) throws ClientException {

        List<Map.Entry<String, String>> parameterList = toUrlParameterList(UNIQUE_ID_PARAMETER, beaconUniqueIds);

        Headers.Builder headersBuilder = new Headers.Builder();

        if (eTagOptional.isPresent()) {
            ETag eTag = eTagOptional.get();
            headersBuilder.add(eTag.getRequestName(), eTag.getValue());
        }

        try {
            final Response response = get(LAST_FIRMWARE_GET, headersBuilder.build(), parameterList);

            return transformJSONToEntity(response,
                    HttpUtils.SC_OK,
                    new SDKFunction<JSONObject, Map<String, IFirmware>>() {
                        @Override
                        public Map<String, IFirmware> apply(JSONObject object) {
                            try {
                                return toFirmwareMap(object, beaconUniqueIds);
                            } catch (Exception e) {
                                throw new IllegalStateException(e);
                            }
                        }
                    });
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    @Override
    public HttpResult<Map<String, IFirmware>> getLatestFirmwareForBeacons(Set<String> beaconUniqueIds) throws ClientException {
        return getLatestFirmwareForBeacons(beaconUniqueIds, SDKOptional.<ETag>absent());
    }

    private Map<String, IFirmware> toFirmwareMap(JSONObject object, Set<String> deviceUniqueIds) throws JSONException {
        final Map<String, IFirmware> firmwareMap = new HashMap<String, IFirmware>();
        for (String beaconUniqueId : deviceUniqueIds) {
            firmwareMap.put(beaconUniqueId, null);
            if (object.has(beaconUniqueId)) {
                final JSONObject jsonObject = object.getJSONObject(beaconUniqueId);
                firmwareMap.put(beaconUniqueId,
                        Firmware.from(jsonObject));
            }
        }
        return firmwareMap;
    }

    @Override
    public void getLatestFirmwareForBeacons(final Set<String> deviceUniqueIds,
                                            final SDKOptional<ETag> eTag,
                                            final ResultApiCallback<Map<String, IFirmware>> callback) {
        getAsyncAndRetrieveFromJSONObject(LAST_FIRMWARE_GET,
                DEFAULT_REQUEST_DESCRIPTION,
                HttpUtils.SC_OK,
                callback,
                new SDKFunction<JSONObject, Map<String, IFirmware>>() {
                    @Override
                    public Map<String, IFirmware> apply(JSONObject object) {
                        try {
                            return toFirmwareMap(object, deviceUniqueIds);
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public void getLatestFirmwareForBeacons(Set<String> deviceUniqueIds, ResultApiCallback<Map<String, IFirmware>> callback) {
        getLatestFirmwareForBeacons(deviceUniqueIds, SDKOptional.<ETag>absent(), callback);
    }

    @Override
    public HttpResult<IFirmware> getFirmware(String firmwareName, DeviceType deviceType, SDKOptional<ETag> eTag) throws ClientException {
        final String uri = String.format(FIRMWARE_GET, firmwareName);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameter(Constants.Devices.DEVICE_TYPE, deviceType.name())
                .build();

        return getAndTransform(uri,
                requestDescription,
                new SDKFunction<JSONObject, IFirmware>() {
                    @Override
                    public IFirmware apply(JSONObject object) {
                        return Firmware.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IFirmware> getFirmware(String firmwareName, DeviceType deviceType) throws ClientException {
        return getFirmware(firmwareName, deviceType, SDKOptional.<ETag>absent());
    }

    @Override
    public void getFirmware(String firmwareName, DeviceType deviceType, SDKOptional<ETag> eTag, ResultApiCallback<IFirmware> apiCallback) {
        final String uri = String.format(FIRMWARE_GET, firmwareName);
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addHeader(Constants.Devices.DEVICE_TYPE, deviceType.name())
                .build();
        getAsyncAndRetrieveFromJSONObject(uri,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                new SDKFunction<JSONObject, IFirmware>() {
                    @Override
                    public IFirmware apply(JSONObject object) {
                        return Firmware.from(object);
                    }
                });
    }

    @Override
    public void getFirmware(String firmwareName, DeviceType deviceType, ResultApiCallback<IFirmware> apiCallback) {
        getFirmware(firmwareName, deviceType, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<FileData> fetchFirmwareFileData(String firmwareName, SDKOptional<ETag> eTag) throws ClientException {
        final String uri = String.format(FIRMWARE_FILE_GET, firmwareName);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return getAndTransformByteArray(uri, requestDescription, new SDKFunction<byte[], FileData>() {
            @Override
            public FileData apply(byte[] object) {
                return FileData.of(object);
            }
        });
    }

    @Override
    public HttpResult<FileData> fetchFirmwareFileData(String firmwareName) throws ClientException {
        return fetchFirmwareFileData(firmwareName, SDKOptional.<ETag>absent());
    }

    @Override
    public void fetchFirmwareFileData(String firmwareName, SDKOptional<ETag> etag, ResultApiCallback<FileData> resultApiCallback) {
        final String uri = String.format(FIRMWARE_FILE_GET, firmwareName);

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();

        getAsyncAndRetrieveFromByteArray(uri, requestDescription, resultApiCallback, new SDKFunction<byte[], FileData>() {
            @Override
            public FileData apply(byte[] object) {
                return FileData.of(object);
            }
        });
    }

    @Override
    public void fetchFirmwareFileData(String firmwareName, ResultApiCallback<FileData> resultApiCallback) {
        fetchFirmwareFileData(firmwareName, SDKOptional.<ETag>absent(), resultApiCallback);
    }

    @Override
    public HttpResult<List<IFirmware>> getFirmwares(Set<String> deviceUniqueIds) throws ClientException {
        String parameters = TextUtils.join(",", deviceUniqueIds);
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Constants.UNIQUE_ID, parameters)
                .build();

        return getAndTransform(FIRMWARE, requestDescription, new SDKFunction<JSONObject, List<IFirmware>>() {
            @Override
            public List<IFirmware> apply(JSONObject object) {
                return Firmware.fromList(object);
            }
        });
    }

    @Override
    public void getFirmwares(Set<String> deviceUniqueIds, ResultApiCallback<List<IFirmware>> apiCallback) {

    }
}
