package com.kontakt.sdk.android.http;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.interfaces.SDKPredicate;
import com.kontakt.sdk.android.common.interfaces.SDKThrowableFunction;
import com.kontakt.sdk.android.common.model.CloudConfig;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.ICloudConfig;
import com.kontakt.sdk.android.common.model.IConfig;
import com.kontakt.sdk.android.common.model.IPreset;
import com.kontakt.sdk.android.common.model.Preset;
import com.kontakt.sdk.android.common.model.SecureSingleConfig;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.data.ConfigData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.ConfigurationApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.squareup.okhttp.ResponseBody;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Collection;
import java.util.List;

import static com.kontakt.sdk.android.http.ApiConstants.Configs.CONFIG_JSON_PARAMETER;
import static com.kontakt.sdk.android.http.ApiConstants.Configs.DEVICE_TYPE_PARAMETER;
import static com.kontakt.sdk.android.http.ApiMethods.Config.CONFIGS_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Config.CONFIGS_SECURE_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Config.CONFIG_CREATE;
import static com.kontakt.sdk.android.http.ApiMethods.Profiles.PRESETS;
import static com.kontakt.sdk.android.http.ApiMethods.Profiles.PRESET_GET;

/**
 * {@link ConfigurationApiAccessor} implementation.
 */
final class ConfigurationApiAccessorImpl extends AbstractApiAccessor implements ConfigurationApiAccessor {

    /**
     * Instantiates a new ConfigurationApiClientDelegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    ConfigurationApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<IConfig> createConfig(ConfigData configData) throws ClientException {
        return createAndTransform(CONFIG_CREATE,
                configData,
                new SDKThrowableFunction<ResponseBody, JSONArray, Exception>() {
                    @Override
                    public JSONArray apply(ResponseBody object) throws Exception {
                        return new JSONArray(object.string());
                    }
                },
                new SDKFunction<JSONArray, IConfig>() {
                    @Override
                    public Config apply(JSONArray object) {
                        try {
                            return Config.from(object.getJSONObject(0));
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public void createConfig(ConfigData configData, ResultApiCallback<IConfig> apiCallback) {

        final RequestDescription requestDescription = RequestDescription.start()
                .addParameters(configData.getParameters())
                .build();

        postAsync(CONFIG_CREATE,
                requestDescription,
                new int[]{HttpUtils.SC_CREATED},
                apiCallback,
                new SDKThrowableFunction<ResponseBody, JSONArray, Exception>() {
                    @Override
                    public JSONArray apply(ResponseBody object) throws Exception {
                        return new JSONArray(object.string());
                    }
                },
                new SDKFunction<JSONArray, IConfig>() {
                    @Override
                    public IConfig apply(JSONArray object) {
                        try {
                            return Config.from(object.getJSONObject(0));
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public HttpResult<ICloudConfig> createCloudConfig(ConfigData configData) throws ClientException {
        return createAndTransform(CONFIG_CREATE,
                configData,
                new SDKThrowableFunction<ResponseBody, JSONArray, Exception>() {
                    @Override
                    public JSONArray apply(ResponseBody object) throws Exception {
                        return new JSONArray(object.string());
                    }
                },
                new SDKFunction<JSONArray, ICloudConfig>() {
                    @Override
                    public CloudConfig apply(JSONArray object) {
                        try {
                            return CloudConfig.from(object.getJSONObject(0));
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public void createCloudConfig(ConfigData configData, ResultApiCallback<ICloudConfig> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameters(configData.getParameters())
                .build();

        postAsync(CONFIG_CREATE,
                requestDescription,
                HttpUtils.SC_CREATED,
                apiCallback,
                JSON_ARRAY_EXTRACT_FUNCTION,
                new SDKFunction<JSONArray, ICloudConfig>() {
                    @Override
                    public ICloudConfig apply(JSONArray object) {
                        try {
                            return CloudConfig.from(object.getJSONObject(0));
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public HttpResult<List<IPreset>> getPresets(SDKOptional<ETag> eTag) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return getAndTransformToList(PRESETS, requestDescription,
                new SDKThrowableFunction<ResponseBody, JSONObject, Exception>() {
                    @Override
                    public JSONObject apply(ResponseBody object) throws Exception {
                        return new JSONObject(object.string());
                    }
                },
                new SDKFunction<JSONObject, List<IPreset>>() {
                    @Override
                    public List<IPreset> apply(JSONObject object) {
                        try {
                            return ExtractUtils.toProfileList(object);
                        } catch (Exception e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public HttpResult<List<IPreset>> getPresets() throws ClientException {
        return getPresets(SDKOptional.<ETag>absent());
    }

    @Override
    public void getPresets(SDKOptional<ETag> eTag, ResultApiCallback<List<IPreset>> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        getAsync(PRESETS,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, List<IPreset>>() {
                    @Override
                    public List<IPreset> apply(JSONObject object) {
                        try {
                            return ExtractUtils.toProfileList(object);
                        } catch (JSONException e) {
                            throw new IllegalStateException(e);
                        }
                    }
                });
    }

    @Override
    public void getPresets(ResultApiCallback<List<IPreset>> apiCallback) {
        getPresets(SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<IPreset> getPreset(String profileName, SDKOptional<ETag> eTag) throws ClientException {

        final String uri = String.format(PRESET_GET, profileName);

        final RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();
        return getAndTransform(uri,
                requestDescription,
                new SDKFunction<JSONObject, IPreset>() {
                    @Override
                    public Preset apply(JSONObject object) {
                        return Preset.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IPreset> getPreset(String profileName) throws ClientException {
        return getPreset(profileName, SDKOptional.<ETag>absent());
    }

    @Override
    public void getPreset(String profileName, SDKOptional<ETag> etag, ResultApiCallback<IPreset> apiCallback) {
        final String uri = String.format(PRESET_GET, profileName);

        final RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();

        getAsync(uri,
                requestDescription,
                HttpUtils.SC_OK,
                apiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, IPreset>() {
                    @Override
                    public Preset apply(JSONObject object) {
                        return Preset.from(object);
                    }
                });
    }

    @Override
    public void getPreset(String profileName, ResultApiCallback<IPreset> apiCallback) {
        getPreset(profileName, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> listSecureConfigs(RequestDescription requestDescription) throws ClientException {
        return getAndTransformToList(CONFIGS_SECURE_GET,
                requestDescription,
                CONFIG_JSON_PARAMETER,
                new SDKFunction<JSONObject, SecureSingleConfig>() {
                    @Override
                    public SecureSingleConfig apply(JSONObject object) {
                        return SecureSingleConfig.fromJson(object);
                    }
                });
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> listSecureConfigs(SDKOptional<ETag> etag) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();
        return listSecureConfigs(requestDescription);
    }

    @Override
    public void listSecureConfigs(SDKOptional<ETag> eTag, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();
        listSecureConfigs(requestDescription, apiCallback);
    }

    @Override
    public void listSecureConfigs(RequestDescription requestDescription, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        transformToListAsynchronously(CONFIGS_SECURE_GET,
                requestDescription,
                HttpUtils.SC_OK,
                CONFIG_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, SecureSingleConfig>() {
                    @Override
                    public SecureSingleConfig apply(JSONObject object) {
                        return SecureSingleConfig.fromJson(object);
                    }
                });
    }

    @Override
    public HttpResult<List<SecureSingleConfig>> listSecureConfigs(Collection<String> uniqueId, SDKOptional<ETag> eTag) throws ClientException {
        String endpoint = CONFIGS_SECURE_GET;

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameters(HttpUtils.toUrlParameterList(Constants.UNIQUE_ID, uniqueId))
                .setOffset(uniqueId.size())
                .build();

        return getAndTransformToList(endpoint,
                requestDescription,
                CONFIG_JSON_PARAMETER,
                new SDKFunction<JSONObject, SecureSingleConfig>() {
                    @Override
                    public SecureSingleConfig apply(JSONObject object) {
                        return SecureSingleConfig.fromJson(object);
                    }
                });
    }

    @Override
    public void listSecureConfigs(Collection<String> uniqueId, SDKOptional<ETag> eTag, ResultApiCallback<List<SecureSingleConfig>> apiCallback) {
        String endpoint = CONFIGS_SECURE_GET;

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .addParameters(HttpUtils.toUrlParameterList(Constants.UNIQUE_ID, uniqueId))
                .setOffset(uniqueId.size())
                .build();

        transformToListAsynchronously(endpoint,
                requestDescription,
                HttpUtils.SC_OK,
                CONFIG_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, SecureSingleConfig>() {
                    @Override
                    public SecureSingleConfig apply(JSONObject object) {
                        return SecureSingleConfig.fromJson(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IConfig>> listConfigs(final RequestDescription requestDescription) throws ClientException {
        return getAndTransformToList(CONFIGS_GET,
                requestDescription,
                CONFIG_JSON_PARAMETER,
                new SDKFunction<JSONObject, IConfig>() {
                    @Override
                    public Config apply(JSONObject object) {
                        return Config.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IConfig>> listConfigs() throws ClientException {
        return listConfigs(DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public void listConfigs(RequestDescription requestDescription, ResultApiCallback<List<IConfig>> apiCallback) {
        transformToListAsynchronously(CONFIGS_GET,
                requestDescription,
                HttpUtils.SC_OK,
                CONFIG_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, IConfig>() {
                    @Override
                    public Config apply(JSONObject object) {
                        return Config.from(object);
                    }
                });
    }

    @Override
    public void listConfigs(ResultApiCallback<List<IConfig>> apiCallback) {
        listConfigs(DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public HttpResult<List<ICloudConfig>> listCloudConfigs(RequestDescription requestDescription) throws ClientException {
        return getAndTransformToList(CONFIGS_GET,
                requestDescription,
                CONFIG_JSON_PARAMETER,
                new SDKFunction<JSONObject, ICloudConfig>() {
                    @Override
                    public CloudConfig apply(JSONObject object) {
                        return CloudConfig.from(object);
                    }
                });
    }

    @Override
    public HttpResult<List<ICloudConfig>> listCloudConfigs() throws ClientException {
        return listCloudConfigs(DEFAULT_REQUEST_DESCRIPTION);
    }

    @Override
    public void listCloudConfigs(RequestDescription requestDescription, ResultApiCallback<List<ICloudConfig>> apiCallback) {
        transformToListAsynchronously(CONFIGS_GET,
                requestDescription,
                HttpUtils.SC_OK,
                CONFIG_JSON_PARAMETER,
                apiCallback,
                new SDKFunction<JSONObject, ICloudConfig>() {
                    @Override
                    public CloudConfig apply(JSONObject object) {
                        return CloudConfig.from(object);
                    }
                });
    }

    @Override
    public void listCloudConfigs(ResultApiCallback<List<ICloudConfig>> apiCallback) {
        listCloudConfigs(DEFAULT_REQUEST_DESCRIPTION, apiCallback);
    }

    @Override
    public HttpResult<IConfig> getConfigForDevice(final String deviceUniqueId, final SDKOptional<ETag> eTag) throws ClientException {

        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(DEVICE_TYPE_PARAMETER, DeviceType.BEACON.name())
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return selectFirstOrReturnAbsent(listConfigs(requestDescription),
                new SDKPredicate<IConfig>() {
                    @Override
                    public boolean test(IConfig target) {
                        return target.getDeviceUniqueId().equals(deviceUniqueId);
                    }
                });
    }

    @Override
    public HttpResult<IConfig> getConfigForDevice(String beaconUniqueId) throws ClientException {
        return getConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent());
    }

    @Override
    public void getConfigForDevice(final String deviceUniqueId,
                                   final SDKOptional<ETag> etag,
                                   final ResultApiCallback<IConfig> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(DEVICE_TYPE_PARAMETER, DeviceType.BEACON.name())
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();
        listConfigs(requestDescription, new ResultApiCallback<List<IConfig>>() {
            @Override
            public void onSuccess(HttpResult<List<IConfig>> result) {
                HttpResult<IConfig> httpResult = selectFirstOrReturnAbsent(result, new SDKPredicate<IConfig>() {
                    @Override
                    public boolean test(IConfig object) {
                        return object.getDeviceUniqueId().equals(deviceUniqueId);
                    }
                });

                if (httpResult.isPresent()) {
                    apiCallback.onSuccess(httpResult);
                } else {
                    onFailure(new ClientException("No pending config found for the device."));
                }
            }

            @Override
            public void onFailure(ClientException e) {
                apiCallback.onFailure(e);
            }
        });
    }

    @Override
    public void getConfigForDevice(String beaconUniqueId, ResultApiCallback<IConfig> apiCallback) {
        getConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }

    @Override
    public HttpResult<ICloudConfig> getCloudConfigForDevice(final String beaconUniqueId, final SDKOptional<ETag> eTag) throws ClientException {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(DEVICE_TYPE_PARAMETER, DeviceType.CLOUD_BEACON.name())
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return selectFirstOrReturnAbsent(listCloudConfigs(requestDescription), new SDKPredicate<ICloudConfig>() {
            @Override
            public boolean test(ICloudConfig target) {
                return target.getDeviceUniqueId().equals(beaconUniqueId);
            }
        });
    }

    @Override
    public HttpResult<ICloudConfig> getCloudConfigForDevice(String beaconUniqueId) throws ClientException {
        return getCloudConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent());
    }

    @Override
    public void getCloudConfigForDevice(final String deviceUniqueId, SDKOptional<ETag> eTag, final ResultApiCallback<ICloudConfig> apiCallback) {
        final RequestDescription requestDescription = RequestDescription.start()
                .addParameter(DEVICE_TYPE_PARAMETER, DeviceType.CLOUD_BEACON.name())
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();
        listCloudConfigs(requestDescription, new ResultApiCallback<List<ICloudConfig>>() {
            @Override
            public void onSuccess(HttpResult<List<ICloudConfig>> result) {
                HttpResult<ICloudConfig> httpResult = selectFirstOrReturnAbsent(result, new SDKPredicate<ICloudConfig>() {
                    @Override
                    public boolean test(ICloudConfig object) {
                        return object.getDeviceUniqueId().equals(deviceUniqueId);
                    }
                });

                if (httpResult.isPresent()) {
                    apiCallback.onSuccess(httpResult);
                } else {
                    onFailure(new ClientException("No pending config found for the device."));
                }
            }

            @Override
            public void onFailure(ClientException e) {
                apiCallback.onFailure(e);
            }
        });
    }

    @Override
    public void getCloudConfigForDevice(String beaconUniqueId, ResultApiCallback<ICloudConfig> apiCallback) {
        getCloudConfigForDevice(beaconUniqueId, SDKOptional.<ETag>absent(), apiCallback);
    }

}
