package com.kontakt.sdk.android.http;

import android.util.Base64;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.model.BrowserAction;
import com.kontakt.sdk.android.common.model.ContentAction;
import com.kontakt.sdk.android.common.model.IAction;
import com.kontakt.sdk.android.common.model.IBrowserAction;
import com.kontakt.sdk.android.common.model.IContentAction;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.HttpUtils;
import com.kontakt.sdk.android.common.util.SDKOptional;
import com.kontakt.sdk.android.http.ApiConstants.Actions;
import com.kontakt.sdk.android.http.data.ActionData;
import com.kontakt.sdk.android.http.exception.ClientException;
import com.kontakt.sdk.android.http.interfaces.ActionsApiAccessor;
import com.kontakt.sdk.android.http.interfaces.ResultApiCallback;
import com.kontakt.sdk.android.http.interfaces.UpdateApiCallback;
import com.squareup.okhttp.Response;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.HttpUtils.toUrlParameter;
import static com.kontakt.sdk.android.http.ApiMethods.Actions.ACTION_CREATE;
import static com.kontakt.sdk.android.http.ApiMethods.Actions.ACTION_DELETE;
import static com.kontakt.sdk.android.http.ApiMethods.Actions.ACTION_GET;
import static com.kontakt.sdk.android.http.ApiMethods.Actions.ACTION_GET_BY_UNIQUE_ID;
import static com.kontakt.sdk.android.http.ApiMethods.Actions.ACTION_GET_CONTENT;
import static com.kontakt.sdk.android.http.ApiMethods.Actions.ACTION_UPDATE;

/**
 * {@link ActionsApiAccessor} implementation.
 */
final class ActionsApiAccessorImpl extends AbstractApiAccessor implements ActionsApiAccessor {

    /**
     * Instantiates a new Actions api client delegate.
     *
     * @param apiKey the api key
     * @param apiUrl the api url
     */
    ActionsApiAccessorImpl(String apiKey, String apiUrl) {
        super(apiKey, apiUrl);
    }

    @Override
    public HttpResult<IContentAction> createContentAction(ActionData actionData, final File file) throws ClientException {

        try {
            RequestDescription desc = RequestDescription.start()
                    .addParameters(actionData.getParameters())
                    .addParameter(Actions.FILE_PARAMETER, Base64.encodeToString(ConversionUtils.convert(file), Base64.DEFAULT))
                    .build();
            return createAndTransform(ACTION_CREATE, desc, new SDKFunction<JSONObject, IContentAction>() {
                @Override
                public ContentAction apply(JSONObject object) {
                    return ContentAction.from(object);
                }
            });
        } catch (IOException e) {
            throw new ClientException(e);
        }
    }

    @Override
    public void createContentAction(ActionData actionData, File contentFile, ResultApiCallback<IContentAction> apiCallback) {
        try {
            RequestDescription desc = RequestDescription.start()
                    .addParameters(actionData.getParameters())
                    .addParameter(Actions.FILE_PARAMETER, Base64.encodeToString(ConversionUtils.convert(contentFile), Base64.DEFAULT))
                    .build();

            postAsyncAndBuildFromJSONObject(ACTION_CREATE,
                    desc,
                    HttpUtils.SC_CREATED,
                    apiCallback,
                    new SDKFunction<JSONObject, IContentAction>() {
                        @Override
                        public ContentAction apply(JSONObject object) {
                            return ContentAction.from(object);
                        }
                    });
        } catch (IOException e) {
            apiCallback.onFailure(new ClientException(e));
        }
    }

    @Override
    public HttpResult<IBrowserAction> createBrowserAction(final ActionData actionData) throws ClientException {

        return createAndTransform(ACTION_CREATE, actionData, new SDKFunction<JSONObject, IBrowserAction>() {
            @Override
            public BrowserAction apply(JSONObject object) {
                return BrowserAction.from(object);
            }
        });
    }

    @Override
    public void createBrowserAction(ActionData actionData, ResultApiCallback<IBrowserAction> resultApiCallback) {
        RequestDescription desc = RequestDescription.start()
                .addParameters(actionData.getParameters())
                .build();
        postAsyncAndBuildFromJSONObject(ACTION_CREATE,
                desc,
                HttpUtils.SC_CREATED,
                resultApiCallback,
                new SDKFunction<JSONObject, IBrowserAction>() {
                    @Override
                    public BrowserAction apply(JSONObject object) {
                        return BrowserAction.from(object);
                    }
                });
    }

    @Override
    public HttpResult<IAction> getAction(UUID actionId, SDKOptional<ETag> eTag) throws ClientException {
        final String uri = String.format(ACTION_GET, actionId.toString());

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTag.isPresent() ? eTag.get() : null)
                .build();

        return getAndTransform(uri, requestDescription, new SDKFunction<JSONObject, IAction>() {
            @Override
            public IAction apply(JSONObject object) {
                try {
                    String actionType = object.getString(Constants.Action.ACTION_TYPE);

                    if (Constants.Action.ACTION_TYPE_CONTENT.equals(actionType)) {
                        return ContentAction.from(object);
                    } else if (Constants.Action.ACTION_TYPE_BROWSER.equals(actionType)) {
                        return BrowserAction.from(object);
                    }
                } catch (JSONException ignored) {
                }

                throw new IllegalStateException("Could not find Action Type: " + object.toString());
            }
        });
    }

    @Override
    public HttpResult<IAction> getAction(UUID actionId) throws ClientException {
        return getAction(actionId, SDKOptional.<ETag>absent());
    }

    @Override
    public void getAction(UUID actionId, SDKOptional<ETag> etag, ResultApiCallback<IAction> resultApiCallback) {
        final String uri = String.format(ACTION_GET, actionId.toString());

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(etag.isPresent() ? etag.get() : null)
                .build();

        getAsyncAndRetrieveFromJSONObject(uri,
                requestDescription,
                HttpUtils.SC_OK,
                resultApiCallback,
                new SDKFunction<JSONObject, IAction>() {
                    @Override
                    public IAction apply(JSONObject object) {
                        try {
                            String actionType = object.getString(Constants.Action.ACTION_TYPE);

                            if (Constants.Action.ACTION_TYPE_CONTENT.equals(actionType)) {
                                return ContentAction.from(object);
                            } else if (Constants.Action.ACTION_TYPE_BROWSER.equals(actionType)) {
                                return BrowserAction.from(object);
                            }
                        } catch (JSONException ignored) {
                        }
                        throw new IllegalStateException("Could not find Action Type: " + object.toString());
                    }
                });
    }

    @Override
    public void getAction(UUID actionId, ResultApiCallback<IAction> resultApiCallback) {
        getAction(actionId, SDKOptional.<ETag>absent(), resultApiCallback);
    }

    @Override
    public int updateAction(final UUID actionId, final File file) throws ClientException {
        try {
            RequestDescription requestDescription = RequestDescription.start()
                    .addParameter(Actions.ACTION_ID_PARAMETER, actionId.toString())
                    .addParameter(Actions.FILE_PARAMETER, Base64.encodeToString(ConversionUtils.convert(file), Base64.DEFAULT))
                    .addHeader(ApiConstants.Common.CONTENT_TYPE, ApiConstants.Common.X_WWW_FORM_URL_ENCODED)
                    .build();

            return postAndReturnHttpStatus(ACTION_UPDATE, requestDescription);
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    @Override
    public void updateAction(UUID actionId, File file, UpdateApiCallback callback) {
        try {
            RequestDescription requestDescription = RequestDescription.start()
                    .addParameter(Actions.ACTION_ID_PARAMETER, actionId.toString())
                    .addParameter(Constants.FILE, Base64.encodeToString(ConversionUtils.convert(file), Base64.DEFAULT))
                    .addHeader(ApiConstants.Common.CONTENT_TYPE, ApiConstants.Common.X_WWW_FORM_URL_ENCODED)
                    .build();

            postAsyncAndReturnHttpStatus(ACTION_UPDATE, requestDescription, callback);
        } catch (Exception e) {
            callback.onFailure(new ClientException(e));
        }
    }

    @Override
    public void updateAction(UUID actionId, String url, UpdateApiCallback callback) {
        try {
            RequestDescription requestDescription = RequestDescription.start()
                    .addParameter(Actions.ACTION_ID_PARAMETER, actionId.toString())
                    .addParameter(Actions.URL, url)
                    .build();
            postAsyncAndReturnHttpStatus(ACTION_UPDATE, requestDescription, callback);
        } catch (Exception e) {
            callback.onFailure(new ClientException(e));
        }
    }

    @Override
    public int updateAction(UUID actionId, String url) throws ClientException {
        try {
            RequestDescription requestDescription = RequestDescription.start()
                    .addParameter(Actions.ACTION_ID_PARAMETER, actionId.toString())
                    .addParameter(Actions.URL, url)
                    .build();

            return postAndReturnHttpStatus(ACTION_UPDATE, requestDescription);
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    @Override
    public int deleteAction(UUID actionId) throws ClientException {

        List<Map.Entry<String, String>> arguments = new ArrayList<Map.Entry<String, String>>(1);
        arguments.add(toUrlParameter(Actions.ACTION_ID_PARAMETER, actionId.toString()));

        try {
            final Response httpResponse = post(ACTION_DELETE, arguments);
            return httpStatusCode(httpResponse);
        } catch (Exception e) {
            throw new ClientException(e);
        }
    }

    @Override
    public void deleteAction(UUID actionId, UpdateApiCallback apiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Actions.ACTION_ID_PARAMETER, actionId.toString())
                .build();

        postAsyncAndReturnHttpStatus(ACTION_DELETE, requestDescription, apiCallback);
    }

    @Override
    public HttpResult<FileData> getActionContent(UUID actionId, final SDKOptional<ETag> eTagOptional) throws ClientException {
        String uri = String.format(ACTION_GET_CONTENT, actionId.toString());
        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTagOptional.isPresent() ? eTagOptional.get() : null)
                .build();
        return getAndTransformByteArray(uri, requestDescription, new SDKFunction<byte[], FileData>() {
            @Override
            public FileData apply(byte[] object) {
                return FileData.of(object);
            }
        });
    }

    @Override
    public void getActionContent(UUID actionId,
                                 SDKOptional<ETag> eTagSDKOptional,
                                 ResultApiCallback<FileData> resultApiCallback) {
        String uri = String.format(ACTION_GET_CONTENT, actionId.toString());

        RequestDescription requestDescription = RequestDescription.start()
                .setETag(eTagSDKOptional.isPresent() ? eTagSDKOptional.get() : null)
                .build();

        getAsyncAndRetrieveFromByteArray(uri,
                requestDescription,
                resultApiCallback,
                new SDKFunction<byte[], FileData>() {
                    @Override
                    public FileData apply(byte[] object) {
                        return FileData.of(object);
                    }
                });
    }

    @Override
    public HttpResult<List<IAction>> getActionsForDevice(String deviceUniqueId) throws ClientException {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Actions.UNIQUE_ID_PARAMETER, deviceUniqueId)
                .build();

        return getAndTransformToList(ACTION_GET_BY_UNIQUE_ID,
                requestDescription,
                Actions.ACTIONS,
                new SDKFunction<JSONObject, IAction>() {
                    @Override
                    public IAction apply(JSONObject object) {
                        return ExtractUtils.extractAction(object);
                    }
                });

    }

    @Override
    public void getActionsForDevice(String deviceUniqueId, ResultApiCallback<List<IAction>> resultApiCallback) {
        RequestDescription requestDescription = RequestDescription.start()
                .addParameter(Actions.UNIQUE_ID_PARAMETER, deviceUniqueId)
                .build();

        getAsync(ACTION_GET_BY_UNIQUE_ID,
                requestDescription,
                new int[]{HttpUtils.SC_OK, HttpUtils.SC_NO_CONTENT},
                resultApiCallback,
                JSON_OBJECT_EXTRACT_FUNCTION,
                new SDKFunction<JSONObject, List<IAction>>() {
                    @Override
                    public List<IAction> apply(JSONObject object) {
                        return ExtractUtils.extractActions(object);
                    }
                });
    }
}
