package com.juphoon.cloud;

import android.support.annotation.IntDef;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;

/**
 * 群组模块
 *
 * @author juphoon
 */
public abstract class JCGroup {

    static final String TAG = JCGroup.class.getSimpleName();

    /**
     * 原因
     */
    @IntDef({REASON_NONE, REASON_NOT_LOGIN, REASON_CALL_FUNCTION_ERROR, REASON_TIME_OUT,
            REASON_NETWORK, REASON_PARAM_INVALID, REASON_OTHER})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Reason {
    }

    /**
     * 无异常
     */
    public static final int REASON_NONE = 0;
    /**
     * 未登陆
     */
    public static final int REASON_NOT_LOGIN = 1;
    /**
     * 函数调用失败
     */
    public static final int REASON_CALL_FUNCTION_ERROR = 2;
    /**
     * 超时
     */
    public static final int REASON_TIME_OUT = 3;
    /**
     * 网络异常
     */
    public static final int REASON_NETWORK = 4;
    /**
     * 参数错误
     */
    public static final int REASON_PARAM_INVALID = 5;
    /**
     * 其他错误
     */
    public static final int REASON_OTHER = 100;

    /**
     * 群组状态变更
     */
    @IntDef({GROUP_CHANGE_STATE_NONE, GROUP_CHANGE_STATE_ADD, GROUP_CHANGE_STATE_UPDATE,
            GROUP_CHANGE_STATE_REMOVE})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ChangeState {
    }

    /**
     * 无
     */
    public static final int GROUP_CHANGE_STATE_NONE = 0;
    /**
     * 新增
     */
    public static final int GROUP_CHANGE_STATE_ADD = 1;
    /**
     * 更新
     */
    public static final int GROUP_CHANGE_STATE_UPDATE = 2;
    /**
     * 删除
     */
    public static final int GROUP_CHANGE_STATE_REMOVE = 3;

    /**
     * 群组成员类型
     */
    @IntDef({GROUP_MEMBER_TYPE_OWNER, GROUP_MEMBER_TYPE_MANAGER, GROUP_MEMBER_TYPE_MEMBER})
    @Retention(RetentionPolicy.SOURCE)
    public @interface MemberType {
    }

    /**
     * 拥有者
     */
    public static final int GROUP_MEMBER_TYPE_OWNER = 0;
    /**
     * 管理员
     */
    public static final int GROUP_MEMBER_TYPE_MANAGER = 1;
    /**
     * 群成员
     */
    public static final int GROUP_MEMBER_TYPE_MEMBER = 2;

    private static JCGroup sGroup;

    /**
     *  创建 JCGroup 对象
     *  @param client JCClient 对象
     *  @param callback JCGroupCallback 回调接口，用于接收 JCGroup 相关通知
     *  @return 返回 JCGroup 对象
     */
    public static JCGroup create(JCClient client, JCGroupCallback callback) {
        if (sGroup != null) {
            return sGroup;
        }
        sGroup = new JCGroupImpl(client, callback);
        return sGroup;
    }

    /**
     * 销毁 JCGroup 对象
     */
    public static void destroy() {
        if (sGroup != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sGroup.destroyObj();
                    sGroup = null;
                }
            });
        }
    }

    /**
     * 销毁对象
     */
    protected abstract void destroyObj();

    /**
     *  获取当前用户所有加入的群列表，结果通过 JCGroupCallback 中相应回调返回
     *  @param updateTime 最新一次记录的群列表服务器更新时间
     *  @return 返回操作id
     */
    public abstract int fetchGroups(long updateTime);

    /**
     *  刷新群组信息
     *  @param groupId 群标识
     *  @param updateTime 最新一次记录的该群服务器更新时间
     *  @return 返回操作id
     */
    public abstract int fetchGroupInfo(String groupId, long updateTime);

    /**
     *  创建群
     *  @param members JCGroupMemeber 队列
     *  @param groupName 群名称
     *  @return 返回操作id
     */
    public abstract int createGroup(List<JCGroupMember> members, String groupName);

    /**
     *  更新群
     *  @param groupItem JCGroupItem 对象，其中 JCGroupItem 中 changeState 值不影响具体操作
     *  @return 返回操作id
     */
    public abstract int updateGroup(JCGroupItem groupItem);

    /**
     *  解散群组，Owner才能解散群组
     *  @param groupId 群标识
     *  @return 返回操作id
     */
    public abstract int dissolve(String groupId);

    /**
     *  离开群组
     *  @param groupId 群标识
     *  @return 返回操作id
     */
    public abstract int leave(String groupId);

    /**
     *  更新昵称
     *  @param selfInfo JCGroupMember 对象，请传入 groupId，displayName，memberType(保持不变)
     *  @return 返回操作id
     */
    public abstract int updateSelfInfo(JCGroupMember selfInfo);

    /**
     *  操作成员
     *  @param groupId 群标识
     *  @param members JCGroupMemeber 对象列表，通过 changeState 值来表明增加，更新，删除成员操作
     *  @return 返回操作id
     */
    public abstract int dealMembers(String groupId, List<JCGroupMember> members);

    /**
     * 添加回调
     *
     * @param callback JCGroupCallback 接口对象
     */
    abstract void addCallback(JCGroupCallback callback);

    /**
     * 删除回调
     *
     * @param callback JCGroupCallback 接口对象
     */
    abstract void removeCallback(JCGroupCallback callback);

}
