package com.juphoon.cloud;

import android.support.annotation.IntDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * 文件存储模块
 *
 * @author juphoon
 */
public abstract class JCStorage {

    /**
     * 文件传输错误枚举
     */
    @IntDef({REASON_NONE, REASON_NOT_LOGIN, REASON_TIMEOUT, REASON_NETWORK, REASON_OTHER, REASON_TOOLARGE, REASON_EXPIRE})
    @Retention(RetentionPolicy.SOURCE)
    public @interface StorageReason {
    }

    /**
     * 无异常
     */
    public static final int REASON_NONE = 0;
    /**
     * 未登陆
     */
    public static final int REASON_NOT_LOGIN = 1;
    /**
     * 超时
     */
    public static final int REASON_TIMEOUT = 2;
    /**
     * 网络异常
     */
    public static final int REASON_NETWORK = 3;
    /**
     * 文件太大
     */
    public static final int REASON_TOOLARGE = 4;
    /**
     * 文件过期
     */
    public static final int REASON_EXPIRE = 5;
    /**
     * 其他错误
     */
    public static final int REASON_OTHER = 100;

    /**
     * 文件传输状态枚举
     */
    @IntDef({ITEM_STATE_INIT, ITEM_STATE_TRANSFERRING, ITEM_STATE_OK, ITEM_STATE_FAIL, ITEM_STATE_CANCEL})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ItemState {
    }

    /**
     * 文件初始状态
     */
    public static final int ITEM_STATE_INIT = 0;
    /**
     * 文件传输中状态
     */
    public static final int ITEM_STATE_TRANSFERRING = 1;
    /**
     * 文件传输成功状态
     */
    public static final int ITEM_STATE_OK = 2;
    /**
     * 文件传输失败状态
     */
    public static final int ITEM_STATE_FAIL = 3;
    /**
     * 文件传输取消状态
     */
    public static final int ITEM_STATE_CANCEL = 4;

    /**
     * 文件传输方向
     */
    @IntDef({DIRECTION_UPLOAD, DIRECTION_DOWNLOAD})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ItemDirection {
    }

    /**
     * 上传
     */
    public static final int DIRECTION_UPLOAD = 0;
    /**
     * 下载
     */
    public static final int DIRECTION_DOWNLOAD = 1;

    /**
     * 文件大小限制
     */
    static final int MAX_FILE_SIZE = 100 * 1024 * 1024;
    /**
     * 文件过期
     */
    static final int FILE_EXIPIRE = 7 * 24 * 3600;

    private static JCStorage sStorage;

    /**
     * 创建 JCStorage 对象
     *
     * @param client   JCClient 对象
     * @param callback JCStorageCallback 回调接口，用于接收 JCStorage 相关通知
     * @return 返回 JCStorage 对象
     */
    public static JCStorage create(JCClient client, JCStorageCallback callback) {
        if (sStorage != null) {
            return sStorage;
        }
        sStorage = new JCStorageImpl(client, callback);
        return sStorage;
    }

    /**
     * 销毁 JCStorage 对象
     */
    public static void destroy() {
        if (sStorage != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sStorage.destroyObj();
                    sStorage = null;
                }
            });
        }
    }

    /**
     * 销毁对象
     */
    protected abstract void destroyObj();

    /**
     * 上传文件
     *
     * @param path 文件路径
     * @return 返回 JCStorageItem 对象, 文件大小不要超过100MB，异常返回 null
     */
    public abstract JCStorageItem uploadFile(String path);

    /**
     * 上传文件
     *
     * @param uri      文件地址
     * @param savePath 本地文件保存地址
     * @return 返回 JCStorageItem 对象，异常返回 null
     */
    public abstract JCStorageItem downloadFile(String uri, String savePath);

    /**
     * 取消正在进行的文件上传下载
     *
     * @param item JCStorageItem对象，由 uploadFile，downloadFile 返回
     * @return 成功返回 true，失败返回 false
     */
    public abstract boolean cancelFile(JCStorageItem item);

}
