package com.juphoon.cloud;

import android.text.TextUtils;

import com.justalk.cloud.lemon.MtcGroup;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.justalk.cloud.lemon.MtcGroupConstants.EN_MTC_GROUP_RELATION_MANAGER;
import static com.justalk.cloud.lemon.MtcGroupConstants.EN_MTC_GROUP_RELATION_OWNER;

class JCGroupImpl extends JCGroup implements MtcEngine.MtcNotifyListener {

    private static final String USER_ID_KEY = "UserId";

    private JCClient mClient;
    private List<JCGroupCallback> mCallbacks = new ArrayList<>();
    private Map<Integer, String> mMapCookieAndGroupId = new HashMap<>();

    JCGroupImpl(JCClient client, JCGroupCallback callback) {
        if (client == null) {
            JCLog.error(TAG, "no JCClient.");
            throw new RuntimeException("JCClient cannot be null!");
        }
        if (callback == null) {
            JCLog.error(TAG, "no JCGroupCallback.");
            throw new RuntimeException("JCGroupCallback cannot be null!");
        }
        if (client.getState() == JCClient.STATE_NOT_INIT) {
            JCLog.error(TAG, "JCClient not initialized.");
            return;
        }
        mClient = client;
        MtcEngine.getInstance().addMtcNotifyListener(this);
        addCallback(callback);
    }

    @Override
    protected void destroyObj() {
        mCallbacks.clear();
        MtcEngine.getInstance().removeMtcNotifyListener(this);
        mClient = null;
    }

    @Override
    public int fetchGroups(long updateTime) {
        JCParam.GroupFetch groupFetchParms = new JCParam.GroupFetch();
        groupFetchParms.updateTime = updateTime;
        JCResult result = MtcEngine.getInstance().fetchGroup(groupFetchParms);
        if (result.succ) {
            logInfo("拉取群列表 操作号:%d", result.cookie);
        } else {
            logError("拉取群列表失败");
            notifyFetchGroups(result.cookie, false, REASON_CALL_FUNCTION_ERROR, null, 0, false);
        }
        return result.cookie;
    }

    @Override
    public int fetchGroupInfo(String groupId, long updateTime) {
        JCParam.GroupFetch groupFetchParms = new JCParam.GroupFetch();
        groupFetchParms.groupId = groupId;
        groupFetchParms.updateTime = updateTime;
        JCResult result = MtcEngine.getInstance().fetchGroup(groupFetchParms);
        if (result.succ) {
            logInfo("拉取群详情 操作号:%d", result.cookie);
        } else {
            logError("拉取群详情失败");
            notifyFetchGroupInfo(result.cookie, false, REASON_CALL_FUNCTION_ERROR,
                    new JCGroupItem(groupId, null, GROUP_CHANGE_STATE_NONE), null, 0, false);
        }
        return result.cookie;
    }

    @Override
    public int createGroup(List<JCGroupMember> members, String groupName) {
        if (members == null || members.size() == 0) {
            logError("请传入群成员");
            int cookie = MtcEngine.getInstance().genCookie();
            notifyCreateGroup(cookie, false, REASON_PARAM_INVALID, null);
            return cookie;
        }
        if (groupName == null) {
            logError("请输入群名字");
            int cookie = MtcEngine.getInstance().genCookie();
            notifyCreateGroup(cookie, false, REASON_PARAM_INVALID, null);
            return cookie;
        }
        JCParam.GroupCreate groupCreate = new JCParam.GroupCreate();
        groupCreate.groupName = groupName;
        groupCreate.members = new HashSet<>();
        for (JCGroupMember groupMember : members) {
            if (TextUtils.isEmpty(groupMember.userId)) {
                logError("过滤长度为0用户标识");
                continue;
            }
            if (TextUtils.equals(mClient.getUserId(), groupMember.userId)) {
                logError("过滤自己");
            }
            JCParam.GroupCreateMember createMember = new JCParam.GroupCreateMember();
            createMember.userId = groupMember.userId;
            createMember.displayName = groupMember.displayName;
            createMember.memberType = translateToMtcMemberType(groupMember.memberType);
            createMember.tag = genTag(groupMember.userId);
            groupCreate.members.add(createMember);
        }
        JCParam.GroupCreateMember selfMember = new JCParam.GroupCreateMember();
        selfMember.userId = mClient.getUserId();
        selfMember.memberType = translateToMtcMemberType(GROUP_MEMBER_TYPE_OWNER);
        selfMember.displayName = TextUtils.isEmpty(mClient.displayName) ? mClient.getUserId() : mClient.displayName;
        selfMember.tag = genTag(mClient.getUserId());
        groupCreate.members.add(selfMember);

        JCResult result = MtcEngine.getInstance().createGroup(groupCreate);
        if (result.succ) {
            logInfo("创建群 操作号:%d", result.cookie);
        } else {
            logError("创建群失败");
            notifyCreateGroup(result.cookie, false, REASON_CALL_FUNCTION_ERROR, null);
        }
        return result.cookie;
    }

    @Override
    public int updateGroup(JCGroupItem groupItem) {
        JCParam.GroupUpdate groupUpdateParms = new JCParam.GroupUpdate();
        groupUpdateParms.groupId = groupItem.groupId;
        groupUpdateParms.groupName = groupItem.name;

        JCResult result = MtcEngine.getInstance().updateGroup(groupUpdateParms);
        if (result.succ) {
            logInfo("更新群 操作号:%d", result.cookie);
        } else {
            logError("更新群失败");
            notifyUpdateGroup(result.cookie, false, REASON_CALL_FUNCTION_ERROR, groupItem.groupId);
        }
        return result.cookie;
    }

    @Override
    public int dissolve(String groupId) {
        JCParam.GroupDissolve groupDissolveParms = new JCParam.GroupDissolve();
        groupDissolveParms.groupId = groupId;

        JCResult result = MtcEngine.getInstance().dissolveGroup(groupDissolveParms);
        if (result.succ) {
            logInfo("解散群 操作号:%d", result.cookie);
        } else {
            logError("解散群失败");
            notifyDissolve(result.cookie, false, REASON_CALL_FUNCTION_ERROR, groupId);
        }
        return result.cookie;
    }

    @Override
    public int leave(String groupId) {
        JCParam.GroupLeave groupLeaveParms = new JCParam.GroupLeave();
        groupLeaveParms.groupId = groupId;
        groupLeaveParms.userId = mClient.getUserId();

        JCResult result = MtcEngine.getInstance().leaveGroup(groupLeaveParms);
        mMapCookieAndGroupId.put(result.cookie, groupId);
        if (result.succ) {
            logInfo("离开群 操作号:%d", result.cookie);
        } else {
            logError("离开群失败");
            notifyLeave(result.cookie, false, REASON_CALL_FUNCTION_ERROR);
        }
        return result.cookie;
    }

    @Override
    public int updateSelfInfo(JCGroupMember selfInfo) {
        JCParam.GroupUpdateSelf groupUpdateSelfParms = new JCParam.GroupUpdateSelf();
        groupUpdateSelfParms.groupId = selfInfo.groupId;
        groupUpdateSelfParms.userId = mClient.getUserId();
        groupUpdateSelfParms.displayName = selfInfo.displayName;
        groupUpdateSelfParms.memberType = translateToMtcMemberType(selfInfo.memberType);
        groupUpdateSelfParms.tag = genTag(mClient.getUserId());

        JCResult result = MtcEngine.getInstance().updateSelf(groupUpdateSelfParms);
        if (result.succ) {
            logInfo("更新自己群信息 操作号:%d", result.cookie);
        } else {
            logError("更新自己群信息");
            notifyDealMembers(result.cookie, false, REASON_CALL_FUNCTION_ERROR);
        }
        return result.cookie;
    }

    @Override
    public int dealMembers(String groupId, List<JCGroupMember> members) {
        JCParam.GroupDealMembers groupDealMembersParms = new JCParam.GroupDealMembers();
        groupDealMembersParms.groupId = groupId;
        groupDealMembersParms.dealMembers = new HashSet<>();
        for (JCGroupMember member : members) {
            JCParam.GroupDealMember groupDealMember = new JCParam.GroupDealMember();
            groupDealMember.userId = member.userId;
            groupDealMember.dealType = translateToDealMemberType(member.changeState);
            groupDealMember.displayName = member.displayName;
            groupDealMember.memberType = translateToMtcMemberType(member.memberType);
            groupDealMember.tag = genTag(member.userId);
            groupDealMembersParms.dealMembers.add(groupDealMember);
        }

        JCResult result = MtcEngine.getInstance().dealGroupMembers(groupDealMembersParms);
        if (result.succ) {
            logInfo("群成员处理 操作号:%d", result.cookie);
        } else {
            logError("群成员处理失败");
            notifyDealMembers(result.cookie, false, REASON_CALL_FUNCTION_ERROR);
        }
        return result.cookie;
    }

    @Override
    void addCallback(JCGroupCallback callback) {
        mCallbacks.add(callback);
    }

    @Override
    void removeCallback(JCGroupCallback callback) {
        mCallbacks.remove(callback);
    }

    @Override
    public void onNotify(JCNotify notify) {
        if (notify.type != JCNotify.GROUP) {
            return;
        }
        logInfo("onNotify name:%s", notify.groupNotify.type);
        JCNotify.Group groupNotify = notify.groupNotify;
        if (groupNotify.type == JCNotify.GROUP_TYPE_CREATE_OK) {
            JCGroupItem groupItem = new JCGroupItem(groupNotify.createOk.groupId, groupNotify.createOk.name, GROUP_CHANGE_STATE_ADD);
            notifyCreateGroup(notify.cookie, true, REASON_NONE, groupItem);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_CREATE_FAIL) {
            notifyCreateGroup(notify.cookie, false, REASON_OTHER, null);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_FETCH_GROUP_OK) {
            List<JCGroupItem> groups = new ArrayList<>();
            for (JCNotify.Group.NotifyItem notifyItem : groupNotify.fetchGroupOk.groupItems) {
                groups.add(new JCGroupItem(notifyItem.groupId, notifyItem.name,
                        translateFromNotifyChangeType(notifyItem.changeType)));
            }
            notifyFetchGroups(notify.cookie, true, REASON_NONE, groups,
                    groupNotify.fetchGroupOk.updateTime, groupNotify.fetchGroupOk.fullUpdate);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_FETCH_GROUP_FAIL) {
            notifyFetchGroups(notify.cookie, false, REASON_OTHER, null, 0, false);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_FETCH_GROUP_INFO_OK) {
            JCGroupItem groupItem = new JCGroupItem(groupNotify.fetchGroupInfoOk.groupId,
                    groupNotify.fetchGroupInfoOk.name, GROUP_CHANGE_STATE_UPDATE);
            List<JCGroupMember> members = new ArrayList<>();
            for (JCNotify.Group.NotifyMember member : groupNotify.fetchGroupInfoOk.members) {
                JCGroupMember groupMember = new JCGroupMember(groupNotify.fetchGroupInfoOk.groupId,
                        getUserIdFromTag(member.tag), member.displayName,
                        translateFromMtcMemberType(member.memberType),
                        translateFromNotifyChangeType(member.changeType));
                groupMember.uid = member.uid;
                members.add(groupMember);
            }
            notifyFetchGroupInfo(notify.cookie, true, REASON_NONE, groupItem, members,
                    groupNotify.fetchGroupInfoOk.updateTime,
                    groupNotify.fetchGroupInfoOk.fullUpdate);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_FETCH_GROUP_INFO_FAIL) {
            notifyFetchGroupInfo(notify.cookie, false, REASON_OTHER, null, null, 0, false);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_LEAVE_OK) {
            notifyLeave(notify.cookie, true, REASON_NONE);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_LEAVE_FAIL) {
            notifyLeave(notify.cookie, false, REASON_OTHER);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_DISSOLVE_OK) {
            notifyDissolve(notify.cookie, true, REASON_NONE, groupNotify.dissolveOk.groupId);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_DISSOLVE_FAIL) {
            notifyDissolve(notify.cookie, false, REASON_OTHER, groupNotify.dissolveFail.groupId);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_GROUP_LIST_CHANGE) {
            notifyGroupListChange();
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_GROUP_INFO_CHANGE) {
            notifyGroupInfoChange(groupNotify.groupInfoChange.groupId);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_DEAL_MEMBERS_OK) {
            notifyDealMembers(notify.cookie, true, REASON_NONE);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_DEAL_MEMBERS_FAIL) {
            notifyDealMembers(notify.cookie, false, REASON_OTHER);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_CHANGE_GROUP_PROP_OK) {
            notifyUpdateGroup(notify.cookie, true, REASON_NONE, groupNotify.changeGroupPropOk.groupId);
        } else if (groupNotify.type == JCNotify.GROUP_TYPE_CHANGE_GROUP_PROP_FAIL) {
            notifyUpdateGroup(notify.cookie, false, REASON_OTHER, groupNotify.changeGroupPropFail.groupId);
        }
    }

    private void notifyCreateGroup(final int operationId, final boolean result,
                                   @JCGroup.Reason final int reason, final JCGroupItem groupItem) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("创建群 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onCreateGroup(operationId, result, reason, groupItem);
                }
            }
        });
    }

    private void notifyFetchGroups(final int operationId, final boolean result,
                                   @JCGroup.Reason final int reason,
                                   final List<JCGroupItem> groups, final long updateTime,
                                   final boolean fullUpdated) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("拉取群列表 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onFetchGroups(operationId, result, reason, groups, updateTime,
                            fullUpdated);
                }
            }
        });
    }

    private void notifyFetchGroupInfo(final int operationId, final boolean result,
                                      @JCGroup.Reason final int reason, final JCGroupItem groupItem,
                                      final List<JCGroupMember> members,
                                      final long updateTime, final boolean fullUpdated) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("拉取群详情 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onFetchGroupInfo(operationId, result, reason, groupItem, members,
                            updateTime, fullUpdated);
                }
            }
        });
    }

    private void notifyLeave(final int operationId, final boolean result,
                             @JCGroup.Reason final int reason) {
        final String groupId = mMapCookieAndGroupId.get(operationId);
        if (!TextUtils.isEmpty(groupId)) {
            mMapCookieAndGroupId.remove(operationId);
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    logInfo("离开 操作号:%d 结果:%b 原因:%d 群id:%s", operationId, result, reason, groupId);
                    for (JCGroupCallback callback : mCallbacks) {
                        callback.onLeave(operationId, result, reason, groupId);
                    }
                }
            });
        }
    }

    private void notifyDissolve(final int operationId, final boolean result,
                                @JCGroup.Reason final int reason, final String groupId) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("解散 操作号:%d 结果:%b 原因:%d 群id:%s", operationId, result, reason, groupId);
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onDissolve(operationId, result, reason, groupId);
                }
            }
        });
    }

    private void notifyGroupListChange() {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("群列表有更新");
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onGroupListChange();
                }
            }
        });
    }

    private void notifyGroupInfoChange(final String groupId) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("群详情有更新");
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onGroupInfoChange(groupId);
                }
            }
        });
    }

    private void notifyDealMembers(final int operationId, final boolean result,
                                   @JCGroup.Reason final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("群成员处理 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onDealMembers(operationId, result, reason);
                }
            }
        });
    }

    private void notifyUpdateGroup(final int operationId, final boolean result,
                                   @JCGroup.Reason final int reason, final String groupId) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                logInfo("更新群 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCGroupCallback callback : mCallbacks) {
                    callback.onUpdateGroup(operationId, result, reason, groupId);
                }
            }
        });
    }

    private String getUserIdFromTag(String tag) {
        // 成员删除tag会传进来null
        if (!TextUtils.isEmpty(tag)) {
            try {
                JSONObject jsonObject = new JSONObject(tag);
                return jsonObject.optString(USER_ID_KEY);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    @ChangeState
    private int translateFromNotifyChangeType(@JCNotify.GroupChangeType int changeType) {
        switch (changeType) {
            case JCNotify.GROUP_CHANGE_TYPE_ADD:
                return GROUP_CHANGE_STATE_ADD;
            case JCNotify.GROUP_CHANGE_TYPE_UPDATE:
                return GROUP_CHANGE_STATE_UPDATE;
            case JCNotify.GROUP_CHANGE_TYPE_REMOVE:
            default:
                return GROUP_CHANGE_STATE_REMOVE;
        }
    }

    private int translateFromMtcMemberType(int memberType) {
        switch (memberType) {
            case EN_MTC_GROUP_RELATION_OWNER:
                return GROUP_MEMBER_TYPE_OWNER;
            case EN_MTC_GROUP_RELATION_MANAGER:
                return GROUP_MEMBER_TYPE_MANAGER;
            default:
                return GROUP_MEMBER_TYPE_MEMBER;
        }
    }

    private int translateToMtcMemberType(@JCGroup.MemberType int memberType) {
        switch (memberType) {
            case GROUP_MEMBER_TYPE_OWNER:
                return EN_MTC_GROUP_RELATION_OWNER;
            case GROUP_MEMBER_TYPE_MANAGER:
                return EN_MTC_GROUP_RELATION_MANAGER;
            case JCGroup.GROUP_MEMBER_TYPE_MEMBER:
            default:
                return MtcGroup.EN_MTC_GROUP_RELATION_MEMBER;
        }
    }

    private String genTag(String userId) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put(USER_ID_KEY, userId);
            return jsonObject.toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return null;
    }

    private int translateToDealMemberType(@JCGroup.ChangeState int changeState) {
        switch (changeState) {
            case GROUP_CHANGE_STATE_ADD:
                return JCParam.GroupDealMember.DEAL_MEMBER_ADD;
            case JCGroup.GROUP_CHANGE_STATE_UPDATE:
                return JCParam.GroupDealMember.DEAL_MEMBER_UPDATE;
            case JCGroup.GROUP_CHANGE_STATE_REMOVE:
            case JCGroup.GROUP_CHANGE_STATE_NONE:
            default:
                return JCParam.GroupDealMember.DEAL_MEMBER_REMOVE;
        }
    }

    private void logInfo(String msg, Object... args) {
        JCLog.info(TAG, msg, args);
    }

    private void logError(String msg, Object... args) {
        JCLog.error(TAG, msg, args);
    }
}
