package com.juphoon.cloud;

import android.support.annotation.IntDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;

/**
 * 1对1通话模块
 *
 * @author juphoon
 */
public abstract class JCCall {

    static final String TAG = JCCall.class.getSimpleName();

    /**
     * 通话方向
     */
    @IntDef({DIRECTION_IN, DIRECTION_OUT})
    @Retention(RetentionPolicy.SOURCE)
    public @interface CallDirection {
    }

    /**
     * 呼入
     */
    public static final int DIRECTION_IN = 0;
    /**
     * 呼出
     */
    public static final int DIRECTION_OUT = 1;

    /**
     * 通话结束原因
     */
    @IntDef({REASON_NONE, REASON_NOT_LOGIN, REASON_CALL_FUNCTION_ERROR, REASON_TIMEOUT, REASON_NETWORK, REASON_OVER_LIMIT, REASON_TERM_BY_SELF,
            REASON_ANSWER_FAIL, REASON_BUSY, REASON_DECLINE, REASON_USER_OFFLINE, REASON_NOT_FOUND, REASON_OTHER})
    @Retention(RetentionPolicy.SOURCE)
    public @interface CallReason {
    }

    /**
     * 无异常
     */
    public static final int REASON_NONE = 0;
    /**
     * 未登录
     */
    public static final int REASON_NOT_LOGIN = 1;
    /**
     * 函数调用失败
     */
    public static final int REASON_CALL_FUNCTION_ERROR = 2;
    /**
     * 超时
     */
    public static final int REASON_TIMEOUT = 3;
    /**
     * 网络异常
     */
    public static final int REASON_NETWORK = 4;
    /**
     * 超过通话数限制
     */
    public static final int REASON_OVER_LIMIT = 5;
    /**
     * 自己挂断
     */
    public static final int REASON_TERM_BY_SELF = 6;
    /**
     * 应答失败
     */
    public static final int REASON_ANSWER_FAIL = 7;
    /**
     * 忙
     */
    public static final int REASON_BUSY = 8;
    /**
     * 拒接
     */
    public static final int REASON_DECLINE = 9;
    /**
     * 用户不在线
     */
    public static final int REASON_USER_OFFLINE = 10;
    /**
     * 无此用户
     */
    public static final int REASON_NOT_FOUND = 11;
    /**
     * 已有通话拒绝视频来电
     */
    public static final int REASON_REJECT_VIDEO_WHEN_HAS_CALL = 12;
    /**
     * 已有视频通话拒绝来电
     */
    public static final int REASON_REJECT_WHEN_HAS_VIDEO_CALL = 13;
    /**
     * 其他错误
     */
    public static final int REASON_OTHER = 100;

    /**
     * 通话状态
     */
    @IntDef({STATE_INIT, STATE_PENDING, STATE_CONNECTING, STATE_TALKING, STATE_OK, STATE_CANCEL, STATE_CANCELED,
            STATE_MISSED, STATE_ERROR})
    @Retention(RetentionPolicy.SOURCE)
    public @interface CallState {
    }

    /**
     * 初始
     */
    public static final int STATE_INIT = 0;
    /**
     * 振铃
     */
    public static final int STATE_PENDING = 1;
    /**
     * 连接中
     */
    public static final int STATE_CONNECTING = 2;
    /**
     * 通话中
     */
    public static final int STATE_TALKING = 3;
    /**
     * 通话正常结束
     */
    public static final int STATE_OK = 4;
    /**
     * 未接通挂断
     */
    public static final int STATE_CANCEL = 5;
    /**
     * 未接通被挂断
     */
    public static final int STATE_CANCELED = 6;
    /**
     * 未接
     */
    public static final int STATE_MISSED = 7;
    /**
     * 异常
     */
    public static final int STATE_ERROR = 8;

    /**
     * 通话网络状态
     */
    @IntDef({NET_STATUS_DISCONNECTED, NET_STATUS_VERY_BAD, NET_STATUS_BAD, NET_STATUS_NORMAL, NET_STATUS_GOOD, NET_STATUS_VERY_GOOD})
    @Retention(RetentionPolicy.SOURCE)
    public @interface NetStatus {
    }

    /**
     * 无网络
     */
    public static final int NET_STATUS_DISCONNECTED = -3;
    /**
     * 很差
     */
    public static final int NET_STATUS_VERY_BAD = -2;
    /**
     * 差
     */
    public static final int NET_STATUS_BAD = -1;
    /**
     * 一般
     */
    public static final int NET_STATUS_NORMAL = 0;
    /**
     * 好
     */
    public static final int NET_STATUS_GOOD = 1;
    /**
     * 很好
     */
    public static final int NET_STATUS_VERY_GOOD = 2;

    private static JCCall sCall;

    /**
     * 最大通话数，当通话超过最大数呼出会失败，收到来电会自动拒绝
     */
    public int maxCallNum = 5;

    /**
     * 登录成功后是否拉取未接电话
     */
    public boolean fetchMissedCallWhenLogin = false;

    /**
     * 创建JCCall实例
     *
     * @param client        JCClient实例
     * @param mediaDevice   JCMediaDevice实例
     * @param callback      回调接口，用于接收 JCCall 相关回调事件
     * @return JCCall       JCCall实例
     */
    public static JCCall create(JCClient client, JCMediaDevice mediaDevice, JCCallCallback callback) {
        if (sCall != null) {
            return sCall;
        }
        sCall = new JCCallImpl(client, mediaDevice, callback);
        return sCall;
    }

    /**
     * 销毁 JCCall 实例
     */
    public static void destroy() {
        if (sCall != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sCall.destroyObj();
                    sCall = null;
                }
            });
        }
    }

    /**
     * 销毁实例
     */
    protected abstract void destroyObj();

    /**
     * 一对一呼叫
     *
     * @param userId        用户标识
     * @param video         是否视频呼叫
     * @param extraParam    透传参数，设置后被叫方可获取该参数
     * @return              返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean call(String userId, boolean video, String extraParam);

    /**
     * 挂断
     *
     * @param item          JCCallItem 对象
     * @param reason        挂断原因
     * @param description   挂断描述
     * @return              返回 true 表示正常执行调用流程，false 表示调用异常
     * @see CallReason
     */
    public abstract boolean term(JCCallItem item, @CallReason int reason, String description);

    /**
     * 接听
     *
     * @param item  JCCallItem 对象
     * @param video 针对视频呼入可以选择以视频接听还是音频接听
     * @return      返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean answer(JCCallItem item, boolean video);

    /**
     * 静音，通过 JCCallItem 对象中的静音状态来决定开启关闭静音
     *
     * @param   item JCCallItem 对象
     * @return  返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean mute(JCCallItem item);

    /**
     * 呼叫保持，通过 JCCallItem 对象中的呼叫保持状态来决定开启关闭呼叫保持
     *
     * @param item  JCCallItem 对象
     * @return      返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean hold(JCCallItem item);

    /**
     * 语音通话录音，通过 JCCallItem 对象中的audioRecord状态来决定开启关闭录音
     *
     * @param item      JCCallItem 对象
     * @param enable    开启关闭录音
     * @param filePath  录音文件路径
     * @return          返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean audioRecord(JCCallItem item, boolean enable, String filePath);

    /**
     * 视频通话录制，通过 JCCallItem 对象中的localVideoRecord, remoteVideoRecord状态来决定开启关闭录制
     *
     * @param item      JCCallItem 对象
     * @param enable    开启关闭录制
     * @param remote    是否为远端视频源
     * @param width     录制视频宽像素
     * @param height    录制视频高像素
     * @param filePath  录制视频文件存储路径
     * @return          返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean videoRecord(JCCallItem item, boolean enable, boolean remote, int width, int height, String filePath);

    /**
     * 切换活跃通话
     *
     * @param item  需要变为活跃状态的 JCCallItem 对象
     * @return      返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean becomeActive(JCCallItem item);

    /**
     * 开启关闭视频流发送，用于视频通话中
     *
     * @param item  JCCallItem 对象
     * @return      返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean enableUploadVideoStream(JCCallItem item);

    /**
     * 设置会议模式，只针对语音
     *
     * @param conference 是否为会议模式
     */
    public abstract void setConference(boolean conference);

    /**
     * 获得音频会议模式
     *
     * @return 是否为音频会议模式
     */
    public abstract boolean getConference();

    /**
     * 获得通话对象列表
     *
     * @return 通话对象列表
     */
    public abstract List<JCCallItem> getCallItems();

    /**
     * 通过通话建立的通道发送数据
     *
     * @param item      需要发送数据的 JCCallItem 对象
     * @param type      文本消息类型，用户可以自定义，例如text，xml等
     * @param content   文本内容
     * @return          返回 true 表示正常执行调用流程，false 表示调用异常
     */
    public abstract boolean sendMessage(JCCallItem item, String type, String content);

    /**
     * 获得当前通话统计信息，以Json字符串形式返回，其中包含 "Audio" 和 "Video" 两个节点
     *
     * @return 当前通话统计信息
     */
    public abstract String getStatistics();

    /**
     * 添加回调
     *
     * @param callback JCCallCallback 接口对象
     */
    protected abstract void addCallback(JCCallCallback callback);

    /**
     * 删除回调
     *
     * @param callback JCCallCallback 接口对象
     */
    protected abstract void removeCallback(JCCallCallback callback);
}
