package com.juphoon.cloud;

import android.support.annotation.IntDef;
import android.text.TextUtils;

import org.json.JSONObject;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import static com.juphoon.cloud.JCPushTemplate.GCM;
import static com.juphoon.cloud.JCPushTemplate.GCM;
import static com.juphoon.cloud.JCPushTemplate.GCM;

/**
 * 推送模版类
 *
 * @author juphoon
 */
public class JCPushTemplate {

    private static final String TAG = JCPushTemplate.class.getSimpleName();

    private String mData;

    /**
     * 推送厂商
     */
    @IntDef({XIAOMI, HUAWEI, GCM})
    @Retention(RetentionPolicy.SOURCE)
    public @interface PushManufacturer {
    }

    /**
     * 小米推送
     */
    public static final int XIAOMI = 0;
    /**
     * 华为推送
     */
    public static final int HUAWEI = 1;

    /**
     * GCM推送
     */
    public static final int GCM = 2;

    /**
     * 生成小米推送参数
     *
     * @param appPackage Justalk cloud 上注册的该应用的小米Push AppPackage 值
     * @param regId 小米推送注册成功返回的 regId
     */
    public void initWithMiPush(String appPackage, String regId) {
        JSONObject json = new JSONObject();
        try {
            json.put("Notify.MiPush.AppId", appPackage);
            json.put("Notify.MiPush.RegId", regId);
        } catch (Exception e) {
            e.printStackTrace();
            JCLog.error(TAG, "生成小米推送失败");
            return;
        }
        mData = json.toString();
    }

    /**
     * 生成华为推送参数
     *
     * @param appId Justalk cloud 上注册的该应用的华为Push 的 AppId 值
     * @param token 华为推送注册成功返回的 regId
     */
    public void initWithHmsPush(String appId, String token) {
        JSONObject json = new JSONObject();
        try {
            json.put("Notify.HMSPush.AppId", appId);
            json.put("Notify.HMSPush.Token", token);
        } catch (Exception e) {
            e.printStackTrace();
            JCLog.error(TAG, "生成华为推送失败");
            return;
        }
        mData = json.toString();
    }

    /* 生成Google推送参数
     *
     * @param senderId Justalk cloud 上注册的该应用的GCM的senderId值
     * @param token GCM注册成功返回的regid
     */
    public void initWithGCM(String senderId, String token) {
        JSONObject json = new JSONObject();
        try {
            json.put("Notify.GCM.SenderId", senderId);
            json.put("Notify.GCM.RegId", token);
        } catch (Exception e) {
            e.printStackTrace();
            JCLog.error(TAG, "生成GCM推送失败");
            return;
        }
        mData = json.toString();
    }

    /**
     * 生成通话推送参数
     *
     * @param pushFactory 推送厂商
     * @param selfId 自身唯一标识符
     * @param description 描述信息
     * @param passThrough 小米推送相关参数，"1"表示透传消息，"0"表示非透传消息，可以参看小米推送文档
     */
    public void initWithCall(@PushManufacturer int pushFactory, String selfId, String description, String passThrough) {
        JSONObject json = new JSONObject();
        try {
            if (pushFactory == XIAOMI) {
                json.put("Notify.MiPush.Invite.Payload",
                        "{\"calltype\":\"${MediaType}\",\"caller\":\"${Caller}\",\"callid\":\"${CallId}\",\"to\":\"" + selfId  + "\",\"resend\":\"${ResendIndex}\"" + "}");
                json.put("Notify.MiPush.Invite.Expiration", String.valueOf(2L*7*24*60*60));
                json.put("Notify.MiPush.Invite.Title", "${Caller}");
                json.put("Notify.MiPush.Invite.Description", description);
                json.put("Notify.MiPush.Invite.PassThrough", passThrough);
                json.put("Notify.MiPush.Invite.ResendCount", "5");
                json.put("Notify.MiPush.Invite.ResendTimeout", "10");
            } else if (pushFactory == HUAWEI) {
                json.put("Notify.HMSPush.Invite.Payload",
                        "{\"calltype\":\"${MediaType}\",\"caller\":\"${Caller}\",\"callid\":\"${CallId}\",\"to\":\"" + selfId + "\",\"resend\":\"${ResendIndex}\"" + "}");
                json.put("Notify.HMSPush.Invite.Expiration", String.valueOf(2L * 7 * 24 * 60 * 60));
                json.put("Notify.HMSPush.Invite.AndroidMessage",
                        "{\"notification_title\":\"${Caller}\",\"notification_content\":\"" + description + "\",\"doings\":\"1\"}");
                json.put("Notify.HMSPush.Invite.PassThrough", "1");
                json.put("Notify.HMSPush.Invite.Priority", "0");
                json.put("Notify.HMSPush.Invite.CacheMode", "1");
                json.put("Notify.HMSPush.Invite.MsgType", "-1");
                json.put("Notify.HMSPush.Invite.UserType", "0");
                json.put("Notify.HMSPush.Invite.ResendCount", "5");
                json.put("Notify.HMSPush.Invite.ResendTimeout", "10");
            } else if (pushFactory == GCM) {
                json.put("Notify.GCM.Invite.Payload",
                        "{\"calltype\":\"${MediaType}\",\"caller\":\"${Caller}\",\"callid\":\"${CallId}\"," + "\"to\":\"" + selfId + "\"" + ",\"resend\":\"${ResendIndex}\"}");
                json.put("Notify.GCM.Invite.Expiration", String.valueOf(2L * 7 * 24 * 60 * 60));
                json.put("Notify.GCM.Invite.ResendCount", "10");
                json.put("Notify.GCM.Invite.ResendTimeout", "10");
            }
        } catch (Exception e) {
            e.printStackTrace();
            JCLog.error(TAG, "生成呼叫参数失败");
            return;
        }
        mData = json.toString();
    }

    /**
     * 生成消息类推送参数
     *
     * @param pushFactory 推送厂商
     * @param selfId 自身唯一标识符
     * @param type 消息类型
     * @param description 提示内容，针对文本消息如果要提示发送内容，则填 null，类似文件消息则要填相应的信息，例如 "发送了一段语音"
     * @param passThrough 小米推送相关参数，"1"表示透传消息，"0"表示非透传消息，可以参看小米推送文档
     */
    public void initWithText(@PushManufacturer int pushFactory, String selfId, String type, String description, String passThrough) {
        JSONObject json = new JSONObject();
        try {
            if (pushFactory == XIAOMI) {
                json.put("Notify.MiPush.Message.Info." + type + ".Title", "${Sender}");
                json.put("Notify.MiPush.Message.Info." + type + ".Description",
                        TextUtils.isEmpty(description) ? "${Text}" : description);
                json.put("Notify.MiPush.Message.Info." + type + ".Payload",
                        "{\"sender\": \"${Sender}\",\"text\":" + (TextUtils.isEmpty(description) ? "\"${Text}\"" : "\"" + description + "\"") + ",\"to\":" + selfId + "}");
                json.put("Notify.MiPush.Message.Info." + type + ".Expiration", String.valueOf(2L * 7 * 24 * 60 * 60));
                json.put("Notify.MiPush.Message.Info." + type + ".PassThrough", passThrough);
            } else if (pushFactory == HUAWEI) {
                json.put("Notify.HMSPush.Message.Info." + type + ".Payload",
                        "{\"sender\":\"${Sender}\",\"text\":" + (TextUtils.isEmpty(description) ? "\"${Text}\"" : "\"" + description + "\"") + ",\"to\":\"" + selfId + "\"}");
                json.put("Notify.HMSPush.Message.Info." + type + ".Expiration", String.valueOf(2L * 7 * 24 * 60 * 60));
                json.put("Notify.HMSPush.Message.Info." + type + ".AndroidMessage",
                        "{\"notification_title\":\"${Sender}\",\"notification_content\":" + (TextUtils.isEmpty(description) ? "\"${Text}\"" : description)
                                + ", \"doings\":\"1\"}");
                json.put("Notify.HMSPush.Message.Info." + type + ".PassThrough", "1");
            } else if (pushFactory == GCM) {
                json.put("Notify.GCM.Message.Info." + type + ".Payload", "{\"sender\":\"${Sender}\",\"text\":\"${Text}\"," + "\"to\":\"" + selfId + "\"" + ",\"resend\":\"${ResendIndex}\"}");
                json.put("Notify.GCM.Message.Info." + type + ".Expiration", String.valueOf(2L * 7 * 24 * 60 * 60));
                json.put("Notify.GCM.Message.Info." + type + ".ResendCount", "5");
                json.put("Notify.GCM.Message.Info." + type + ".ResendTimeout", "20");
                json.put("Notify.GCM.Message.Info." + type + ".PassThrough", "1");
            }
        } catch (Exception e) {
            e.printStackTrace();
            JCLog.error(TAG, "生成呼叫参数失败");
            return;
        }
        mData = json.toString();
    }

    /**
     * 返回推送参数
     *
     * @return 推送参数
     */
    public String getData() {
        return mData;
    }

}
