package com.juphoon.cloud;

import android.content.Context;
import android.support.annotation.IntDef;
import android.support.annotation.StringDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Map;

/**
 * 登陆模块
 *
 * @author juphoon
 */
public abstract class JCClient {

    static final String TAG = JCClient.class.getSimpleName();

    /**
     * 状态
     */
    @IntDef({STATE_NOT_INIT, STATE_IDLE, STATE_LOGINING, STATE_LOGINED, STATE_LOGOUTING,})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ClientState {
    }

    /**
     * 未初始化
     */
    public static final int STATE_NOT_INIT = 0;
    /**
     * 未登陆
     */
    public static final int STATE_IDLE = 1;
    /**
     * 登陆中
     */
    public static final int STATE_LOGINING = 2;
    /**
     * 登陆成功
     */
    public static final int STATE_LOGINED = 3;
    /**
     * 登出中
     */
    public static final int STATE_LOGOUTING = 4;

    /**
     * 原因
     */
    @IntDef({REASON_NONE, REASON_SDK_NOT_INIT, REASON_INVALID_PARAM, REASON_CALL_FUNCTION_ERROR, REASON_STATE_CANNOT_LOGIN,
            REASON_TIMEOUT, REASON_NETWORK, REASON_APPKEY, REASON_AUTH, REASON_NOUSER, REASON_SERVER_LOGOUT, REASON_OTHER})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ClientReason {
    }

    /**
     * 正常
     */
    public static final int REASON_NONE = 0;
    /**
     * sdk 未初始化
     */
    public static final int REASON_SDK_NOT_INIT = 1;
    /**
     * 无效参数
     */
    public static final int REASON_INVALID_PARAM = 2;
    /**
     * 函数调用失败
     */
    public static final int REASON_CALL_FUNCTION_ERROR = 3;
    /**
     * 当前状态无法再次登录
     */
    public static final int REASON_STATE_CANNOT_LOGIN = 4;
    /**
     * 超时
     */
    public static final int REASON_TIMEOUT = 5;
    /**
     * 网络异常
     */
    public static final int REASON_NETWORK = 6;
    /**
     * appkey 错误
     */
    public static final int REASON_APPKEY = 7;
    /**
     * 账号密码错误
     */
    public static final int REASON_AUTH = 8;
    /**
     * 无该用户
     */
    public static final int REASON_NOUSER = 9;
    /**
     * 强制登出
     */
    public static final int REASON_SERVER_LOGOUT = 10;
    /**
     * 其他错误
     */
    public static final int REASON_OTHER = 100;

    /**
     * 配置关键字
     */
    @StringDef({CONFIG_KEY_SERVER_ADDRESS})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ConfigKey {
    }

    /**
     * 服务器地址
     */
    public static final String CONFIG_KEY_SERVER_ADDRESS = "config_key_server_address";
    /**
     * 设备id
     */
    public static final String CONFIG_KEY_DEVICE_ID = "config_key_device_id";

    /**
     * 创建参数关键字
     */
    @StringDef({CREATE_EXTRA_SDK_INFO_DIR, CREATE_EXTRA_SDK_LOG_DIR})
    @Retention(RetentionPolicy.SOURCE)
    public @interface CreateExtra {
    }

    /**
     * SDK 信息存放目录，包括账号信息，日志信息等, 如果设置 CREATE_EXTRA_SDK_LOG_DIR 则日志不会存放在此目录
     */
    public static final String CREATE_EXTRA_SDK_INFO_DIR = "create_extra_sdk_info_dir";
    /**
     * 日志目录设置
     */
    public static final String CREATE_EXTRA_SDK_LOG_DIR = "create_extra_sdk_log_dir";

    private static JCClient sClient;

    /**
     * 创建 JCClient 实例
     *
     * @param appKey      用户从 Juphoon Cloud 平台上申请的 AppKey 字符串
     * @param callback    回调接口，用于接收 JCClient 相关通知
     * @param extraParams 额外参数，没有则填null
     * @return JCClient 对象
     * @see CreateExtra
     */
    public static JCClient create(Context context, String appKey, JCClientCallback callback,
                                  Map<String, String> extraParams) {
        if (sClient != null) {
            return sClient;
        }
        sClient = new JCClientImpl(context, appKey, callback, extraParams);
        return sClient;
    }

    /**
     * 销毁 JCClient 对象
     */
    public static void destroy() {
        if (sClient != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sClient.destroyObj();
                    sClient = null;
                }
            });
        }
    }

    /**
     * 昵称，用于通话，消息等，可以更直观的表明身份
     */
    public String displayName;

    /**
     * 销毁对象
     */
    protected abstract void destroyObj();

    /**
     * 设置是否为前台, 在有控制后台网络的手机上当进入前台时主动触发
     *
     * @param foreground 是否为前台
     */
    public abstract void setForeground(boolean foreground);

    /**
     * 获得用户标识
     *
     * @return 用户标识
     */
    public abstract String getUserId();

    /**
     * 当前状态
     *
     * @return 当前状态
     * @see JCClient.ClientState
     */
    public abstract int getState();

    /**
     * 登陆 Juphoon Cloud 平台，只有登陆成功后才能进行平台上的各种业务<br>
     * 登陆结果通过 JCCallCallback 通知<br>
     * 注意:用户名为英文数字和'+' '-' '_' '.'，长度不要超过64字符<br>
     *
     * @param userId   用户名
     * @param password 密码，但不能为空
     * @return 返回 true 表示正常执行调用流程，false 表示调用异常，异常错误通过 JCClientCallback 通知
     */
    public abstract boolean login(String userId, String password);

    /**
     * 登出 Juphoon Cloud 平台，登出后不能进行平台上的各种业务
     *
     * @return 返回 true 表示正常执行调用流程，false 表示调用异常，异常错误通过 JCClientCallback 通知
     */
    public abstract boolean logout();

    /**
     * 设置配置相关参数
     *
     * @param key   参数关键字
     * @param value 参数值
     * @return 返回 true 表示设置成功，false 表示设置失败
     * @see JCClient.ConfigKey
     */
    public abstract boolean setConfig(@ConfigKey String key, String value);

    /**
     * 获取配置相关参数
     *
     * @param key 参数关键字
     * @return 成功返回字符串类型具体值, 失败返回 NULL
     * @see JCClient.ConfigKey
     */
    public abstract String getConfig(@ConfigKey String key);

    /**
     * 获得上下文
     *
     * @return 返回 Context 对象
     */
    protected abstract Context getContext();

    /**
     * 添加回调
     *
     * @param callback JCClientCallback 接口对象
     */
    protected abstract void addCallback(JCClientCallback callback);

    /**
     * 删除回调
     *
     * @param callback JCClientCallback 接口对象
     */
    protected abstract void removeCallback(JCClientCallback callback);

}
