package com.juphoon.cloud;

import android.support.annotation.IntDef;
import android.support.annotation.StringDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Map;

/**
 * 消息通道模块
 *
 * @author juphoon
 */
public abstract class JCMessageChannel {

    /**
     * 消息归属类型
     */
    @IntDef({TYPE_1TO1, TYPE_GROUP})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Type {
    }

    /**
     * 一对一消息
     */
    public static final int TYPE_1TO1 = 0;
    /**
     * 群组消息
     */
    public static final int TYPE_GROUP = 1;

    /**
     * 消息错误枚举
     */
    @IntDef({REASON_NONE, REASON_NOT_LOGIN, REASON_TOO_LONG, REASON_OTHER})
    @Retention(RetentionPolicy.SOURCE)
    public @interface MessageChannelReason {
    }

    /**
     * 无异常
     */
    public static final int REASON_NONE = 0;
    /**
     * 未登陆
     */
    public static final int REASON_NOT_LOGIN = 1;
    /**
     * 消息内容太长
     */
    public static final int REASON_TOO_LONG = 2;
    /**
     * 其他错误
     */
    public static final int REASON_OTHER = 100;

    /**
     * 消息状态
     */
    @IntDef({ITEM_STATE_INIT, ITEM_STATE_SENDING, ITEM_STATE_SENDOK, ITEM_STATE_SENDFAIL, ITEM_STATE_RECEIVED})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ItemState {
    }

    /**
     * 消息初始状态
     */
    public static final int ITEM_STATE_INIT = 0;
    /**
     * 消息发送中状态
     */
    public static final int ITEM_STATE_SENDING = 1;
    /**
     * 消息发送成功状态
     */
    public static final int ITEM_STATE_SENDOK = 2;
    /**
     * 消息发送失败状态
     */
    public static final int ITEM_STATE_SENDFAIL = 3;
    /**
     * 收到消息
     */
    public static final int ITEM_STATE_RECEIVED = 4;

    /**
     * 消息传输方向
     */
    @IntDef({DIRECTION_SEND, DIRECTION_RECEIVE})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ItemDirection {
    }

    /**
     * 发送
     */
    public static final int DIRECTION_SEND = 0;
    /**
     * 接收
     */
    public static final int DIRECTION_RECEIVE = 1;

    /**
     * 消息关键字
     */
    @StringDef({ITEM_MESSAGE_ID, ITEM_FILE_URI, ITEM_THUMB_DATA, ITEM_FILE_SIZE, ITEM_DURATION, ITEM_EXTRA_DATA})
    @Retention(RetentionPolicy.SOURCE)
    @interface ItemData {
    }

    static final String ITEM_MESSAGE_ID = "MessageId";
    static final String ITEM_FILE_URI = "FileUri";
    static final String ITEM_THUMB_DATA = "ThumbData";
    static final String ITEM_FILE_SIZE = "FileSize";
    static final String ITEM_DURATION = "Duration";
    static final String ITEM_EXTRA_DATA = "ExtraData";

    @IntDef({ITEM_MAX_THUMB_SIZE, ITEM_MAX_TEXT_SIZE})
    @Retention(RetentionPolicy.SOURCE)
    @interface ItemLimit {
    }

    /**
     * 缩略图限制大小，单位字节
     */
    static final int ITEM_MAX_THUMB_SIZE = 6 * 1024;
    static final int ITEM_MAX_TEXT_SIZE = 10 * 1024;

    private static JCMessageChannel sMessageChannel;

    /**
     * 缩率图保存路径，初始化会默认设置，同时用户可以自己设置
     */
    public String thumbDir;

    /**
     * 创建 JCMessageChannel 对象
     *
     * @param client   JCClient 对象
     * @param callback JCMessageChannelCallback 回调接口，用于接收 JCMessageChannel 相关通知
     * @return 返回 JCMessageChannel 对象
     */
    public static JCMessageChannel create(JCClient client, JCMessageChannelCallback callback) {
        if (sMessageChannel != null) {
            return sMessageChannel;
        }
        sMessageChannel = new JCMessageChannelImpl(client, callback);
        return sMessageChannel;
    }

    /**
     * 销毁 JCMessageChannel 对象
     */
    public static void destroy() {
        if (sMessageChannel != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sMessageChannel.destroyObj();
                    sMessageChannel = null;
                }
            });
        }
    }

    /**
     * 销毁对象
     */
    protected abstract void destroyObj();

    /**
     * 发送文本消息, 文本内容不要超过10KB
     *
     * @param type        类型，参见 JCMessageChannelType
     * @param keyId       对方唯一标识，当 type 为 JCMessageChannelType1To1 时为用户标识，当 type 为 JCMessageChannelTypeGroup 时为群组标识
     * @param messageType 文本消息类型，用户可以自定义，例如text，xml等
     * @param text        文本内容
     * @param extraParams 自定义参数集
     * @return 返回 JCMessageChannelItem 对象，异常返回 null
     */
    public abstract JCMessageChannelItem sendMessage(@Type int type, String keyId, String messageType, String text,
                                                     Map<String, String> extraParams);

    /**
     * 发送文件消息
     *
     * @param type        类型，参见 JCMessageChannelType
     * @param keyId       对方唯一标识，当 type 为 JCMessageChannelType1To1 时为用户标识，当 type 为 JCMessageChannelTypeGroup 时为群组标识
     * @param messageType 文件消息类型，用户可以自定义，例如image，video等
     * @param fileUri     文件链接地址
     * @param thumbPath   缩略图路径，针对视频，图片等消息
     * @param size        文件大小(字节)
     * @param duration    文件时长，针对语音，视频等消息
     * @param extraParams 自定义参数集
     * @return 返回 JCMessageChannelItem 对象，异常返回 nil
     */
    public abstract JCMessageChannelItem sendFile(@Type int type, String keyId, String messageType, String fileUri, String thumbPath,
                                                  int size, int duration, Map<String, String> extraParams);


    /**
     * 添加回调
     *
     * @param callback JCMessageChannelCallback 接口对象
     */
    protected abstract void addCallback(JCMessageChannelCallback callback);

    /**
     * 删除回调
     *
     * @param callback JCMessageChannelCallback 接口对象
     */
    protected abstract void removeCallback(JCMessageChannelCallback callback);

}
