package com.juphoon.cloud;

import android.content.Context;
import android.text.TextUtils;

import com.justalk.cloud.lemon.MtcCli;
import com.justalk.cloud.lemon.MtcCliConstants;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

class JCClientImpl extends JCClient implements MtcCliConstants, MtcEngine.MtcNotifyListener,
        JCNetCallback {

    Context mContext;

    private List<JCClientCallback> mCallbacks = new ArrayList<>();
    private String mUserId;
    private String mServer;
    private String mAppKey;
    private String mDeviceId;
    private String mUid;
    @ClientState
    private int mState;

    JCClientImpl(Context context, String appKey, JCClientCallback callback, Map<String, String> extraParams) {
        JCLog.info(TAG, "创建");
        if (context == null) {
            throw new RuntimeException("JCClient Context cannot be null!");
        }
        if (TextUtils.isEmpty(appKey)) {
            throw new RuntimeException("JCClient app key cannot be empty!");
        }
        if (callback == null) {
            throw new RuntimeException("JCClient callback cannot be null!");
        }
        mContext = context.getApplicationContext();
        mState = STATE_NOT_INIT;

        JCParam.Init param = new JCParam.Init();
        param.context = mContext;
        if (extraParams != null) {
            if (extraParams.containsKey(JCClient.CREATE_EXTRA_SDK_INFO_DIR)) {
                param.sdkInfoDir = extraParams.get(JCClient.CREATE_EXTRA_SDK_INFO_DIR);
            }
            if (extraParams.containsKey(JCClient.CREATE_EXTRA_SDK_LOG_DIR)) {
                param.sdkLogDir = extraParams.get(JCClient.CREATE_EXTRA_SDK_LOG_DIR);
            }
        }

        if (MtcEngine.getInstance().initialize(param).succ) {
            JCLog.info(TAG, "SDK初始化成功");
            mServer = "http:cn.router.justalkcloud.com:8080";
            mState = STATE_IDLE;
            mCallbacks.add(callback);
            mAppKey = appKey;
            MtcEngine.getInstance().addMtcNotifyListener(this);
            JCNet.getInstance().initialize(mContext);
            JCNet.getInstance().addCallback(this);
        } else {
            JCLog.error(TAG, "SDK初始化失败");
        }
    }

    @Override
    protected void destroyObj() {
        mCallbacks.clear();
        MtcEngine.getInstance().removeMtcNotifyListener(this);
        JCNet.getInstance().removeCallback(this);
        JCNet.getInstance().uninitialize();
        MtcEngine.getInstance().uninitialize();
    }

    @Override
    public void setForeground(final boolean foreground) {
        JCClientThreadImpl.getInstance().postDelayed(new Runnable() {
            @Override
            public void run() {
                JCNet.getInstance().updateNetType();
                MtcEngine.getInstance().setForeground(foreground);
            }
        }, 1000);
    }

    @Override
    public String getUserId() {
        return mUserId;
    }

    @Override
    public boolean setConfig(@ConfigKey String key, String value) {
        boolean ret = true;
        if (key.equals(CONFIG_KEY_SERVER_ADDRESS)) {
            mServer = value;
        } else if (key.equals(CONFIG_KEY_DEVICE_ID)) {
            mDeviceId = value;
        } else {
            ret = false;
        }
        return ret;
    }

    @Override
    public String getConfig(@ConfigKey String key) {
        if (key.equals(CONFIG_KEY_SERVER_ADDRESS)) {
            return mServer;
        } else if (key.equals(CONFIG_KEY_DEVICE_ID)) {
            return TextUtils.isEmpty(mDeviceId) ? MtcEngine.getInstance().getDeviceId() : mDeviceId;
        }
        return "";
    }

    @Override
    public int getState() {
        return mState;
    }

    @Override
    public boolean login(String userId, String password) {
        if (mState == STATE_NOT_INIT) {
            JCLog.info(TAG, "login SDK未初始化");
            notifyLogin(false, REASON_SDK_NOT_INIT);
        } else if (mState == STATE_IDLE) {
            if (TextUtils.isEmpty(userId) || TextUtils.isEmpty(password)) {
                JCLog.error(TAG, "账号密码内容无效");
                notifyLogin(false, REASON_INVALID_PARAM);
                return false;
            }
            String format = "^[A-Za-z0-9_+][A-Za-z0-9_+.-]{1,64}$";
            if (!userId.matches(format)) {
                JCLog.error(TAG, "账号不符合规则");
                notifyLogin(false, REASON_INVALID_PARAM);
                return false;
            }
            mUserId = userId;
            JCParam.Login param = new JCParam.Login();
            param.username = userId;
            param.password = password;
            param.server = mServer;
            param.appkey = mAppKey;
            param.deviceId = mDeviceId;
            param.autoCreate = true;
            if (MtcEngine.getInstance().login(param).succ) {
                JCLog.info(TAG, "进行登录");
                setState(STATE_LOGINING);
                return true;
            } else {
                JCLog.info(TAG, "登陆调用失败");
                notifyLogin(false, REASON_CALL_FUNCTION_ERROR);
            }
        } else if (mState == STATE_LOGINED) {
            JCLog.info(TAG, "已登录");
            notifyLogin(true, REASON_NONE);
            return true;
        } else if (mState == STATE_LOGINING) {
            MtcCli.Mtc_CliRefresh();
            return true;
        } else {
            JCLog.info(TAG, "当前状态不允许进行登录");
            notifyLogin(false, REASON_STATE_CANNOT_LOGIN);
        }
        return false;
    }

    @Override
    public boolean logout() {
        if (mState > STATE_IDLE) {
            if (MtcEngine.getInstance().logout(new JCParam.Logout()).succ) {
                JCLog.info(TAG, "登出");
            } else {
                notifyLogout(REASON_NONE);
            }
        } else {
            JCLog.info(TAG, "当前已登出");
            notifyLogout(REASON_NONE);
        }
        return true;
    }

    @Override
    public void onNotify(JCNotify notify) {
        if (notify.type == JCNotify.CLI) {
            if (notify.cliNotify.type == JCNotify.CLI_LOGIN_OK) {
                mUid = notify.cliNotify.loginOk.uid;
                notifyLogin(true, REASON_NONE);
                setState(STATE_LOGINED);
            } else if (notify.cliNotify.type == JCNotify.CLI_LOGIN_FAIL) {
                notifyLogin(false, translateFromMtc(notify.cliNotify.loginFail.statusCode));
                setState(STATE_IDLE);
            } else if (notify.cliNotify.type == JCNotify.CLI_DID_LOGOUT) {
                notifyLogout(REASON_NONE);
                setState(STATE_IDLE);
            } else if (notify.cliNotify.type == JCNotify.CLI_LOGOUTED) {
                notifyLogout(translateFromMtc(notify.cliNotify.logouted.statusCode));
                setState(STATE_IDLE);
            } else if (notify.cliNotify.type == JCNotify.CLI_RECONNECTING) {
                setState(STATE_LOGINING);
            } else if (notify.cliNotify.type == JCNotify.CLI_RECONNECT_OK) {
                setState(STATE_LOGINED);
            }
        }
    }

    @Override
    public void onNetChange(int newNetType, int oldNetType) {
        JCLog.info(TAG, "onNetChange");
        JCParam.Net param = new JCParam.Net();
        param.type = JCParam.Net.CHANGE;
        param.newNetType = translateToMtcNetType(newNetType);
        param.oldNetType = translateToMtcNetType(oldNetType);
        MtcEngine.getInstance().netChange(param);
    }

    @Override
    protected Context getContext() {
        return mContext;
    }

    @Override
    protected void addCallback(JCClientCallback callback) {
        mCallbacks.add(callback);
    }

    @Override
    protected void removeCallback(JCClientCallback callback) {
        mCallbacks.remove(callback);
    }

    void notifyLogin(final boolean result, @ClientReason final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "登陆结果:%b 原因:%d", result, reason);
                for (JCClientCallback callback : mCallbacks) {
                    callback.onLogin(result, reason);
                }
            }
        });
    }

    void notifyLogout(@ClientReason final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "登出原因:%d", reason);
                for (JCClientCallback callback : mCallbacks) {
                    callback.onLogout(reason);
                }
            }
        });
    }

    void notifyStateChange(@ClientState final int state, @ClientState final int oldState) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "状态 %d->%d", oldState, state);
                for (JCClientCallback callback : mCallbacks) {
                    callback.onClientStateChange(state, oldState);
                }
            }
        });
    }

    void setState(@ClientState int state) {
        if (mState != state) {
            int old = mState;
            mState = state;
            notifyStateChange(mState, old);
        }
    }

    @JCClient.ClientReason
    int translateFromMtc(int code) {
        switch (code) {
            case MTC_CLI_ERR_NO:
                return REASON_NONE;
            case MTC_CLI_REG_ERR_AUTH_FAILED:
                return REASON_AUTH;
            case MTC_CLI_REG_ERR_INVALID_USER:
                return REASON_NOUSER;
            case MTC_CLI_REG_ERR_TIMEOUT:
                return REASON_TIMEOUT;
            case MTC_CLI_REG_ERR_NETWORK:
                return REASON_NETWORK;
            case MTC_CLI_REG_ERR_APP_KEY:
                return REASON_APPKEY;
            case MTC_CLI_REG_ERR_DEACTED:
                return REASON_SERVER_LOGOUT;
            default:
                return REASON_OTHER;
        }
    }

    private int translateToMtcNetType(@JCNet.NetType int type) {
        if (type == JCNet.MOBILE) {
            return MTC_ANET_MOBILE;
        } else if (type == JCNet.WIFI) {
            return MTC_ANET_WIFI;
        } else {
            return MTC_ANET_UNAVAILABLE;
        }
    }

}
