package com.invms.awprotocol;

import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import android.app.Activity;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.view.KeyEvent;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.widget.Toast;

public class WebActivity extends Activity {
    private static final String tag = WebActivity.class.getName();

    public static final int showFileResultCode = 1;

    protected long backTime;
    protected boolean webDebug;
    protected String webKey;
    protected String webUrl;
    protected String webInvokers;
    protected WebView webView;
    protected WebClient webClient;
    protected WebChrome webChrome;
    protected WebHandler webHandler;

    protected List<IWebInvoker> listWebInvokers = new CopyOnWriteArrayList<>();
    protected Map<String, IWebInvoker> nameWebInvokers = new ConcurrentHashMap<>();
    protected Map<Integer, IWebInvoker> resultCodeWebInvokers = new ConcurrentHashMap<>();

    public boolean isWebDebug() {
        return webDebug;
    }

    public String getWebKey() {
        return webKey;
    }

    public String getWebUrl() {
        return webUrl;
    }

    public String getWebInvokers() {
        return webInvokers;
    }

    public WebView getWebView() {
        return webView;
    }

    public WebClient getWebClient() {
        return webClient;
    }

    public WebChrome getWebChrome() {
        return webChrome;
    }

    public WebHandler getWebHandler() {
        return webHandler;
    }

    public List<IWebInvoker> getListWebInvokers() {
        return listWebInvokers;
    }

    public Map<String, IWebInvoker> getNameWebInvokers() {
        return nameWebInvokers;
    }

    public Map<Integer, IWebInvoker> getResultCodeWebInvokers() {
        return resultCodeWebInvokers;
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        try {
            ActivityInfo activityInfo = getPackageManager().getActivityInfo(getComponentName(), PackageManager.GET_META_DATA);
            Bundle bundle = activityInfo.metaData;
            webDebug = AWRuntime.getConfig("webDebug", bundle.getBoolean("webDebug", false));
            Log.d(tag, "webDebug : " + webDebug);
            webKey = AWRuntime.getConfig("webKey", bundle.getString("webKey", "awprotocol"));
            Log.d(tag, "webKey : " + webKey);
            webUrl = AWRuntime.getConfig("webUrl", bundle.getString("webUrl", "file:///android_res/raw/index.html"));
            Log.d(tag, "webUrl : " + webUrl);
            webInvokers = AWRuntime.getConfig("webInvokers", bundle.getString("webInvokers", ""));
            webInvokers = webInvokers.replaceAll(" ", "").replaceAll("\r", "").replaceAll("\n", "");
            Log.d(tag, "webInvokers : " + webInvokers);
            WebView.setWebContentsDebuggingEnabled(webDebug);
            webView = new WebView(this);
            setContentView(webView);
            webClient = new WebClient(this);
            webView.setWebViewClient(webClient);
            webChrome = new WebChrome(this);
            webView.setWebChromeClient(webChrome);
            WebSettings webSettings = webView.getSettings();
            webSettings.setJavaScriptEnabled(true);
            webSettings.setJavaScriptCanOpenWindowsAutomatically(true);
            webSettings.setAllowFileAccessFromFileURLs(true);
            webSettings.setAllowUniversalAccessFromFileURLs(true);
            webHandler = new WebHandler(this);
            webView.addJavascriptInterface(webHandler, webKey);
            webView.loadUrl(webUrl);
            String[] classNames = TextUtils.split(webInvokers, ",");
            for (String item : classNames) {
                if (!TextUtils.isEmpty(item)) {
                    try {
                        addWebInvoker((IWebInvoker) Class.forName(item).newInstance());
                    } catch (Exception e) {
                        Log.e(tag, e.getLocalizedMessage(), e);
                    }
                }
            }
            for (IWebInvoker webInvoker : listWebInvokers) {
                Log.d(tag, webInvoker.getName() + " :onCreate: " + webInvoker.onCreate(savedInstanceState));
            }
        } catch (Exception e) {
            Log.e(tag, e.getLocalizedMessage(), e);
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        for (IWebInvoker webInvoker : listWebInvokers) {
            Log.d(tag, webInvoker.getName() + " :onResume: " + webInvoker.onResume());
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        for (IWebInvoker webInvoker : listWebInvokers) {
            Log.d(tag, webInvoker.getName() + " :onPause: " + webInvoker.onPause());
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        for (IWebInvoker webInvoker : listWebInvokers) {
            Log.d(tag, webInvoker.getName() + " :onDestroy: " + webInvoker.onDestroy());
        }
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        boolean result = false;
        for (IWebInvoker webInvoker : listWebInvokers) {
            if (webInvoker.onKeyDown(keyCode, event)) {
                result = true;
                Log.d(tag, webInvoker.getName() + " :onKeyDown: true");
            } else {
                Log.d(tag, webInvoker.getName() + " :onKeyDown: false");
            }
        }
        if (!result) {
            switch (keyCode) {
                case KeyEvent.KEYCODE_BACK:
                    if (webView.canGoBack()) {
                        webView.goBack();
                    } else {
                        long nowTime = (new Date()).getTime();
                        if (nowTime - backTime < 2000) {
                            finish();
                        } else {
                            backTime = nowTime;
                            Toast.makeText(this, "再按一次退出", Toast.LENGTH_SHORT).show();
                        }
                    }
                    return true;
            }
        }
        return super.onKeyDown(keyCode, event);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        switch (requestCode) {
            case showFileResultCode:
                webChrome.callbackShowFile(WebChrome.FileChooserParams.parseResult(resultCode, data));
                return;
            default:
                IWebInvoker webInvoker = resultCodeWebInvokers.get(requestCode);
                if (webInvoker != null) {
                    Log.d(tag, webInvoker.getName() + " :onActivityResult: " + webInvoker.onActivityResult(requestCode, resultCode, data));
                    return;
                }
                break;
        }
        super.onActivityResult(requestCode, resultCode, data);
    }

    public void webCallback(String callback, Object data) {
        String script = "javascript:" + callback + "(" + AWRuntime.toJson(data) + ")";
        Log.d(tag, "webCallback : " + script);
        runOnUiThread(() -> webView.evaluateJavascript(script, null));
    }

    public void addWebInvoker(IWebInvoker webInvoker) {
        if (webInvoker.getWebActivity() == null) {
            webInvoker.setWebActivity(this);
        }
        listWebInvokers.add(webInvoker);
        nameWebInvokers.put(webInvoker.getName().toLowerCase(Locale.CHINA), webInvoker);
        nameWebInvokers.put(webInvoker.getClass().getName().toLowerCase(Locale.CHINA), webInvoker);
        resultCodeWebInvokers.put(webInvoker.getRequestCode(), webInvoker);
    }
}