package com.instabug.featuresrequest.ui.addcomment;


import androidx.annotation.Nullable;

import com.instabug.featuresrequest.models.NewComment;
import com.instabug.featuresrequest.network.timelinerepository.FeatureRequestsRepository;
import com.instabug.featuresrequest.network.timelinerepository.RepositoryObserver;
import com.instabug.featuresrequest.settings.FeaturesRequestSettings;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONObject;

/**
 * Created by mzelzoghbi on 3/8/18.
 */

public class AddCommentPresenter extends BasePresenter<AddCommentContract.View>
        implements AddCommentContract.Presenter, RepositoryObserver<JSONObject> {

    @Nullable
    private FeatureRequestsRepository repository;
    @Nullable
    private final AddCommentContract.View viewInstance;
    @Nullable
    private volatile String enteredEmail;

    public AddCommentPresenter(AddCommentContract.View view) {
        super(view);
        PoolProvider.postIOTask(() -> {
            enteredEmail = InstabugCore.getEnteredEmail();
        });
        viewInstance = this.view.get();
        if (view.getViewContext() != null && view.getViewContext().getContext() != null) {
            repository = FeatureRequestsRepository.getInstance();
        }
    }

    @Override
    public void onCloseButtonClicked() {
        if (viewInstance != null) {
            viewInstance.onCloseButtonClicked();
        }
    }

    @Override
    public void onAddCommentButtonClicked() {
        if (viewInstance != null) {
            viewInstance.validate();
        }
    }

    @Override
    public void fillEmailAndName() {
        PoolProvider.postIOTask(() -> {
            final String userName = getEnteredUsername();
            final String email = getEnteredEmail();
            PoolProvider.postMainThreadTask(() -> {
                if (viewInstance != null) {
                    if (userName != null)
                        viewInstance.setUserName(userName);
                    if (email != null)
                        viewInstance.setUserEmail(email);
                }
            });
        });
    }

    @Override
    public void addComment(NewComment newComment) {
        if (viewInstance != null) {
            if (isUserIdentificationStateEnabled()) {
                InstabugCore.setEnteredUsername(viewInstance.getUsername());
                InstabugCore.setEnteredEmail(viewInstance.getEmail());
            }
            viewInstance.showLoadingDialog();
        }
        if (repository != null) {
            repository.addComment(newComment, this);
        }
    }

    private static boolean isUserIdentificationStateEnabled() {
        return FeaturesRequestSettings.getInstance().isUserIdentificationStateEnabled();
    }

    @Override
    public void onDataChanged(JSONObject featureComments) {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (viewInstance != null) {
                    viewInstance.dismissLoadingDialog();
                    viewInstance.onCommentAddedSuccessfully();
                }
            }
        });
    }

    @Override
    public void onError(Throwable error) {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (viewInstance != null) {
                    viewInstance.dismissLoadingDialog();
                    viewInstance.showError();
                }
            }
        });
    }

    @Override
    public boolean isCommenterEmailRequired() {
        return FeaturesRequestSettings.getInstance().isCommenterEmailFieldRequired()
                && isUserIdentificationStateEnabled();
    }

    @Override
    @Nullable

    public String getEnteredEmail() {
        return enteredEmail != null ? enteredEmail : InstabugCore.getEnteredEmail();
    }

    @Override
    @Nullable
    public String getEnteredUsername() {
        return InstabugCore.getEnteredUsername();
    }

    @Override
    public void handelRequiredFieldsHints() {
        if (viewInstance != null) {
            viewInstance.setEmailHintStringWithAsterisk(FeaturesRequestSettings.getInstance().isCommenterEmailFieldRequired());
        }
    }
}
