package com.instabug.featuresrequest.network.service;


import androidx.annotation.NonNull;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;

import java.net.HttpURLConnection;

/**
 * @author hossam.
 */

public class AddNewFeatureService {


    private static final String PARAM_FEATURE_REQUEST = "feature_request";
    private static final String PARAM_EMAIL = "email";
    private static final String PARAM_NAME = "name";
    private static final String PARAM_PUSH_TOKEN = "push_token";
    private static volatile AddNewFeatureService addNewFeatureServiceInstance;
    private NetworkManager networkManager;

    private AddNewFeatureService() {
        if (addNewFeatureServiceInstance != null) {
            throw new RuntimeException("Use getInstance() method to get the single instance" +
                    " of this class");
        }
        networkManager = new NetworkManager();
    }

    public static AddNewFeatureService getInstance() {
        if (addNewFeatureServiceInstance == null) {
            synchronized (AddNewFeatureService.class) {
                if (addNewFeatureServiceInstance == null)
                    addNewFeatureServiceInstance = new AddNewFeatureService();
            }
        }
        return addNewFeatureServiceInstance;
    }

    public void sendNewFeature(final FeatureRequest featureRequest,
                               final Request.Callbacks<Boolean, Throwable> sendFeatureCallbacks
    ) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Sending new feature");
        // create new feature request
        PoolProvider.postIOTask(() -> {
            try {
                Request sendFeatureRequest = createSendFeatureRequest(featureRequest);
                Request.Callbacks<RequestResponse, Throwable> requestCallbacks = createRequestCallback(sendFeatureCallbacks);
                // do request with default connection timeout.
                networkManager.doRequest(IBGNetworkWorker.FEATURES_REQUEST,
                        RequestType.NORMAL,
                        sendFeatureRequest,
                        requestCallbacks);
            } catch (JSONException exception) {
                sendFeatureCallbacks.onFailed(exception);
            }
        });
    }

    private Request createSendFeatureRequest(final @NonNull FeatureRequest featureRequest) throws JSONException {
        Request.Builder builder = new Request.Builder().endpoint(Endpoints.ADD_NEW_FEATURE)
                .method(RequestMethod.POST).addParameter(new RequestParameter(PARAM_PUSH_TOKEN, featureRequest.getDeviceToken()))
                .addParameter(new RequestParameter(PARAM_FEATURE_REQUEST, featureRequest.getFeatureJsonObject()));
        String requesterEmail = featureRequest.getRequesterEmail();
        if (requesterEmail != null && !requesterEmail.trim().isEmpty())
            builder.addParameter(new RequestParameter<>(PARAM_EMAIL, requesterEmail));
        String requesterName = featureRequest.getRequesterName();
        if (requesterName != null && !requesterName.trim().isEmpty())
            builder.addParameter(new RequestParameter<>(PARAM_NAME, requesterName));

        return builder.build();
    }

    private Request.Callbacks<RequestResponse, Throwable> createRequestCallback(final Request.Callbacks<Boolean, Throwable> sendFeatureCallbacks) {
        return new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "sendFeatureRequest request Succeeded, Response code: " +
                        requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "Sending feature request Response body: " + requestResponse.getResponseBody());
                if (requestResponse.getResponseCode() == HttpURLConnection.HTTP_OK &&
                        requestResponse.getResponseBody() != null) {
                    sendFeatureCallbacks.onSucceeded(true);
                } else {
                    sendFeatureCallbacks.onSucceeded(false);
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "sendFeatureRequest request got error: ", error);
                sendFeatureCallbacks.onFailed(error);
            }
        };
    }
}
