package com.instabug.featuresrequest.cache;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.models.TimelineResponse;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.util.InstabugSDKLogger;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;


/**
 * Created by mzelzoghbi on 3/5/18.
 */

public class TimelineCacheManager {

    public static final String TIMELINE_DISK_CACHE_FILE_NAME = "/timeline.cache";
    public static final String TIMELINE_DISK_CACHE_KEY = "timeline_disk_cache";
    public static final String TIMELINE_MEMORY_CACHE_KEY = "timeline_memory_cache";


    /**
     * Loads timeline cache from disk if it's not in memory
     *
     * @return in-memory cache for timeline
     * @throws IllegalArgumentException if the from cache is not found
     */
    @Nullable
    public static InMemoryCache<Long, TimelineResponse> getCache() throws IllegalArgumentException {
        if (!CacheManager.getInstance().cacheExists(TIMELINE_MEMORY_CACHE_KEY)) {
            CacheManager.getInstance().migrateCache(TIMELINE_DISK_CACHE_KEY,
                    TIMELINE_MEMORY_CACHE_KEY, new CacheManager.KeyExtractor<Long,
                            TimelineResponse>() {
                        @Override
                        public Long extractKey(TimelineResponse value) {
                            return value.getFeatureId();
                        }
                    });
            Cache commentsMemoryCache =
                    CacheManager.getInstance().getCache(TIMELINE_MEMORY_CACHE_KEY);
            if (commentsMemoryCache != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG,
                        "In-memory Comments cache restored from disk, "
                                + commentsMemoryCache.size()
                                + " elements restored");
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory Comments cache found");
        return (InMemoryCache<Long, TimelineResponse>) CacheManager.getInstance().getCache
                (TIMELINE_MEMORY_CACHE_KEY);
    }


    @VisibleForTesting
    static void tearDown() {
        CacheManager.getInstance().deleteCache(TIMELINE_DISK_CACHE_KEY);
        CacheManager.getInstance().deleteCache(TIMELINE_MEMORY_CACHE_KEY);
    }

    /**
     * Saves all cached timeline from {@link TimelineCacheManager#TIMELINE_MEMORY_CACHE_KEY }to disk cache
     */
    public static void saveCacheToDisk() {

        Cache timelineDiskCache = CacheManager.getInstance().getCache(TIMELINE_DISK_CACHE_KEY);
        Cache timelineMemoryCache = CacheManager.getInstance().getCache(TIMELINE_MEMORY_CACHE_KEY);

        if (timelineDiskCache != null && timelineMemoryCache != null) {
            CacheManager.getInstance().migrateCache(timelineMemoryCache, timelineDiskCache,
                    new CacheManager.KeyExtractor<String, TimelineResponse>() {
                        @Override
                        public String extractKey(TimelineResponse value) {
                            return String.valueOf(value.getFeatureId());
                        }
                    });
        }
    }

    @Nullable
    public static TimelineResponse getComments(long featureId) {
        if(getCache() == null) return null;
        return getCache().get(featureId);
    }

    public static void addTimeline(TimelineResponse timeline) {
        InMemoryCache<Long, TimelineResponse> cache = getCache();
        if (cache != null) {
            cache.put(timeline.getFeatureId(), timeline);
        }
    }
}
