package com.instabug.chat.model;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.chat.util.PlaceHolderResolver;
import com.instabug.library.Feature;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.encryption.EncryptionManager;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.model.BaseReport;
import com.instabug.library.model.State;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;

/**
 * @author mesbah
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class Chat extends BaseReport implements Cacheable, Serializable {

    // keys used by toJson() & fromJson() methods
    public static final String KEY_ID = "id";
    public static final String KEY_MESSAGES = "messages";
    public static final String KEY_STATE = "state";
    public static final String KEY_CHAT_STATE = "chat_state";
    private String id;
    @Nullable
    private State state;
    private ArrayList<Message> messages;
    @Nullable
    private ChatState chatState;

    public Chat(@NonNull String id, @Nullable State state, @NonNull ChatState chatState) {
        this.id = id;
        this.state = state;
        this.chatState = chatState;
        messages = new ArrayList<>();
    }

    public Chat(@NonNull String id) {
        this.id = id;
        messages = new ArrayList<>();
        setChatState(ChatState.SENT);
    }

    public Chat() {
        this.chatState = ChatState.NOT_AVAILABLE;
        messages = new ArrayList<>();
    }

    /**
     * @return Id of this chat.
     */
    public String getId() {
        return id;
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public Chat setId(String id) {
        this.id = id;
        updateMessagesWithChatId();
        return this;
    }

    /**
     * @return Messages in this chat.
     */
    public ArrayList<Message> getMessages() {
        return messages;
    }

    public Chat setMessages(ArrayList<Message> messages) {
        this.messages = messages;
        ArrayList<Message> nullList = new ArrayList<>();
        nullList.add(null);
        messages.removeAll(nullList);
        updateMessagesWithChatId();
        return this;
    }

    /**
     * @return State of this chat if exist.
     */
    @Nullable
    public State getState() {
        return state;
    }

    public Chat setState(@Nullable State state) {
        this.state = state;
        return this;
    }

    /**
     * @return ChatState of this chat {#code ChatState}
     */
    @Nullable
    public ChatState getChatState() {
        return chatState;
    }

    public Chat setChatState(@Nullable ChatState chatState) {
        this.chatState = chatState;
        return this;
    }

    /**
     * @return number of unread messages in this chat
     */
    public int getUnreadCount() {
        int unread = 0;
        for (Message message : messages) {
            if (!message.isRead()) {
                unread++;
            }
        }
        return unread;
    }

    /**
     * Marks all messages in this chat as read
     *
     * @throws IllegalStateException if no messages are in this chat
     */
    public void markRead() {
        for (int i = messages.size() - 1; i >= 0; i--) {
            messages.get(i).setRead(true);
        }
    }

    /**
     * @return last message sender avatar <b>assumes list is sorted, no sorting here to enhance
     * performance</b>
     * message must be outbound
     * @throws IllegalStateException if no messages are in this chat
     */
    @Nullable
    public String getSenderAvatarUrl() {
        Message lastOutboundMessage = getLastOutboundMessage();
        if (lastOutboundMessage != null) {
            return lastOutboundMessage.getSenderAvatarUrl();
        }
        return null;
    }

    /**
     * @return last message sender name or Default name if non found <b>assumes list is sorted,
     * no sorting here to enhance performance</b>
     * message must be outbound
     * @throws IllegalStateException if no messages are in this chat
     */
    @Nullable
    public String getSenderName() {
        Message lastOutboundMessage = getLastOutboundMessage();
        if (lastOutboundMessage != null) {
            return lastOutboundMessage.getSenderName();
        } else if (messages.size() != 0) {
            return messages.get(messages.size() - 1).getSenderName();
        } else {
            return "";
        }

    }

    /**
     * @return last outbound message or null if no outbound messages exist
     */
    @Nullable
    private Message getLastOutboundMessage() {
        Message lastSyncedMessage = getLastSyncedMessage();
        if (lastSyncedMessage != null && lastSyncedMessage.isInbound()) {
            for (Message message : messages) {
                if (!message.isInbound()) {
                    return message;
                }
            }
            return null;
        } else {
            return lastSyncedMessage;
        }
    }

    /**
     * @return last message timestamp <b>assumes list is sorted, no sorting here to enhance
     * performance</b>
     * @throws IllegalStateException if no messages are in this chat
     */
    public long getLastMessageDate() {
        if (getLastMessage() != null)
            return getLastMessage().getMessagedAt();
        else
            return 0L;
    }

    /**
     * <b>Note: </b> assumes list is sorted, no sorting here to enhance performance
     *
     * @return last message in {@link #messages}
     * @throws IllegalStateException if no messages are in this conversation
     */
    @Nullable
    public Message getLastMessage() {
        if (messages != null && messages.size() != 0) {
            Collections.sort(messages, new Message.Comparator(Message.Comparator.COMPARE_BY_DATE));
            return messages.get(messages.size() - 1);
        } else
            return null;
    }

    /**
     * <b>Note: </b> assumes list is sorted, no sorting here to enhance performance
     *
     * @return last message synced to server in {@link #messages}
     * @throws IllegalStateException if no messages are in this chat
     */
    @Nullable
    public Message getLastSyncedMessage() {
        for (int i = messages.size() - 1; i >= 0; i--) {
            if (messages.get(i).getMessageState() == Message.MessageState.SYNCED) {
                return messages.get(i);
            }
        }
        return null;
    }

    private void updateMessagesWithChatId() {
        for (int i = 0; i < getMessages().size(); i++) {
            getMessages().get(i).setChatId(id);
        }
    }

    public String getTitle() {
        String senderName = getSenderName();
        if (senderName != null && !senderName.equals("") && !senderName.equals(" ") &&
                !senderName.equals("null") && getLastMessage() != null
                && !getLastMessage().isInbound()) {
            return senderName;
        } else {
            return PlaceHolderResolver.getTeamTitleResolved();
        }
    }

    public String toJson() throws JSONException {
        JSONObject chatJsonObject = new JSONObject();
        chatJsonObject.put(KEY_ID, getId())
                .put(KEY_MESSAGES, Message.toJson(getMessages()));
        if (getChatState() != null) {
            chatJsonObject.put(KEY_CHAT_STATE, getChatState().toString());
        }
        if (getState() != null)
            chatJsonObject.put(KEY_STATE, getState().toJson());
        if (InstabugCore.getEncryptionState() == Feature.State.ENABLED) {
            String encrypted = EncryptionManager.encrypt(chatJsonObject.toString());
            return encrypted != null ? encrypted : chatJsonObject.toString();
        } else {
            return chatJsonObject.toString();
        }
    }

    @Override
    public void fromJson(String chatASJson) throws JSONException {
        chatASJson = EncryptionManager.decrypt(chatASJson);
        if (chatASJson != null) {
            JSONObject chatJsonObject = new JSONObject(chatASJson);
            if (chatJsonObject.has(KEY_ID))
                setId(chatJsonObject.getString(KEY_ID));
            if (chatJsonObject.has(KEY_MESSAGES))
                setMessages(Message.fromJson(chatJsonObject.getJSONArray(KEY_MESSAGES)));
            if (chatJsonObject.has(KEY_CHAT_STATE))
                setChatState(ChatState.valueOf(chatJsonObject.getString(KEY_CHAT_STATE)));
            if (chatJsonObject.has(KEY_STATE)) {
                State state = new State();
                state.fromJson(chatJsonObject.getString(KEY_STATE));
                setState(state);
            }
        }
    }

    @Override
    @NonNull
    public String toString() {
        return "Chat:[" + id + " chatState: " + getChatState() + "]";
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public boolean equals(Object chat) {
        if (chat instanceof Chat) {
            Chat comparedChat = (Chat) chat;
            if (String.valueOf(comparedChat.getId()).equals(getId())
                    && comparedChat.getChatState() == getChatState() &&
                    ((comparedChat.getState() == null && getState() == null)
                            || (getState() != null && comparedChat.getState() != null
                            && comparedChat.getState().equals(getState())))) {
                for (int i = 0; i < comparedChat.getMessages().size(); i++) {
                    if (!(comparedChat.getMessages().get(i).equals(getMessages().get(i))))
                        return false;
                }
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        if (getId() != null)
            return getId().hashCode();
        else
            return -1;
    }

    public enum ChatState {
        WAITING_ATTACHMENT_MESSAGE, READY_TO_BE_SENT, LOGS_READY_TO_BE_UPLOADED, SENT, NOT_AVAILABLE
    }

    public static class Comparator implements java.util.Comparator<Chat>, Serializable {

        @Override
        public int compare(Chat lhs, Chat rhs) {
            Date lhsDate = new Date(lhs.getLastMessageDate());
            Date rhsDate = new Date(rhs.getLastMessageDate());
            return lhsDate.compareTo(rhsDate);
        }
    }

    public static class Factory {
        public Chat create(final Context context) {
            final Chat chat = new Chat(System.currentTimeMillis() + "", null,
                    ChatState.READY_TO_BE_SENT);
            PoolProvider.postIOTask(new Runnable() {
                @Override
                public void run() {
                    chat.setState(new State.Builder(context).build(true));
                }
            });
            return chat;
        }
    }

}
