package com.instabug.bug.view.extrafields;

import static com.instabug.bug.view.reporting.BaseReportingPresenter.SHOW_THANK_YOU_ACTIVITY;
import static com.instabug.library.util.DrawableUtils.getRotateDrawable;
import static com.instabug.library.util.ThemeApplier.applyPrimaryTextStyle;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.os.Handler;
import android.os.SystemClock;
import android.text.Editable;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.ContextCompat;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.instabug.bug.Constants;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.R;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.model.ExtraReportField;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.view.BugReportingActivityCallback;
import com.instabug.bug.view.InstabugThanksActivity;
import com.instabug.bug.view.reporting.ReportingContainerActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.KeyboardUtils;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.SimpleTextWatcher;
import com.instabug.library.util.ThemeApplier;

import java.lang.ref.WeakReference;
import java.util.List;

/**
 * Created by tarek on 7/26/17.
 */

public class ExtraFieldsFragment extends InstabugBaseFragment<ExtraFieldsPresenter> implements
        ExtraFieldsContract.View {

    public static final String TAG = "ExtraFieldsFragment";

    @Nullable
    @VisibleForTesting
    String title;
    private static final String KEY_TITLE = "title";
    @Nullable
    List<ExtraReportField> extraReportFields;
    @Nullable
    private LinearLayout itemsLinearLayout;
    private long lastClickTime;
    private boolean isCommittingBug;
    @Nullable
    private BugReportingActivityCallback activityCallback;
    private String currentTitle = "";

    public static ExtraFieldsFragment newInstance(@Nullable String title) {
        Bundle args = new Bundle();
        args.putString(KEY_TITLE, title);

        ExtraFieldsFragment fragment = new ExtraFieldsFragment();
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setHasOptionsMenu(true);
        if (getActivity() != null) {
            getActivity().getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_HIDDEN);
        }

        if (getArguments() != null) {
            title = getArguments().getString(KEY_TITLE);
        }
        presenter = new ExtraFieldsPresenter(this);
        if (activityCallback != null) {
            currentTitle = activityCallback.getToolbarTitle();
            if (title != null) {
                activityCallback.setToolbarTitle(title);
            }
            activityCallback.setToolbarUpIconBack();
        }
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        if (getActivity() instanceof BugReportingActivityCallback) {
            try {
                activityCallback = (BugReportingActivityCallback) getActivity();
            } catch (Exception e) {
                throw new RuntimeException("Must implement BugReportingActivityCallback");
            }
        }
    }

    @Override
    protected int getLayout() {
        return R.layout.ibg_bug_lyt_extra_fields;
    }

    @Override
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        handleExtraFields();
    }

    @Nullable
    protected String getTitle() {
        return title;
    }

    protected void onDoneButtonClicked() {
        if (presenter != null) {
            if (!presenter.validateExtraReportFields()) {
                return;
            }
            if (extraReportFields != null) {
                presenter.appendExtraReportFieldsToMessage(extraReportFields);
            }
            isCommittingBug = true;
            if (getContext() != null) {
                LiveBugManager.getInstance().commit();
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't commit the Bug due to Null context");
            }
            navigateToSuccessFragment();
        }
    }

    @Override
    public void onStart() {
        super.onStart();
        if (getActivity() instanceof ReportingContainerActivity) {
            ((ReportingContainerActivity) getActivity()).setToolbarNavigationContentDescription(R.string.ibg_core_extended_report_ic_close_content_description);
        }
    }

    private void handleExtraFields() {
        if (presenter != null) {
            List<ExtraReportField> extraReportFields = presenter.getExtraReportFields();
            if (extraReportFields != null && getContext() != null) {
                itemsLinearLayout = findViewById(R.id.linearLayout);
                for (int i = 0; i < extraReportFields.size(); i++) {
                    LinearLayout instabugEditText = (LinearLayout) LayoutInflater.from(getContext())
                            .inflate(R.layout.ibg_bug_item_edittext, itemsLinearLayout, false);
                    instabugEditText.setId(i);
                    final InstabugEditTextViewholder instabugEditTextViewholder = new InstabugEditTextViewholder(instabugEditText);
                    if (instabugEditTextViewholder.getEditText() != null) {
                        instabugEditTextViewholder.getEditText().setHint(extraReportFields.get(i).isRequired()
                                ? extraReportFields.get(i).getHint() + " *"
                                : extraReportFields.get(i).getHint());
                        if (extraReportFields.get(i).getValue() != null) {
                            instabugEditTextViewholder.getEditText().setText(extraReportFields.get(i).getValue());
                        }
                        instabugEditTextViewholder.getEditText().setId(i);
                        instabugEditTextViewholder.getEditText().addTextChangedListener(
                                new CustomTextSwitcher(instabugEditTextViewholder.getEditText()));
                        instabugEditTextViewholder.getEditText().setImeOptions(EditorInfo.IME_ACTION_DONE);

                        // Configure Accessibility
                        if (AccessibilityUtils.isTalkbackEnabled()) {
                            EditText editText = instabugEditTextViewholder.getEditText();
                            if (editText != null) {
                                final int finalIndex = i;
                                ViewCompat.setAccessibilityDelegate(editText, new AccessibilityDelegateCompat() {
                                    @Override
                                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                                        super.onInitializeAccessibilityNodeInfo(host, info);
                                        final ExtraReportField extraReportField = extraReportFields.get(finalIndex);
                                        if (extraReportField != null && extraReportField.getFieldAccessibilityContentDescription() != null) {
                                            info.setText(extraReportField.getFieldAccessibilityContentDescription());
                                            info.setShowingHintText(true);
                                        }
                                    }
                                });
                            }
                        }
                    }
                    if (itemsLinearLayout != null) {
                        itemsLinearLayout.addView(instabugEditText);
                    }
                    applyTheme(instabugEditTextViewholder);
                }
            }
            this.extraReportFields = extraReportFields;
        }
    }

    @VisibleForTesting
    void applyTheme(InstabugEditTextViewholder instabugEditTextViewholder) {
        if(theme != null) {
            if (instabugEditTextViewholder != null) {
                applyPrimaryTextStyle(instabugEditTextViewholder.editText, theme);
                if (theme.getSecondaryTextColor() != ThemeApplier.DEFAULT_COLOR && instabugEditTextViewholder.editText != null) {
                    instabugEditTextViewholder.editText.setHintTextColor(theme.getSecondaryTextColor());
                }
            }

            LinearLayout container = rootView.findViewById(R.id.linearLayout);
            if(container != null && theme.getBackgroundColor() != ThemeApplier.DEFAULT_COLOR) {
                container.setBackgroundColor(theme.getBackgroundColor());
            }
        }
    }

    @Override
    public void clearError(int editTextId) {
        InstabugEditTextViewholder instabugEditTextViewholder = new InstabugEditTextViewholder(findViewById(editTextId));
        instabugEditTextViewholder.reset();
    }

    public void navigateToSuccessFragment() {
        if (getActivity() != null) {
            KeyboardUtils.hide(getActivity());
        }
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                if (BugSettings.getInstance().successDialogEnabled() &&
                        !ServiceLocator.getConfigurationsProvider().isBugReportingUsageExceeded()) {
                    Activity ownerActivity = getActivity();
                    if (ownerActivity != null) {
                        Intent intent = new Intent(ownerActivity, InstabugThanksActivity.class);
                        ownerActivity.startActivityForResult(intent, SHOW_THANK_YOU_ACTIVITY);
                    }
                } else {
                    finishActivity();
                }
            }
        }, 200);
    }

    @Override
    public void focusAndSetError(int editTextId) {
        if (extraReportFields != null) {
            ExtraReportField extraReportField = extraReportFields.get(editTextId);
            String message = getLocalizedString(R.string
                    .instabug_err_invalid_extra_field, extraReportField.getHint());
            InstabugEditTextViewholder instabugEditTextViewholder = new InstabugEditTextViewholder(findViewById(editTextId));
            if (instabugEditTextViewholder.getEditText() != null) {
                instabugEditTextViewholder.getEditText().requestFocus();
            }
            instabugEditTextViewholder.setError(message);
        }
    }

    private class CustomTextSwitcher extends SimpleTextWatcher {
        private final WeakReference<EditText> mEditText;

        public CustomTextSwitcher(EditText editText) {
            mEditText = new WeakReference<>(editText);
        }

        @Override
        public void afterTextChanged(Editable s) {
            super.afterTextChanged(s);
            final EditText et = mEditText.get();
            if (et != null && extraReportFields != null) {
                extraReportFields.get(et.getId()).setValue(s.toString());
            }
        }
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        inflater.inflate(R.menu.ibg_bug_menu_extended_reporting, menu);
        MenuItem sendMenuItem = menu.findItem(R.id.instabug_bugreporting_send);
        MenuItem nextIcon = menu.findItem(R.id.instabug_bugreporting_next);
        if (nextIcon != null) {
            nextIcon.setVisible(false);
        }
        if (sendMenuItem != null) {
            sendMenuItem.setVisible(true);
            sendMenuItem.setTitle(getLocalizedString(R.string.ibg_report_send_content_description));
        }
        if (getContext() != null) {
            if (LocaleUtils.isRTL(InstabugCore.getLocale(getContext()))) {
                MenuItem menuItemSend = menu.findItem(R.id.instabug_bugreporting_send);
                if (menuItemSend != null) {
                    Drawable icon = menuItemSend.getIcon();
                    if (icon != null)
                        menu.findItem(R.id.instabug_bugreporting_send).setIcon(
                                getRotateDrawable(icon, 180));
                }
            }
        }
    }

    @Override
    public void onDestroy() {
        if (activityCallback != null) {
            activityCallback.setToolbarUpIconClose();
            activityCallback.setToolbarTitle(currentTitle);
        }
        super.onDestroy();
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (isCommittingBug)
            return false;
        // mis-clicking prevention, using threshold of 1000 ms
        if (SystemClock.elapsedRealtime() - lastClickTime < 1000) {
            return false;
        }
        lastClickTime = SystemClock.elapsedRealtime();

        if (item.getItemId() == R.id.instabug_bugreporting_send) {
            onDoneButtonClicked();
            return true;
        } else if (item.getItemId() == android.R.id.home) {
            if (getActivity() != null)
                getActivity().onBackPressed();
        }

        return false;
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        if (itemsLinearLayout != null) {
            itemsLinearLayout.clearFocus();
            itemsLinearLayout.removeAllViews();
        }
        itemsLinearLayout = null;
    }

    static class InstabugEditTextViewholder extends RecyclerView.ViewHolder {

        @Nullable
        private EditText editText;
        @Nullable
        private TextView error;
        @Nullable
        private View seperator;

        private int separatorColor = -1;

        public InstabugEditTextViewholder(View itemView) {
            super(itemView);
            if (itemView instanceof LinearLayout) {
                LinearLayout layout = (LinearLayout) itemView;
                for (int i = 0; i < layout.getChildCount(); i++) {
                    View child = layout.getChildAt(i);
                    if (child instanceof EditText) {
                        editText = (EditText) child;
                    } else if (child instanceof TextView) {
                        error = (TextView) child;
                    } else {
                        seperator = child;
                    }
                }

                if (SettingsManager.getInstance().getIBGTheme() != null) {
                    separatorColor = SettingsManager.getInstance().getIBGTheme().getSecondaryTextColor();
                }

                if (separatorColor != ThemeApplier.DEFAULT_COLOR && seperator != null) {
                    seperator.setBackgroundColor(separatorColor);
                }
            }
        }

        @Nullable
        public EditText getEditText() {
            return editText;
        }

        @Nullable
        public TextView getErrorTextView() {
            return error;
        }

        public void setError(String errorMsg) {
            if (error != null && seperator != null) {
                error.setText(errorMsg);
                seperator.setBackgroundColor(ContextCompat.getColor(itemView.getContext(), R.color.instabug_extrafield_error));
            }
        }

        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public void reset() {
            if (error != null && seperator != null) {
                error.setText(null);
                if (separatorColor != ThemeApplier.DEFAULT_COLOR) {
                    seperator.setBackgroundColor(separatorColor);

                } else {
                    seperator.setBackgroundColor(AttrResolver.resolveAttributeColor(itemView.getContext(), R.attr.ibg_bug_vus_separator_color));
                }
            }
        }
    }
}
