package com.instabug.bug.network;

import static com.instabug.bug.Constants.FEATURE_NAME;
import static com.instabug.bug.utils.DeleteBugsUtilKt.deleteBug;
import static com.instabug.bug.utils.DeleteBugsUtilKt.deleteBugAndStateFile;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.cache.BugTable;
import com.instabug.bug.configurations.BugReportingConfigurationsProvider;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.model.Bug;
import com.instabug.bug.testingreport.ReportUploadingStateEventBus;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.IOException;
import java.util.List;

import kotlin.Pair;


public abstract class InstabugBugsUploaderJob extends InstabugNetworkJob {

    private static final String TAG = "InstabugBugsUploaderJob";

    private final static BugReportingConfigurationsProvider configurationProvider = ServiceLocator.getConfigurationsProvider();

    private static boolean continueUploadBugs;

    protected abstract List<Bug> getBugs(Context context);

    protected abstract void onBugUploaded();

    private  void uploadBugs(final Context context) {
        continueUploadBugs = true;
        List<Bug> bugList = getBugs(context);
        InstabugSDKLogger.d(Constants.LOG_TAG, "Found " + bugList.size() + " bugs in cache");
        for (final Bug bug : bugList) {
            if (!continueUploadBugs) break;

            if (bug.getBugState().equals(Bug.BugState.READY_TO_BE_SENT)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading bug: " + bug);
                if (configurationProvider.isBugReportingRateLimited()) {
                    deleteBug(bug, context);
                    logRateIsLimited();
                    continue;
                }
                configurationProvider.setLastBugRequestStartedAt(System.currentTimeMillis());
                BugsService.getInstance().reportBug(context, bug, new Request.Callbacks<String, Throwable>() {
                    @Override
                    public void onSucceeded(@Nullable String temporaryServerToken) {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "Bug uploaded successfully, " +
                                "setting bug TemporaryServerToken equal " + temporaryServerToken);
                        bug.setTemporaryServerToken(temporaryServerToken);
                        bug.setBugState(Bug.BugState.LOGS_READY_TO_BE_UPLOADED);

                        //updating bug in db
                        IBGContentValues contentValues = new IBGContentValues();
                        if (temporaryServerToken != null) {
                            Pair<String, Boolean> tempServerTokenKey = BugTable.getCOLUMN_TEMPORARY_SERVER_TOKEN();
                            contentValues.put(tempServerTokenKey.component1(), temporaryServerToken, tempServerTokenKey.component2());
                        }
                        Pair<String, Boolean> bugStateKey = BugTable.getCOLUMN_BUG_STATE();
                        contentValues.put(bugStateKey.component1(), Bug.BugState.LOGS_READY_TO_BE_UPLOADED.name(), bugStateKey.component2());
                        if (bug.getId() != null) {
                            ServiceLocator.getBugReportsDbHelper().update(bug.getId(), contentValues);
                        }
                        configurationProvider.setLastBugRequestStartedAt(0L);
                        uploadBugLogs(bug, context);
                        onBugUploaded();
                    }

                    @Override
                    public void onFailed(Throwable throwable) {
                        if (throwable instanceof RateLimitedException)
                            handleRateLimitedException((RateLimitedException) throwable, bug, context);
                        else
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while uploading bug");

                        ReportUploadingStateEventBus.INSTANCE.postError(throwable);

                        handleNetworkFailure(throwable);
                        updateBugConnectionErrorIfNeeded(bug, throwable);
                    }

                    @Override
                    public void onDisconnected() {
                        ServiceLocator.getBugReportsDbHelper().updateConnectionError(bug.getId(), Bug.CONNECTION_ERROR_DISCONNECTED);
                    }

                    @Override
                    public void onRetrying(Throwable throwable) {
                        updateBugConnectionErrorIfNeeded(bug, throwable);
                    }
                });
            } else if (bug.getBugState().equals(Bug.BugState.LOGS_READY_TO_BE_UPLOADED)) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Bug: " + bug +
                        " already uploaded but has unsent logs, uploading now");
                uploadBugLogs(bug, context);
            } else if (bug.getBugState().equals(Bug.BugState.ATTACHMENTS_READY_TO_BE_UPLOADED)) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Bug: " + bug +
                        " already uploaded but has unsent attachments, uploading now");
                uploadAttachments(bug, context);
            }
        }
    }

    private static void handleRateLimitedException(RateLimitedException exception, @NonNull Bug bug, Context context) {
        configurationProvider.setBugReportingLimitedUntil(exception.getPeriod());
        logRateIsLimited();
        deleteBug(bug, context);
    }

    private static void logRateIsLimited() {
        InstabugSDKLogger.d(Constants.LOG_TAG, String.format(RateLimitedException.RATE_LIMIT_REACHED, FEATURE_NAME));
    }

    private void uploadBugLogs(final Bug bug, final Context context) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "START uploading all logs related to this bug id = " + bug.getId());
        BugsService.getInstance().uploadBugLogs(bug, new Request.Callbacks<Boolean, Throwable>() {

            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Bug logs uploaded successfully, " +
                        "change its state");
                if (bug.getId() == null) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't update the bug's state because its ID is null");
                    return;
                }
                bug.setBugState(Bug.BugState.ATTACHMENTS_READY_TO_BE_UPLOADED);

                //updating bug in db
                Pair<String, Boolean> bugStateKey = BugTable.getCOLUMN_BUG_STATE();
                IBGContentValues contentValues = new IBGContentValues();
                contentValues.put(bugStateKey.component1(), Bug.BugState.ATTACHMENTS_READY_TO_BE_UPLOADED.name(), bugStateKey.component2());
                ServiceLocator.getBugReportsDbHelper().update(bug.getId(), contentValues);

                try {
                    uploadAttachments(bug, context);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while" +
                            " uploading bug attachments e: " + e.getMessage());
                    handleNetworkFailure(e);
                }
            }

            @Override
            public void onFailed(Throwable throwable) {
                String uploadLogsErrorMessage = "Something went wrong while uploading bug logs";
                InstabugSDKLogger.d(Constants.LOG_TAG, uploadLogsErrorMessage);
                ReportUploadingStateEventBus.INSTANCE.postError(new Exception(uploadLogsErrorMessage));
                handleNetworkFailure(throwable);
            }
        });
    }

    private static void updateBugConnectionErrorIfNeeded(Bug bug, Throwable throwable) {
        if (bug.getConnectionError() == null && throwable instanceof IOException) {
            String errorType = throwable.getClass().getName();
            ServiceLocator.getBugReportsDbHelper().updateConnectionError(bug.getId(), errorType);
            bug.setConnectionError(errorType);
        }
    }

    private void uploadAttachments(final Bug bug, final Context context) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Found " + bug.getAttachments().size() + " attachments related to " +
                "bug: " + bug.getMessage());
        BugsService.getInstance().uploadBugAttachments(bug, new Request.Callbacks<Boolean, Throwable>() {
            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Bug attachments uploaded successfully");
                if (context != null)
                    deleteBugAndStateFile(bug, context);
                else
                    InstabugSDKLogger.e(Constants.LOG_TAG, "unable to delete state file for Bug with id: " + bug.getId()
                            + "due to null context reference");

            }

            @Override
            public void onFailed(Throwable throwable) {
                String errorMessage = "Something went wrong while uploading bug attachments";
                InstabugSDKLogger.d(Constants.LOG_TAG, errorMessage);
                ReportUploadingStateEventBus.INSTANCE.postError(new Exception(errorMessage));
                handleNetworkFailure(throwable);
            }
        });
    }

    public static void handleNetworkFailure(Throwable throwable) {
        //Toggle upload bugs flag to stop sending the remaining bugs if a network error is occurred.
        if (throwable instanceof IOException)
            continueUploadBugs = false;
    }

    public void start() {
        enqueueRetryingJob(TAG, () -> {
                    if (Instabug.getApplicationContext() != null) {
                        uploadBugs(Instabug.getApplicationContext());
                    } else {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "Context was null during Bugs syncing");
                    }
                },
                (e) -> InstabugSDKLogger.e(Constants.LOG_TAG, "Error occurred while uploading bugs", e)
        );
    }
}
