package com.instabug.bug

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.instabug.bug.di.ServiceLocator
import com.instabug.bug.network.ProactiveReportsBugsUploaderJob
import com.instabug.bug.network.NormalBugsUploaderJob
import com.instabug.bug.settings.BugSettings
import com.instabug.bug.view.actionList.service.FetchReportCategoriesJob
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.screenshot.analytics.AnalyticsEvent
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus
import com.instabug.library.util.InstabugSDKLogger

object BugsCoreEventsHandler {

    @VisibleForTesting
    var screenshotsAnalyticsDisposables: IBGDisposable? = null

    @JvmStatic
    fun handleCoreEvents(context: Context?, coreEvent: IBGSdkCoreEvent) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "receive new IBG core event: $coreEvent")
        when (coreEvent) {
            IBGSdkCoreEvent.NetworkActivated -> handleNetworkActivated()
            IBGSdkCoreEvent.Session.SessionStarted -> handleSessionStarted()
            IBGSdkCoreEvent.User.LoggedOut -> clearUserActivities()
            IBGSdkCoreEvent.SdkVersionChanged -> context?.let(::migrateBugsToEncryptedTable)
            is IBGSdkCoreEvent.FeaturesFetched ->  handleConfigurations(coreEvent.response)
            is IBGSdkCoreEvent.ReproState -> handleReproState(coreEvent.modesMap)
            else -> {}
        }
    }

    private fun handleSessionStarted() {
        FetchReportCategoriesJob.getInstance().start()
    }

    private fun handleNetworkActivated() {
        NormalBugsUploaderJob.getInstance().start()
        ProactiveReportsBugsUploaderJob.getInstance().start()
    }

    private fun clearUserActivities() = BugSettings.getInstance()
        .run {
            lastBugTime = 0L
        }

    private fun migrateBugsToEncryptedTable(context: Context) {
        ServiceLocator.getBugReportsDbMigrationHelper().migrateToEncrypted(context)
    }

    private fun handleConfigurations(configurations: String) {
        ServiceLocator.getConfigurationsHandler().handleConfigurations(configurations)
        ServiceLocator.getReproProxy().evaluate(ServiceLocator.getConfigurationsProvider())
        ScreenshotsAnalyticsEventBus.post(AnalyticsEvent.ScreenshotEvent.OnConfigurationChanged)
    }

    private fun handleReproState(modesMap: Map<Int, Int>) {
        ServiceLocator.getConfigurationsHandler().handle(modesMap)
        ServiceLocator.getReproProxy().evaluate(ServiceLocator.getConfigurationsProvider())
        ScreenshotsAnalyticsEventBus.post(AnalyticsEvent.ScreenshotEvent.OnConfigurationChanged)
    }

    @JvmStatic
    fun handleSDKDismissing() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "SDK dismissed Handle sdk dismissing")
        runOnSdkDismissedRunnable()
        removeBug()
    }

    private fun removeBug() {
        LiveBugManager.getInstance().removeBug()
    }

    private fun runOnSdkDismissedRunnable() {
        if (BugSettings.getInstance().onSdkDismissCallback != null &&
            LiveBugManager.getInstance().bug != null &&
            LiveBugManager.getInstance().dismissType != null
        ) {
            BugSettings.getInstance().onSdkDismissCallback!!.call(
                ReportingPluginWrapper.getDismissType(LiveBugManager.getInstance().dismissType),
                ReportingPluginWrapper.getReportType(LiveBugManager.getInstance().bug!!.type)
            )
        }
    }

    @JvmStatic
    fun subscribeToProductAnalyticsCollector() {
        if (screenshotsAnalyticsDisposables == null) {
            screenshotsAnalyticsDisposables =
                ScreenshotsAnalyticsEventBus.subscribe(ServiceLocator.getBugsProductAnalyticsCollector())
        }
    }

    @JvmStatic
    fun unsubscribeFromProductAnalyticsCollector() {
        screenshotsAnalyticsDisposables?.dispose()
        screenshotsAnalyticsDisposables = null

    }
}