package com.instabug.bug;

import static com.instabug.bug.BugsCoreEventsHandler.unsubscribeFromProductAnalyticsCollector;

import android.content.Context;
import android.content.Intent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.internal.video.ScreenRecordingService;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.bug.network.NormalBugsUploaderJob;
import com.instabug.bug.network.ProactiveReportsBugsUploaderJob;
import com.instabug.bug.onboardingbugreporting.State;
import com.instabug.bug.onboardingbugreporting.utils.WelcomeMessageHelperApiContract;
import com.instabug.bug.onboardingbugreporting.utils.WelcomeMessageHelperApiImp;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.settings.PerSessionSettings;
import com.instabug.bug.utils.CleanUpUtils;
import com.instabug.bug.view.actionList.service.FetchReportCategoriesJob;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.ReproConfigurations;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ScreenRecordingEventBus;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.eventpublisher.IBGCompositeDisposable;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.frustratingexperience.FrustratingExperienceEventBus;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.video.ScreenRecordingEvent;
import com.instabug.library.internal.video.ScreenRecordingServiceEventBus;
import com.instabug.library.invocation.util.InstabugFloatingButtonEdge;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tokenmapping.MappedTokenChangedEventBus;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Locale;

/**
 * Created by mohamedzakaria on 6/5/17.
 */

public class BugPlugin extends Plugin {

    public static final String SCREEN_RECORDING_EVENT_NAME = "SendDataToRecordingService";
    @Nullable
    private IBGCompositeDisposable disposables;

    @Nullable
    @VisibleForTesting
    IBGDisposable mappedTokenChangedDisposable;

    @Override
    public void init(Context context) {
        initInvocationManager();
        ServiceLocator.getConfigurationsProvider().restoreFromCache();
        ServiceLocator.getReproProxy().evaluate(ServiceLocator.getConfigurationsProvider());
        ServiceLocator.getReproScreenshotsCacheDir().addWatcher(Constants.REPRO_SCREENSHOTS_DIR_WATCHER_ID);
        ServiceLocator.getHubDataWatcher().addWatcher(Constants.REPRO_SCREENSHOTS_DIR_WATCHER_ID);
        CleanUpUtils.cleanupUnsentBugsFiles(context);
        super.init(context);
    }

    private void initInvocationManager() {
        CoreServiceLocator.setInvocationManagerContract(InvocationManager.getInstance());
        InvocationManager.getInstance().getCurrentInvocationSettings().setFloatingButtonEdge(InstabugFloatingButtonEdge.RIGHT);
        InvocationManager.getInstance().notifyPrimaryColorChanged();
        InvocationManager.getInstance().setInstabugInvocationEvent(InstabugCore.getInvocationEvents());
    }

    @Override
    public void wake() {
        ServiceLocator.getReproScreenshotsCacheDir().consentOnCleansing(Constants.REPRO_SCREENSHOTS_DIR_WATCHER_ID);
        ServiceLocator.getHubDataWatcher().consentOnCleansing(Constants.REPRO_SCREENSHOTS_DIR_WATCHER_ID);
        PerSessionSettings perSessionSettings = PerSessionSettings.getInstance();
        ScreenRecordingEvent recordReadyEvent = perSessionSettings.getRecordReadyEvent();
        if(recordReadyEvent != null) {
            PoolProvider.postMainThreadTask(() -> ScreenRecordingEventBus.getInstance().post(recordReadyEvent));
            perSessionSettings.setRecordReadyEvent(null);
        }
    }

    @Override
    public void sleep() {
        // TODO map APIs
    }

    private void subscribeOnCoreEvents() {
       if (contextWeakReference != null && contextWeakReference.get() != null) {
            getOrCreateCompositeDisposables().add(subscribeToCoreEvents());
        }


    }

    @NonNull
    private IBGDisposable subscribeToCoreEvents() {
        return IBGCoreEventSubscriber
                .subscribe((event -> {
                    if (contextWeakReference != null) {
                        BugsCoreEventsHandler.handleCoreEvents(contextWeakReference.get(), event);
                    }
                }));
    }

    private void subscribeOnMappedTokenChangedEvent() {
        if (mappedTokenChangedDisposable == null) {
            mappedTokenChangedDisposable = MappedTokenChangedEventBus.INSTANCE.subscribe((event -> {
                FetchReportCategoriesJob.getInstance().resetLastFetchTimeout();
                FetchReportCategoriesJob.getInstance().start();
            }));
        }
    }

    private void subscribeOnFrustratingExperienceEventBus() {
        getOrCreateCompositeDisposables().add(FrustratingExperienceEventBus.INSTANCE.subscribe(event -> ServiceLocator.getFrustratingExperienceEventsHandler().handle(event)));
    }

    private void unsubscribeFromMappedTokenChangedEvents() {
        if (mappedTokenChangedDisposable != null) {
            mappedTokenChangedDisposable.dispose();
            mappedTokenChangedDisposable = null;
        }
    }

    @Override
    public void initDefaultPromptOptionAvailabilityState() {
        ReportingPluginWrapper.initDefaultPromptOptionAvailabilityState();
    }

    @Override
    public ArrayList<PluginPromptOption> getPromptOptions() {
        if (contextWeakReference != null && contextWeakReference.get() != null) {
            return ReportingPluginWrapper.getPromptOptions(contextWeakReference.get());
        }
        return new ArrayList<>();
    }

    @Override
    @Nullable
    public ArrayList<PluginPromptOption> getPluginOptions(boolean ignoreBaseFeature) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[BugPlugin#getPluginOptions] Getting plugin options");
        if (contextWeakReference != null && contextWeakReference.get() != null) {
            return ReportingPluginWrapper.getReportingOptions(ignoreBaseFeature, contextWeakReference.get());
        }
        InstabugSDKLogger.d(Constants.LOG_TAG, "[BugPlugin#getPluginOptions] No options, returning null");
        return null;
    }

    @Override
    public boolean isFeatureEnabled() {
        return InstabugCore.isFeatureEnabled(IBGFeature.BUG_REPORTING);
    }

    @Override
    public void stop() {
        ReportingPluginWrapper.release();
        unSubscribeFromEvents();
        unsubscribeFromMappedTokenChangedEvents();
        InvocationManager.getInstance().sleep();
        ScreenRecordingServiceEventBus.unregisterHandler(SCREEN_RECORDING_EVENT_NAME);
        unsubscribeFromProductAnalyticsCollector();
    }

    @Override
    public void start(final Context context) {
        ReportingPluginWrapper.init(context);
        if (Instabug.isEnabled() && BugSettings.getInstance().isBugReportingStateEnabled() && BugSettings.getInstance().getWelcomeMessageState() != State.DISABLED) {
            WelcomeMessageHelperApiContract welcomeMessageHelperApiContract = new WelcomeMessageHelperApiImp();
            welcomeMessageHelperApiContract.showIntroMessage();
        }
        loadAndApplyCachedReproConfigurations();
        subscribeOnCoreEvents();
        subscribeOnMappedTokenChangedEvent();
        NormalBugsUploaderJob.getInstance().start();
        ProactiveReportsBugsUploaderJob.getInstance().start();
        InvocationManager.getInstance().listen();
        retrieveIntentFromPermissionsActivity();
        subscribeOnFrustratingExperienceEventBus();
        BugsCoreEventsHandler.subscribeToProductAnalyticsCollector();

    }

    public void retrieveIntentFromPermissionsActivity() {
        ScreenRecordingServiceEventBus.registerHandler(SCREEN_RECORDING_EVENT_NAME, data -> {
            Context context = data.getContext();
            if (context != null && data.getResultOk() != null && data.isSent() != null && data.getMediaProjectionIntent() != null) {
                Intent capturedIntent = ScreenRecordingService.newIntent(data.getContext(), data.getResultOk(), data.getMediaProjectionIntent(), data.isSent());
                ScreenRecordingService.startScreenRecordingService(context, capturedIntent);
            }
        });


    }

    private void loadAndApplyCachedReproConfigurations() {
        SettingsManager settingsManager = SettingsManager.getInstance();
        if (settingsManager != null) {
            ReproConfigurations reproConfigurations = settingsManager.getReproConfigurations();
            WeakReference<Context> localContextReference = contextWeakReference;
            if (reproConfigurations != null && localContextReference != null) {
                BugsCoreEventsHandler.handleCoreEvents(
                        localContextReference.get(),
                        new IBGSdkCoreEvent.ReproState(reproConfigurations.getModesMap())
                );
            }
        }
    }

    private void unSubscribeFromEvents() {
        if (disposables != null) disposables.dispose();
    }

    @Override
    public long getLastActivityTime() {
        return BugSettings.getInstance().getLastBugTime();
    }

    private IBGCompositeDisposable getOrCreateCompositeDisposables() {
        return disposables != null ? disposables : (disposables = new IBGCompositeDisposable());
    }

    @Override
    public void onLocaleChanged(Locale oldLocale, Locale newLocale) {
        super.onLocaleChanged(oldLocale, newLocale);
        FetchReportCategoriesJob.getInstance().start();
    }
}
