package com.ideabus.ideabuslibrary.cloud;

import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.ideabus.ideabuslibrary.BaseGlobal;
import com.ideabus.ideabuslibrary.BaseValuePair;

import org.apache.http.Consts;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * ERROR_CODE_NO_INTERNET = 1
 * ERROR_CODE_SERVER_ERROR = 2
 * ERROR_CODE_UNKNOWN_ERROR = 3
 */
public abstract class BaseHttpModule {

	private static final String TAG = BaseHttpModule.class.getSimpleName();
	protected Handler handler;

	public static final int ERROR_CODE_NO_INTERNET = 1;
    public static final int ERROR_CODE_SERVER_ERROR = 2;
    public static final int ERROR_CODE_UNKNOWN_ERROR = 3;

	public BaseHttpModule(){
		handler = new Handler();
	}

	/**
	 * Post成功
	 * @param apiUrl {@link BaseHttpPost#postArchives(String, String, String, byte[], ArrayList)}的apiUrl
	 * @param result Json data
	 */
	protected abstract void apiPostArchivesSucceeded(String apiUrl, String result);
	/**
	 * Post失敗
	 * @param apiUrl {@link BaseHttpPost#postArchives(String, String, String, byte[], ArrayList)}的apiUrl
	 * @param errorCode
	 *          {@link BaseHttpModule#ERROR_CODE_NO_INTERNET}, {@link BaseHttpModule#ERROR_CODE_SERVER_ERROR}, {@link BaseHttpModule#ERROR_CODE_UNKNOWN_ERROR}<br>
	 *
	 * @param errorMessage 錯誤訊息
	 */
	protected abstract void apiPostArchivesFailed(String apiUrl, int errorCode, String errorMessage);

	/**
	 * Post成功
	 * @param apiUrl {@link BaseHttpPost#postText(String, String, ArrayList)}的apiUrl
	 * @param result Json data
	 */
	protected abstract void apiPostSucceeded(String apiUrl, String result);
	/**
	 * Post失敗
	 * @param apiUrl {@link BaseHttpPost#postText(String, String, ArrayList)}}的apiUrl
	 * @param errorCode
	 *          {@link BaseHttpModule#ERROR_CODE_NO_INTERNET}, {@link BaseHttpModule#ERROR_CODE_SERVER_ERROR}, {@link BaseHttpModule#ERROR_CODE_UNKNOWN_ERROR}<br>
	 *
	 * @param errorMessage 錯誤訊息
	 */
	protected abstract void apiPostFailed(String apiUrl, int errorCode, String errorMessage);


	/**
	 * Get成功
	 * @param apiUrl {@link BaseHttpGet#httpGetMethod(String, String, ArrayList)}的apiUrl
	 * @param result Json data
	 */
	protected abstract void apiGetSucceeded(String apiUrl, String result);
	/**
	 * Get失敗
	 * @param apiUrl {@link BaseHttpGet#httpGetMethod(String, String, ArrayList)}的apiUrl
	 * @param errorCode
	 *          {@link BaseHttpModule#ERROR_CODE_NO_INTERNET},
	 *          {@link BaseHttpModule#ERROR_CODE_SERVER_ERROR},
	 *          {@link BaseHttpModule#ERROR_CODE_UNKNOWN_ERROR}<br>
	 *
	 * @param errorMessage 錯誤訊息
	 */
	protected abstract void apiGetFailed(String apiUrl, int errorCode, String errorMessage);

	public static String encodeJson(ArrayList<BaseValuePair> params, String fieldName) throws JSONException {
		ArrayList<ArrayList<BaseValuePair>> multiParams = new ArrayList<>();
		multiParams.add(params);
		return encodeJsonMulti(multiParams, fieldName);
	}

	/**
	 * int[]轉成Json
	 * @param params 數值陣列
	 * @param fieldName 欄位名稱
	 * @return [1,2,3] or {"array":[1,2,3,4,5,6,7,8]}
	 * @throws JSONException
	 */
	public static String encodeIntArrayToJson(int[] params, String fieldName) throws JSONException {
        String[] strs = new String[params.length];
        for(int i = 0 ; i < params.length ; i++){
            strs[i] = String.valueOf(params[i]);
        }
        return encodeIntArrayToJson(strs, fieldName);
	}

	public static String encodeIntArrayToJson(String[] params, String fieldName) throws JSONException {

        JSONObject resultJson = new JSONObject();
        JSONArray jsonArray = new JSONArray();

        if(fieldName != null && fieldName.length() > 0){
            for(String values : params){
                jsonArray.put(values);
            }
            resultJson.put(fieldName, jsonArray);
            return resultJson.toString();
        }else{
            for(String values : params){
                jsonArray.put(values);
            }
            return jsonArray.toString();
        }
    }

	/**
	 * 最多只能Encode 2層
	 * @param params 數據
	 * @param fieldName 欄位名稱
	 * @return
	 * @throws JSONException
	 */
	public static String encodeJsonMulti(ArrayList<ArrayList<BaseValuePair>> params, String fieldName) throws JSONException {
		JSONObject resultJson = new JSONObject();

		if(fieldName != null && fieldName.length() > 0){
			JSONArray jsonArray = new JSONArray();

			for(ArrayList<BaseValuePair> valuesArray : params){

				JSONObject jsonObject = new JSONObject();

				for(BaseValuePair values : valuesArray){
					String valueStr = values.getStringValue();
					if(valueStr != null)
						jsonObject.put(values.getName(), valueStr);
					else
						jsonObject.put(values.getName(), values.getIntValue());
				}
				jsonArray.put(jsonObject);
			}
			resultJson.put(fieldName, jsonArray);
		}else{
			for(ArrayList<BaseValuePair> valuesArray : params){
				for(BaseValuePair values : valuesArray){
					String valueStr = values.getStringValue();
					if(valueStr != null)
						resultJson.put(values.getName(), valueStr);
					else
						resultJson.put(values.getName(), values.getIntValue());
				}
			}
		}
		return resultJson.toString();
	}

	/**
	 * parse沒有包含key值的 json array字串
	 * @param jsonString e.g. ["0911111111","0488888888"] or ["21","7","5","2","1"]
	 * @return
     */
	public static ArrayList<String> parseJsonArrayNoContainKey(@NonNull String jsonString) {
		ArrayList<String> dataList = new ArrayList<>();

		try{
			JSONArray jsonList = new JSONArray(jsonString);
			for(int i = 0 ; i < jsonList.length() ; i ++){
				dataList.add(jsonList.getString(i));
			}
		}catch(Exception e){
			e.printStackTrace();
		}

		return dataList.size() == 0 ? null : dataList;
	}

	/**
	 * parse包含key跟value的 json array字串
	 * @param jsonString [{"tel":"0423015352","tel2":"10"},{"tel":"0412345678","tel2":"55"}]
	 * @return
     */
	public static ArrayList<Map<String, String>> parseJsonArrayContainKey(@NonNull String jsonString) {
		ArrayList<Map<String, String>> dataList = new ArrayList<>();

		try{
			JSONArray jsonList = new JSONArray(jsonString);

			for(int i = 0 ; i < jsonList.length() ; i ++){
				JSONObject jsonObject = new JSONObject(jsonList.getString(i));
				Map<String, String> map = addDecodingJson(null, jsonObject);
				dataList.add(map);
			}
		}catch(Exception e){
			e.printStackTrace();
		}

		return dataList.size() == 0 ? null : dataList;
	}

	/**
	 * 解析JSon字串
	 * @param jsonString 要解析的JSon完整字串
	 * @param fieldNames 巢狀JSon的欄位Name...
	 * @return
	 */
	@Nullable
	public static ArrayList<Map<String, String>> parseJson(@NonNull String jsonString, String... fieldNames) {
		JSONObject jsonObject;
		try {
			jsonObject = new JSONObject(jsonString);
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
		ArrayList<Map<String, String>> dataList = new ArrayList<>();

		if(fieldNames != null && fieldNames.length > 0){
			for(String fieldName : fieldNames){
				JSONArray jsonList = null;

				//如果Json最外層有包起來
				Map<String, String> outsideMap = new HashMap<>();
				if(fieldName != null && !fieldName.equals("")){
					try{
						//第一層的取出來的data暫存在Map<>
//						outsideMap.putAll(addDecodingJson(jsonObject));
                        outsideMap = addDecodingJson(null, jsonObject);
						jsonList = jsonObject.getJSONArray(fieldName);
					}catch(JSONException e){
						e.printStackTrace();
						//不是JSON List，是一般JSONObject
                        dataList.add(addDecodingJson(null, jsonObject));
					}
				}

				//如果Json最外層有包起來，且正確取出資料
				if(jsonList != null){
					try{
						for(int i = 0 ; i < jsonList.length() ; i ++){
							jsonObject = new JSONObject(jsonList.getString(i));

							//第二層的
							//與之前暫存在Map<>的第一層data合併
							Map<String, String> insideMap = addDecodingJson(outsideMap, jsonObject);

//							insideMap.putAll(outsideMap);
							//刪除第二層資料未parse原始的資料
                            insideMap.remove(fieldName);
							dataList.add(insideMap);
						}
					}catch(JSONException e){
						e.printStackTrace();
					}
				}else {
					dataList.add(addDecodingJson(null, jsonObject));
				}
			}
		}else{
			dataList.add(addDecodingJson(null, jsonObject));
		}

		return dataList.size() == 0 ? null : dataList;
	}

	protected static Map<String, String> addDecodingJson(Map<String, String> map, JSONObject jsonObject){
		String key;
		String value;

		Iterator<String> keyIter = jsonObject.keys();

        Map<String, String> lastMap = new HashMap<>();
        if(map != null)
            lastMap.putAll(map);

		try{
			while (keyIter.hasNext()) {
				key = keyIter.next();
				value = jsonObject.get(key).toString();
				//如果遇到第二層重複的話，key後面+'1'
				if(map != null && lastMap.containsKey(key))
					key += "1";
                lastMap.put(key, value);
			}
		}catch(Exception e){
			e.printStackTrace();
		}

		return lastMap;
	}

    protected StringBuffer getRequestData(List<NameValuePair> params, String encode) {
        StringBuffer stringBuffer = new StringBuffer();

        for(NameValuePair param : params) {
            String name = param.getName();
            String value = param.getValue();
			if(name == null || value == null || name.length() == 0 || value.length() == 0)
				continue;
            try {
                stringBuffer.append(name)
                        .append("=")
                        .append(URLEncoder.encode(value, encode))
                        .append("&");
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        //刪除最後一個"&"
        deleteEndWord(stringBuffer, "&");

        return stringBuffer;
    }

    /**
     * 刪除最後一個Word,如果Word不存在則不做任何動作
     * @param jsonBuffer 字串
     */
	public static void deleteEndWord(StringBuffer jsonBuffer, @NonNull String word){
        int length = jsonBuffer.length();
        if(length == 0)
            return;
        //刪除最後的word
        if(word.equals(jsonBuffer.substring(length - 1, length))){
            jsonBuffer.delete(length - 1, length);
        }
    }

}
