/*
 * Copyright 2016 Google Inc. All rights reserved.
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this
 * file except in compliance with the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF
 * ANY KIND, either express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.google.maps;

import com.google.gson.FieldNamingPolicy;
import com.google.maps.errors.ApiException;
import com.google.maps.internal.ApiConfig;
import com.google.maps.internal.ApiResponse;
import com.google.maps.model.LatLng;
import com.google.maps.model.PlaceType;
import com.google.maps.model.PlacesSearchResponse;
import com.google.maps.model.PlacesSearchResult;
import com.google.maps.model.PriceLevel;

public class RadarSearchRequest
    extends PendingResultBase<
        PlacesSearchResponse, RadarSearchRequest, RadarSearchRequest.Response> {

  static final ApiConfig API_CONFIG =
      new ApiConfig("/maps/api/place/radarsearch/json")
          .fieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES);

  protected RadarSearchRequest(GeoApiContext context) {
    super(context, API_CONFIG, Response.class);
  }

  /**
   * The latitude/longitude around which to retrieve place information.
   *
   * @param location The location around which to search.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest location(LatLng location) {
    return param("location", location);
  }

  /**
   * The distance (in meters) within which to return place results. The maximum allowed radius is
   * 50,000 meters. Note that radius must not be included if rankby=DISTANCE is specified.
   *
   * @param distance The radius distance to restrict results.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest radius(int distance) {
    if (distance > 50000) {
      throw new IllegalArgumentException("The maximum allowed radius is 50,000 meters.");
    }
    return param("radius", String.valueOf(distance));
  }

  /**
   * A term to be matched against all content that Google has indexed for this place, including but
   * not limited to name, type, and address, as well as customer reviews and other third-party
   * content.
   *
   * @param keyword The keyword to search for.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest keyword(String keyword) {
    return param("keyword", keyword);
  }

  /**
   * Restricts to places that are at least this price level.
   *
   * @param priceLevel The minimum price level to restrict results with.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest minPrice(PriceLevel priceLevel) {
    return param("minprice", priceLevel);
  }

  /**
   * Restricts to places that are at most this price level.
   *
   * @param priceLevel The maximum price level to restrict results with.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest maxPrice(PriceLevel priceLevel) {
    return param("maxprice", priceLevel);
  }

  /**
   * One or more terms to be matched against the names of places, separated with space characters.
   *
   * @param name The name to restrict results with.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest name(String name) {
    return param("name", name);
  }

  /**
   * Restricts the results to only those places that are open for business at the time the query is
   * sent.
   *
   * @param openNow Whether to restrict to results that are open now.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest openNow(boolean openNow) {
    return param("opennow", String.valueOf(openNow));
  }

  /**
   * Restricts the results to places matching the specified type.
   *
   * @param type The Place Type to restrict results to.
   * @return Returns this {@code RadarSearchRequest} for call chaining.
   */
  public RadarSearchRequest type(PlaceType type) {
    return param("type", type);
  }

  @Override
  protected void validateRequest() {
    if (!params().containsKey("keyword")
        && !params().containsKey("name")
        && !params().containsKey("type")) {
      throw new IllegalArgumentException("Request must contain 'keyword', 'name' or 'type'.");
    }
  }

  public static class Response implements ApiResponse<PlacesSearchResponse> {

    public String status;
    public String htmlAttributions[];
    public PlacesSearchResult results[];
    public String nextPageToken;
    public String errorMessage;

    @Override
    public boolean successful() {
      return "OK".equals(status) || "ZERO_RESULTS".equals(status);
    }

    @Override
    public PlacesSearchResponse getResult() {
      PlacesSearchResponse result = new PlacesSearchResponse();
      result.htmlAttributions = htmlAttributions;
      result.results = results;
      result.nextPageToken = nextPageToken;
      return result;
    }

    @Override
    public ApiException getError() {
      if (successful()) {
        return null;
      }
      return ApiException.from(status, errorMessage);
    }
  }
}
