/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.dev.shell.remoteui;

import com.google.gwt.dev.util.log.AbstractTreeLogger;

/**
 * A tree logger that creates log entries using a ViewerService.
 */
public final class ViewerServiceTreeLogger extends AbstractTreeLogger {

  private int logHandle = -1;
  private final ViewerServiceClient viewerServiceClient;

  /**
   * Creates a new instance with the given Viewer Service requestor.
   * 
   * @param viewerServiceClient An object that can be used to make requests to a
   *          viewer service server.
   */
  public ViewerServiceTreeLogger(ViewerServiceClient viewerServiceClient) {
    this.viewerServiceClient = viewerServiceClient;
  }

  /**
   * Creates a new logger for a branch. Note that the logger's handle has not
   * been set as yet; it will only be set once the branch is committed.
   */
  @Override
  public AbstractTreeLogger doBranch() {
    ViewerServiceTreeLogger childLogger = new ViewerServiceTreeLogger(
        viewerServiceClient);
    return childLogger;
  }

  /**
   * Commits the branch, and sets the log handle of the branch logger.
   */
  @Override
  public void doCommitBranch(AbstractTreeLogger childBeingCommitted, Type type,
      String msg, Throwable caught, HelpInfo helpInfo) {
    int branchLogHandle = viewerServiceClient.addLogBranch(type, msg, caught,
        helpInfo, getLogHandle(), childBeingCommitted.getBranchedIndex());
    ((ViewerServiceTreeLogger) childBeingCommitted).setLogHandle(branchLogHandle);
  }

  @Override
  public void doLog(int indexOfLogEntryWithinParentLogger, Type type,
      String msg, Throwable caught, HelpInfo helpInfo) {
    viewerServiceClient.addLogEntry(indexOfLogEntryWithinParentLogger, type,
        msg, caught, helpInfo, getLogHandle());
  }

  /**
   * Get the handle of this logger. The handle is generated by the viewer
   * service when this logger is first created. If this logger was created as a
   * branch of another logger, then the handle will not be available until the
   * branch has been committed.
   * 
   * @return the handle for this logger, or -1 if it has not been set
   */
  public int getLogHandle() {
    return logHandle;
  }

  public void setLogHandle(int logHandle) {
    this.logHandle = logHandle;
  }
}
