/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.crashlytics.internal.common;

import android.app.ActivityManager;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.os.StatFs;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.google.android.gms.tasks.SuccessContinuation;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.TaskCompletionSource;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.crashlytics.internal.CrashlyticsNativeComponent;
import com.google.firebase.crashlytics.internal.Logger;
import com.google.firebase.crashlytics.internal.NativeSessionFileProvider;
import com.google.firebase.crashlytics.internal.analytics.AnalyticsEventLogger;
import com.google.firebase.crashlytics.internal.common.AppData;
import com.google.firebase.crashlytics.internal.common.BytesBackedNativeSessionFile;
import com.google.firebase.crashlytics.internal.common.CLSUUID;
import com.google.firebase.crashlytics.internal.common.CommonUtils;
import com.google.firebase.crashlytics.internal.common.CrashlyticsBackgroundWorker;
import com.google.firebase.crashlytics.internal.common.CrashlyticsCore;
import com.google.firebase.crashlytics.internal.common.CrashlyticsFileMarker;
import com.google.firebase.crashlytics.internal.common.CrashlyticsUncaughtExceptionHandler;
import com.google.firebase.crashlytics.internal.common.DataCollectionArbiter;
import com.google.firebase.crashlytics.internal.common.DeliveryMechanism;
import com.google.firebase.crashlytics.internal.common.FileBackedNativeSessionFile;
import com.google.firebase.crashlytics.internal.common.IdManager;
import com.google.firebase.crashlytics.internal.common.NativeSessionFile;
import com.google.firebase.crashlytics.internal.common.NativeSessionFileGzipper;
import com.google.firebase.crashlytics.internal.common.SessionReportingCoordinator;
import com.google.firebase.crashlytics.internal.common.Utils;
import com.google.firebase.crashlytics.internal.metadata.LogFileManager;
import com.google.firebase.crashlytics.internal.metadata.UserMetadata;
import com.google.firebase.crashlytics.internal.model.StaticSessionData;
import com.google.firebase.crashlytics.internal.persistence.FileStore;
import com.google.firebase.crashlytics.internal.settings.Settings;
import com.google.firebase.crashlytics.internal.settings.SettingsProvider;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.SortedSet;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

class CrashlyticsController {
    static final String FIREBASE_CRASH_TYPE = "fatal";
    static final String FIREBASE_TIMESTAMP = "timestamp";
    static final String FIREBASE_APPLICATION_EXCEPTION = "_ae";
    static final String APP_EXCEPTION_MARKER_PREFIX = ".ae";
    static final FilenameFilter APP_EXCEPTION_MARKER_FILTER = (directory, filename) -> filename.startsWith(APP_EXCEPTION_MARKER_PREFIX);
    static final String NATIVE_SESSION_DIR = "native-sessions";
    static final int FIREBASE_CRASH_TYPE_FATAL = 1;
    private static final String GENERATOR_FORMAT = "Crashlytics Android SDK/%s";
    private final Context context;
    private final DataCollectionArbiter dataCollectionArbiter;
    private final CrashlyticsFileMarker crashMarker;
    private final UserMetadata userMetadata;
    private final CrashlyticsBackgroundWorker backgroundWorker;
    private final IdManager idManager;
    private final FileStore fileStore;
    private final AppData appData;
    private final LogFileManager logFileManager;
    private final CrashlyticsNativeComponent nativeComponent;
    private final AnalyticsEventLogger analyticsEventLogger;
    private final SessionReportingCoordinator reportingCoordinator;
    private CrashlyticsUncaughtExceptionHandler crashHandler;
    private SettingsProvider settingsProvider = null;
    final TaskCompletionSource<Boolean> unsentReportsAvailable = new TaskCompletionSource();
    final TaskCompletionSource<Boolean> reportActionProvided = new TaskCompletionSource();
    final TaskCompletionSource<Void> unsentReportsHandled = new TaskCompletionSource();
    final AtomicBoolean checkForUnsentReportsCalled = new AtomicBoolean(false);

    CrashlyticsController(Context context, CrashlyticsBackgroundWorker backgroundWorker, IdManager idManager, DataCollectionArbiter dataCollectionArbiter, FileStore fileStore, CrashlyticsFileMarker crashMarker, AppData appData, UserMetadata userMetadata, LogFileManager logFileManager, SessionReportingCoordinator sessionReportingCoordinator, CrashlyticsNativeComponent nativeComponent, AnalyticsEventLogger analyticsEventLogger) {
        this.context = context;
        this.backgroundWorker = backgroundWorker;
        this.idManager = idManager;
        this.dataCollectionArbiter = dataCollectionArbiter;
        this.fileStore = fileStore;
        this.crashMarker = crashMarker;
        this.appData = appData;
        this.userMetadata = userMetadata;
        this.logFileManager = logFileManager;
        this.nativeComponent = nativeComponent;
        this.analyticsEventLogger = analyticsEventLogger;
        this.reportingCoordinator = sessionReportingCoordinator;
    }

    private Context getContext() {
        return this.context;
    }

    void enableExceptionHandling(String sessionIdentifier, Thread.UncaughtExceptionHandler defaultHandler, SettingsProvider settingsProvider) {
        this.settingsProvider = settingsProvider;
        this.openSession(sessionIdentifier);
        CrashlyticsUncaughtExceptionHandler.CrashListener crashListener = new CrashlyticsUncaughtExceptionHandler.CrashListener(){

            @Override
            public void onUncaughtException(@NonNull SettingsProvider settingsDataProvider, @NonNull Thread thread, @NonNull Throwable ex) {
                CrashlyticsController.this.handleUncaughtException(settingsDataProvider, thread, ex);
            }
        };
        this.crashHandler = new CrashlyticsUncaughtExceptionHandler(crashListener, settingsProvider, defaultHandler, this.nativeComponent);
        Thread.setDefaultUncaughtExceptionHandler(this.crashHandler);
    }

    void handleUncaughtException(@NonNull SettingsProvider settingsProvider, @NonNull Thread thread, @NonNull Throwable ex) {
        this.handleUncaughtException(settingsProvider, thread, ex, false);
    }

    synchronized void handleUncaughtException(final @NonNull SettingsProvider settingsProvider, final @NonNull Thread thread, final @NonNull Throwable ex, final boolean isOnDemand) {
        Logger.getLogger().d("Handling uncaught exception \"" + ex + "\" from thread " + thread.getName());
        final long timestampMillis = System.currentTimeMillis();
        Task<Void> handleUncaughtExceptionTask = this.backgroundWorker.submitTask(new Callable<Task<Void>>(){

            @Override
            public Task<Void> call() throws Exception {
                long timestampSeconds = CrashlyticsController.getTimestampSeconds(timestampMillis);
                final String currentSessionId = CrashlyticsController.this.getCurrentSessionId();
                if (currentSessionId == null) {
                    Logger.getLogger().e("Tried to write a fatal exception while no session was open.");
                    return Tasks.forResult(null);
                }
                CrashlyticsController.this.crashMarker.create();
                CrashlyticsController.this.reportingCoordinator.persistFatalEvent(ex, thread, currentSessionId, timestampSeconds);
                CrashlyticsController.this.doWriteAppExceptionMarker(timestampMillis);
                CrashlyticsController.this.doCloseSessions(settingsProvider);
                CrashlyticsController.this.doOpenSession(new CLSUUID(CrashlyticsController.this.idManager).toString());
                if (!CrashlyticsController.this.dataCollectionArbiter.isAutomaticDataCollectionEnabled()) {
                    return Tasks.forResult(null);
                }
                final Executor executor = CrashlyticsController.this.backgroundWorker.getExecutor();
                return settingsProvider.getSettingsAsync().onSuccessTask(executor, (SuccessContinuation)new SuccessContinuation<Settings, Void>(){

                    @NonNull
                    public Task<Void> then(@Nullable Settings settings) throws Exception {
                        if (settings == null) {
                            Logger.getLogger().w("Received null app settings, cannot send reports at crash time.");
                            return Tasks.forResult(null);
                        }
                        return Tasks.whenAll((Task[])new Task[]{CrashlyticsController.this.logAnalyticsAppExceptionEvents(), CrashlyticsController.this.reportingCoordinator.sendReports(executor, isOnDemand ? currentSessionId : null)});
                    }
                });
            }
        });
        try {
            Utils.awaitEvenIfOnMainThread(handleUncaughtExceptionTask);
        }
        catch (TimeoutException e) {
            Logger.getLogger().e("Cannot send reports. Timed out while fetching settings.");
        }
        catch (Exception e) {
            Logger.getLogger().e("Error handling uncaught exception", e);
        }
    }

    private Task<Boolean> waitForReportAction() {
        if (this.dataCollectionArbiter.isAutomaticDataCollectionEnabled()) {
            Logger.getLogger().d("Automatic data collection is enabled. Allowing upload.");
            this.unsentReportsAvailable.trySetResult((Object)false);
            return Tasks.forResult((Object)true);
        }
        Logger.getLogger().d("Automatic data collection is disabled.");
        Logger.getLogger().v("Notifying that unsent reports are available.");
        this.unsentReportsAvailable.trySetResult((Object)true);
        Task collectionEnabled = this.dataCollectionArbiter.waitForAutomaticDataCollectionEnabled().onSuccessTask((SuccessContinuation)new SuccessContinuation<Void, Boolean>(){

            @NonNull
            public Task<Boolean> then(@Nullable Void aVoid) throws Exception {
                return Tasks.forResult((Object)true);
            }
        });
        Logger.getLogger().d("Waiting for send/deleteUnsentReports to be called.");
        return Utils.race(collectionEnabled, this.reportActionProvided.getTask());
    }

    boolean didCrashOnPreviousExecution() {
        if (!this.crashMarker.isPresent()) {
            String sessionId = this.getCurrentSessionId();
            return sessionId != null && this.nativeComponent.hasCrashDataForSession(sessionId);
        }
        Logger.getLogger().v("Found previous crash marker.");
        this.crashMarker.remove();
        return Boolean.TRUE;
    }

    @NonNull
    Task<Boolean> checkForUnsentReports() {
        if (!this.checkForUnsentReportsCalled.compareAndSet(false, true)) {
            Logger.getLogger().w("checkForUnsentReports should only be called once per execution.");
            return Tasks.forResult((Object)false);
        }
        return this.unsentReportsAvailable.getTask();
    }

    Task<Void> sendUnsentReports() {
        this.reportActionProvided.trySetResult((Object)true);
        return this.unsentReportsHandled.getTask();
    }

    Task<Void> deleteUnsentReports() {
        this.reportActionProvided.trySetResult((Object)false);
        return this.unsentReportsHandled.getTask();
    }

    Task<Void> submitAllReports(final Task<Settings> settingsDataTask) {
        if (!this.reportingCoordinator.hasReportsToSend()) {
            Logger.getLogger().v("No crash reports are available to be sent.");
            this.unsentReportsAvailable.trySetResult((Object)false);
            return Tasks.forResult(null);
        }
        Logger.getLogger().v("Crash reports are available to be sent.");
        return this.waitForReportAction().onSuccessTask((SuccessContinuation)new SuccessContinuation<Boolean, Void>(){

            @NonNull
            public Task<Void> then(final @Nullable Boolean send) throws Exception {
                return CrashlyticsController.this.backgroundWorker.submitTask(new Callable<Task<Void>>(){

                    @Override
                    public Task<Void> call() throws Exception {
                        if (!send.booleanValue()) {
                            Logger.getLogger().v("Deleting cached crash reports...");
                            CrashlyticsController.deleteFiles(CrashlyticsController.this.listAppExceptionMarkerFiles());
                            CrashlyticsController.this.reportingCoordinator.removeAllReports();
                            CrashlyticsController.this.unsentReportsHandled.trySetResult(null);
                            return Tasks.forResult(null);
                        }
                        Logger.getLogger().d("Sending cached crash reports...");
                        boolean dataCollectionToken = send;
                        CrashlyticsController.this.dataCollectionArbiter.grantDataCollectionPermission(dataCollectionToken);
                        final Executor executor = CrashlyticsController.this.backgroundWorker.getExecutor();
                        return settingsDataTask.onSuccessTask(executor, (SuccessContinuation)new SuccessContinuation<Settings, Void>(){

                            @NonNull
                            public Task<Void> then(@Nullable Settings appSettingsData) throws Exception {
                                if (appSettingsData == null) {
                                    Logger.getLogger().w("Received null app settings at app startup. Cannot send cached reports");
                                    return Tasks.forResult(null);
                                }
                                CrashlyticsController.this.logAnalyticsAppExceptionEvents();
                                CrashlyticsController.this.reportingCoordinator.sendReports(executor);
                                CrashlyticsController.this.unsentReportsHandled.trySetResult(null);
                                return Tasks.forResult(null);
                            }
                        });
                    }
                });
            }
        });
    }

    void writeToLog(final long timestamp, final String msg) {
        this.backgroundWorker.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                if (!CrashlyticsController.this.isHandlingException()) {
                    CrashlyticsController.this.logFileManager.writeToLog(timestamp, msg);
                }
                return null;
            }
        });
    }

    void writeNonFatalException(final @NonNull Thread thread, final @NonNull Throwable ex) {
        final long timestampMillis = System.currentTimeMillis();
        this.backgroundWorker.submit(new Runnable(){

            @Override
            public void run() {
                if (!CrashlyticsController.this.isHandlingException()) {
                    long timestampSeconds = CrashlyticsController.getTimestampSeconds(timestampMillis);
                    String currentSessionId = CrashlyticsController.this.getCurrentSessionId();
                    if (currentSessionId == null) {
                        Logger.getLogger().w("Tried to write a non-fatal exception while no session was open.");
                        return;
                    }
                    CrashlyticsController.this.reportingCoordinator.persistNonFatalEvent(ex, thread, currentSessionId, timestampSeconds);
                }
            }
        });
    }

    void logFatalException(Thread thread, Throwable ex) {
        if (this.settingsProvider == null) {
            Logger.getLogger().w("settingsProvider not set");
            return;
        }
        this.handleUncaughtException(this.settingsProvider, thread, ex, true);
    }

    void setUserId(String identifier) {
        this.userMetadata.setUserId(identifier);
    }

    void setCustomKey(String key, String value) {
        try {
            this.userMetadata.setCustomKey(key, value);
        }
        catch (IllegalArgumentException ex) {
            if (this.context != null && CommonUtils.isAppDebuggable(this.context)) {
                throw ex;
            }
            Logger.getLogger().e("Attempting to set custom attribute with null key, ignoring.");
        }
    }

    void setCustomKeys(Map<String, String> keysAndValues) {
        this.userMetadata.setCustomKeys(keysAndValues);
    }

    void setInternalKey(String key, String value) {
        try {
            this.userMetadata.setInternalKey(key, value);
        }
        catch (IllegalArgumentException ex) {
            if (this.context != null && CommonUtils.isAppDebuggable(this.context)) {
                throw ex;
            }
            Logger.getLogger().e("Attempting to set custom attribute with null key, ignoring.");
        }
    }

    void openSession(final String sessionIdentifier) {
        this.backgroundWorker.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                CrashlyticsController.this.doOpenSession(sessionIdentifier);
                return null;
            }
        });
    }

    @Nullable
    private String getCurrentSessionId() {
        SortedSet<String> sortedOpenSessions = this.reportingCoordinator.listSortedOpenSessionIds();
        return !sortedOpenSessions.isEmpty() ? sortedOpenSessions.first() : null;
    }

    boolean finalizeSessions(SettingsProvider settingsProvider) {
        this.backgroundWorker.checkRunningOnThread();
        if (this.isHandlingException()) {
            Logger.getLogger().w("Skipping session finalization because a crash has already occurred.");
            return Boolean.FALSE;
        }
        Logger.getLogger().v("Finalizing previously open sessions.");
        try {
            this.doCloseSessions(true, settingsProvider);
        }
        catch (Exception e) {
            Logger.getLogger().e("Unable to finalize previously open sessions.", e);
            return false;
        }
        Logger.getLogger().v("Closed all previously open sessions.");
        return true;
    }

    private void doOpenSession(String sessionIdentifier) {
        long startedAtSeconds = CrashlyticsController.getCurrentTimestampSeconds();
        Logger.getLogger().d("Opening a new session with ID " + sessionIdentifier);
        String generator = String.format(Locale.US, GENERATOR_FORMAT, CrashlyticsCore.getVersion());
        StaticSessionData.AppData appData = CrashlyticsController.createAppData(this.idManager, this.appData);
        StaticSessionData.OsData osData = CrashlyticsController.createOsData();
        StaticSessionData.DeviceData deviceData = CrashlyticsController.createDeviceData();
        this.nativeComponent.prepareNativeSession(sessionIdentifier, generator, startedAtSeconds, StaticSessionData.create(appData, osData, deviceData));
        this.logFileManager.setCurrentSession(sessionIdentifier);
        this.reportingCoordinator.onBeginSession(sessionIdentifier, startedAtSeconds);
    }

    void doCloseSessions(SettingsProvider settingsProvider) {
        this.doCloseSessions(false, settingsProvider);
    }

    private void doCloseSessions(boolean skipCurrentSession, SettingsProvider settingsProvider) {
        int offset = skipCurrentSession ? 1 : 0;
        ArrayList<String> sortedOpenSessions = new ArrayList<String>(this.reportingCoordinator.listSortedOpenSessionIds());
        if (sortedOpenSessions.size() <= offset) {
            Logger.getLogger().v("No open sessions to be closed.");
            return;
        }
        String mostRecentSessionIdToClose = (String)sortedOpenSessions.get(offset);
        if (settingsProvider.getSettingsSync().featureFlagData.collectAnrs) {
            this.writeApplicationExitInfoEventIfRelevant(mostRecentSessionIdToClose);
        } else {
            Logger.getLogger().v("ANR feature disabled.");
        }
        if (this.nativeComponent.hasCrashDataForSession(mostRecentSessionIdToClose)) {
            this.finalizePreviousNativeSession(mostRecentSessionIdToClose);
        }
        String currentSessionId = null;
        if (skipCurrentSession) {
            currentSessionId = (String)sortedOpenSessions.get(0);
        }
        this.reportingCoordinator.finalizeSessions(CrashlyticsController.getCurrentTimestampSeconds(), currentSessionId);
    }

    List<File> listAppExceptionMarkerFiles() {
        return this.fileStore.getCommonFiles(APP_EXCEPTION_MARKER_FILTER);
    }

    private void finalizePreviousNativeSession(String previousSessionId) {
        Logger.getLogger().v("Finalizing native report for session " + previousSessionId);
        NativeSessionFileProvider nativeSessionFileProvider = this.nativeComponent.getSessionFileProvider(previousSessionId);
        File minidumpFile = nativeSessionFileProvider.getMinidumpFile();
        if (minidumpFile == null || !minidumpFile.exists()) {
            Logger.getLogger().w("No minidump data found for session " + previousSessionId);
            return;
        }
        long eventTime = minidumpFile.lastModified();
        LogFileManager previousSessionLogManager = new LogFileManager(this.fileStore, previousSessionId);
        File nativeSessionDirectory = this.fileStore.getNativeSessionDir(previousSessionId);
        if (!nativeSessionDirectory.isDirectory()) {
            Logger.getLogger().w("Couldn't create directory to store native session files, aborting.");
            return;
        }
        this.doWriteAppExceptionMarker(eventTime);
        List<NativeSessionFile> nativeSessionFiles = CrashlyticsController.getNativeSessionFiles(nativeSessionFileProvider, previousSessionId, this.fileStore, previousSessionLogManager.getBytesForLog());
        NativeSessionFileGzipper.processNativeSessions(nativeSessionDirectory, nativeSessionFiles);
        Logger.getLogger().d("CrashlyticsController#finalizePreviousNativeSession");
        this.reportingCoordinator.finalizeSessionWithNativeEvent(previousSessionId, nativeSessionFiles);
        previousSessionLogManager.clearLog();
    }

    private static long getCurrentTimestampSeconds() {
        return CrashlyticsController.getTimestampSeconds(System.currentTimeMillis());
    }

    private static long getTimestampSeconds(long timestampMillis) {
        return timestampMillis / 1000L;
    }

    private void doWriteAppExceptionMarker(long eventTime) {
        try {
            if (!this.fileStore.getCommonFile(APP_EXCEPTION_MARKER_PREFIX + eventTime).createNewFile()) {
                throw new IOException("Create new file failed.");
            }
        }
        catch (IOException e) {
            Logger.getLogger().w("Could not create app exception marker file.", e);
        }
    }

    private static StaticSessionData.AppData createAppData(IdManager idManager, AppData appData) {
        return StaticSessionData.AppData.create(idManager.getAppIdentifier(), appData.versionCode, appData.versionName, idManager.getCrashlyticsInstallId(), DeliveryMechanism.determineFrom(appData.installerPackageName).getId(), appData.developmentPlatformProvider);
    }

    private static StaticSessionData.OsData createOsData() {
        return StaticSessionData.OsData.create(Build.VERSION.RELEASE, Build.VERSION.CODENAME, CommonUtils.isRooted());
    }

    private static StaticSessionData.DeviceData createDeviceData() {
        StatFs statFs = new StatFs(Environment.getDataDirectory().getPath());
        long diskSpace = (long)statFs.getBlockCount() * (long)statFs.getBlockSize();
        return StaticSessionData.DeviceData.create(CommonUtils.getCpuArchitectureInt(), Build.MODEL, Runtime.getRuntime().availableProcessors(), CommonUtils.getTotalRamInBytes(), diskSpace, CommonUtils.isEmulator(), CommonUtils.getDeviceState(), Build.MANUFACTURER, Build.PRODUCT);
    }

    UserMetadata getUserMetadata() {
        return this.userMetadata;
    }

    boolean isHandlingException() {
        return this.crashHandler != null && this.crashHandler.isHandlingException();
    }

    private Task<Void> logAnalyticsAppExceptionEvents() {
        ArrayList<Task<Void>> events = new ArrayList<Task<Void>>();
        List<File> appExceptionMarkers = this.listAppExceptionMarkerFiles();
        for (File markerFile : appExceptionMarkers) {
            try {
                long timestamp = Long.parseLong(markerFile.getName().substring(APP_EXCEPTION_MARKER_PREFIX.length()));
                events.add(this.logAnalyticsAppExceptionEvent(timestamp));
            }
            catch (NumberFormatException nfe) {
                Logger.getLogger().w("Could not parse app exception timestamp from file " + markerFile.getName());
            }
            markerFile.delete();
        }
        return Tasks.whenAll(events);
    }

    private Task<Void> logAnalyticsAppExceptionEvent(final long timestamp) {
        if (CrashlyticsController.firebaseCrashExists()) {
            Logger.getLogger().w("Skipping logging Crashlytics event to Firebase, FirebaseCrash exists");
            return Tasks.forResult(null);
        }
        Logger.getLogger().d("Logging app exception event to Firebase Analytics");
        ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1);
        return Tasks.call((Executor)executor, (Callable)new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                Bundle params = new Bundle();
                params.putInt(CrashlyticsController.FIREBASE_CRASH_TYPE, 1);
                params.putLong(CrashlyticsController.FIREBASE_TIMESTAMP, timestamp);
                CrashlyticsController.this.analyticsEventLogger.logEvent(CrashlyticsController.FIREBASE_APPLICATION_EXCEPTION, params);
                return null;
            }
        });
    }

    private static void deleteFiles(List<File> files) {
        for (File file : files) {
            file.delete();
        }
    }

    private static boolean firebaseCrashExists() {
        try {
            Class<?> clazz = Class.forName("com.google.firebase.crash.FirebaseCrash");
            return true;
        }
        catch (ClassNotFoundException e) {
            return false;
        }
    }

    @NonNull
    static List<NativeSessionFile> getNativeSessionFiles(NativeSessionFileProvider fileProvider, String previousSessionId, FileStore fileStore, byte[] logBytes) {
        File userFile = fileStore.getSessionFile(previousSessionId, "user-data");
        File keysFile = fileStore.getSessionFile(previousSessionId, "keys");
        ArrayList<NativeSessionFile> nativeSessionFiles = new ArrayList<NativeSessionFile>();
        nativeSessionFiles.add(new BytesBackedNativeSessionFile("logs_file", "logs", logBytes));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("crash_meta_file", "metadata", fileProvider.getMetadataFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("session_meta_file", "session", fileProvider.getSessionFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("app_meta_file", "app", fileProvider.getAppFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("device_meta_file", "device", fileProvider.getDeviceFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("os_meta_file", "os", fileProvider.getOsFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("minidump_file", "minidump", fileProvider.getMinidumpFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("user_meta_file", "user", userFile));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("keys_file", "keys", keysFile));
        return nativeSessionFiles;
    }

    private void writeApplicationExitInfoEventIfRelevant(String sessionId) {
        if (Build.VERSION.SDK_INT >= 30) {
            ActivityManager activityManager = (ActivityManager)this.context.getSystemService("activity");
            List applicationExitInfoList = activityManager.getHistoricalProcessExitReasons(null, 0, 0);
            if (applicationExitInfoList.size() != 0) {
                LogFileManager relevantSessionLogManager = new LogFileManager(this.fileStore, sessionId);
                UserMetadata relevantUserMetadata = UserMetadata.loadFromExistingSession(sessionId, this.fileStore, this.backgroundWorker);
                this.reportingCoordinator.persistRelevantAppExitInfoEvent(sessionId, applicationExitInfoList, relevantSessionLogManager, relevantUserMetadata);
            } else {
                Logger.getLogger().v("No ApplicationExitInfo available. Session: " + sessionId);
            }
        } else {
            Logger.getLogger().v("ANR feature enabled, but device is API " + Build.VERSION.SDK_INT);
        }
    }
}

