/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.crashlytics.internal.common;

import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.os.StatFs;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.google.android.gms.tasks.SuccessContinuation;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.TaskCompletionSource;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.crashlytics.internal.CrashlyticsNativeComponent;
import com.google.firebase.crashlytics.internal.Logger;
import com.google.firebase.crashlytics.internal.NativeSessionFileProvider;
import com.google.firebase.crashlytics.internal.analytics.AnalyticsEventLogger;
import com.google.firebase.crashlytics.internal.common.AppData;
import com.google.firebase.crashlytics.internal.common.BytesBackedNativeSessionFile;
import com.google.firebase.crashlytics.internal.common.CLSUUID;
import com.google.firebase.crashlytics.internal.common.CommonUtils;
import com.google.firebase.crashlytics.internal.common.CrashlyticsBackgroundWorker;
import com.google.firebase.crashlytics.internal.common.CrashlyticsCore;
import com.google.firebase.crashlytics.internal.common.CrashlyticsFileMarker;
import com.google.firebase.crashlytics.internal.common.CrashlyticsUncaughtExceptionHandler;
import com.google.firebase.crashlytics.internal.common.DataCollectionArbiter;
import com.google.firebase.crashlytics.internal.common.DeliveryMechanism;
import com.google.firebase.crashlytics.internal.common.FileBackedNativeSessionFile;
import com.google.firebase.crashlytics.internal.common.IdManager;
import com.google.firebase.crashlytics.internal.common.MetaDataStore;
import com.google.firebase.crashlytics.internal.common.NativeSessionFile;
import com.google.firebase.crashlytics.internal.common.NativeSessionFileGzipper;
import com.google.firebase.crashlytics.internal.common.SessionReportingCoordinator;
import com.google.firebase.crashlytics.internal.common.UserMetadata;
import com.google.firebase.crashlytics.internal.common.Utils;
import com.google.firebase.crashlytics.internal.log.LogFileManager;
import com.google.firebase.crashlytics.internal.persistence.FileStore;
import com.google.firebase.crashlytics.internal.settings.SettingsDataProvider;
import com.google.firebase.crashlytics.internal.settings.model.AppSettingsData;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;

class CrashlyticsController {
    static final String FIREBASE_CRASH_TYPE = "fatal";
    static final String FIREBASE_TIMESTAMP = "timestamp";
    static final String FIREBASE_APPLICATION_EXCEPTION = "_ae";
    static final String APP_EXCEPTION_MARKER_PREFIX = ".ae";
    static final FilenameFilter APP_EXCEPTION_MARKER_FILTER = (directory, filename) -> filename.startsWith(APP_EXCEPTION_MARKER_PREFIX);
    static final String NATIVE_SESSION_DIR = "native-sessions";
    static final int FIREBASE_CRASH_TYPE_FATAL = 1;
    private static final String GENERATOR_FORMAT = "Crashlytics Android SDK/%s";
    private final Context context;
    private final DataCollectionArbiter dataCollectionArbiter;
    private final CrashlyticsFileMarker crashMarker;
    private final UserMetadata userMetadata;
    private final CrashlyticsBackgroundWorker backgroundWorker;
    private final IdManager idManager;
    private final FileStore fileStore;
    private final AppData appData;
    private final LogFileManager.DirectoryProvider logFileDirectoryProvider;
    private final LogFileManager logFileManager;
    private final CrashlyticsNativeComponent nativeComponent;
    private final String unityVersion;
    private final AnalyticsEventLogger analyticsEventLogger;
    private final SessionReportingCoordinator reportingCoordinator;
    private CrashlyticsUncaughtExceptionHandler crashHandler;
    final TaskCompletionSource<Boolean> unsentReportsAvailable = new TaskCompletionSource();
    final TaskCompletionSource<Boolean> reportActionProvided = new TaskCompletionSource();
    final TaskCompletionSource<Void> unsentReportsHandled = new TaskCompletionSource();
    final AtomicBoolean checkForUnsentReportsCalled = new AtomicBoolean(false);

    CrashlyticsController(Context context, CrashlyticsBackgroundWorker backgroundWorker, IdManager idManager, DataCollectionArbiter dataCollectionArbiter, FileStore fileStore, CrashlyticsFileMarker crashMarker, AppData appData, UserMetadata userMetadata, LogFileManager logFileManager, LogFileManager.DirectoryProvider logFileDirectoryProvider, SessionReportingCoordinator sessionReportingCoordinator, CrashlyticsNativeComponent nativeComponent, AnalyticsEventLogger analyticsEventLogger) {
        this.context = context;
        this.backgroundWorker = backgroundWorker;
        this.idManager = idManager;
        this.dataCollectionArbiter = dataCollectionArbiter;
        this.fileStore = fileStore;
        this.crashMarker = crashMarker;
        this.appData = appData;
        this.userMetadata = userMetadata;
        this.logFileManager = logFileManager;
        this.logFileDirectoryProvider = logFileDirectoryProvider;
        this.nativeComponent = nativeComponent;
        this.unityVersion = appData.unityVersionProvider.getUnityVersion();
        this.analyticsEventLogger = analyticsEventLogger;
        this.reportingCoordinator = sessionReportingCoordinator;
    }

    private Context getContext() {
        return this.context;
    }

    void enableExceptionHandling(Thread.UncaughtExceptionHandler defaultHandler, SettingsDataProvider settingsProvider) {
        this.openSession();
        CrashlyticsUncaughtExceptionHandler.CrashListener crashListener = new CrashlyticsUncaughtExceptionHandler.CrashListener(){

            @Override
            public void onUncaughtException(@NonNull SettingsDataProvider settingsDataProvider, @NonNull Thread thread, @NonNull Throwable ex) {
                CrashlyticsController.this.handleUncaughtException(settingsDataProvider, thread, ex);
            }
        };
        this.crashHandler = new CrashlyticsUncaughtExceptionHandler(crashListener, settingsProvider, defaultHandler);
        Thread.setDefaultUncaughtExceptionHandler(this.crashHandler);
    }

    synchronized void handleUncaughtException(final @NonNull SettingsDataProvider settingsDataProvider, final @NonNull Thread thread, final @NonNull Throwable ex) {
        Logger.getLogger().d("Handling uncaught exception \"" + ex + "\" from thread " + thread.getName());
        final Date time = new Date();
        Task<Void> handleUncaughtExceptionTask = this.backgroundWorker.submitTask(new Callable<Task<Void>>(){

            @Override
            public Task<Void> call() throws Exception {
                long timestampSeconds = CrashlyticsController.getTimestampSeconds(time);
                String currentSessionId = CrashlyticsController.this.getCurrentSessionId();
                if (currentSessionId == null) {
                    Logger.getLogger().e("Tried to write a fatal exception while no session was open.");
                    return Tasks.forResult(null);
                }
                CrashlyticsController.this.crashMarker.create();
                CrashlyticsController.this.reportingCoordinator.persistFatalEvent(ex, thread, currentSessionId, timestampSeconds);
                CrashlyticsController.this.doWriteAppExceptionMarker(time.getTime());
                CrashlyticsController.this.doCloseSessions();
                CrashlyticsController.this.doOpenSession();
                if (!CrashlyticsController.this.dataCollectionArbiter.isAutomaticDataCollectionEnabled()) {
                    return Tasks.forResult(null);
                }
                final Executor executor = CrashlyticsController.this.backgroundWorker.getExecutor();
                return settingsDataProvider.getAppSettings().onSuccessTask(executor, (SuccessContinuation)new SuccessContinuation<AppSettingsData, Void>(){

                    @NonNull
                    public Task<Void> then(@Nullable AppSettingsData appSettingsData) throws Exception {
                        if (appSettingsData == null) {
                            Logger.getLogger().w("Received null app settings, cannot send reports at crash time.");
                            return Tasks.forResult(null);
                        }
                        return Tasks.whenAll((Task[])new Task[]{CrashlyticsController.this.logAnalyticsAppExceptionEvents(), CrashlyticsController.this.reportingCoordinator.sendReports(executor)});
                    }
                });
            }
        });
        try {
            Utils.awaitEvenIfOnMainThread(handleUncaughtExceptionTask);
        }
        catch (Exception e) {
            Logger.getLogger().e("Error handling uncaught exception", e);
        }
    }

    private Task<Boolean> waitForReportAction() {
        if (this.dataCollectionArbiter.isAutomaticDataCollectionEnabled()) {
            Logger.getLogger().d("Automatic data collection is enabled. Allowing upload.");
            this.unsentReportsAvailable.trySetResult((Object)false);
            return Tasks.forResult((Object)true);
        }
        Logger.getLogger().d("Automatic data collection is disabled.");
        Logger.getLogger().v("Notifying that unsent reports are available.");
        this.unsentReportsAvailable.trySetResult((Object)true);
        Task collectionEnabled = this.dataCollectionArbiter.waitForAutomaticDataCollectionEnabled().onSuccessTask((SuccessContinuation)new SuccessContinuation<Void, Boolean>(){

            @NonNull
            public Task<Boolean> then(@Nullable Void aVoid) throws Exception {
                return Tasks.forResult((Object)true);
            }
        });
        Logger.getLogger().d("Waiting for send/deleteUnsentReports to be called.");
        return Utils.race(collectionEnabled, this.reportActionProvided.getTask());
    }

    boolean didCrashOnPreviousExecution() {
        if (!this.crashMarker.isPresent()) {
            String sessionId = this.getCurrentSessionId();
            return sessionId != null && this.nativeComponent.hasCrashDataForSession(sessionId);
        }
        Logger.getLogger().v("Found previous crash marker.");
        this.crashMarker.remove();
        return Boolean.TRUE;
    }

    @NonNull
    Task<Boolean> checkForUnsentReports() {
        if (!this.checkForUnsentReportsCalled.compareAndSet(false, true)) {
            Logger.getLogger().w("checkForUnsentReports should only be called once per execution.");
            return Tasks.forResult((Object)false);
        }
        return this.unsentReportsAvailable.getTask();
    }

    Task<Void> sendUnsentReports() {
        this.reportActionProvided.trySetResult((Object)true);
        return this.unsentReportsHandled.getTask();
    }

    Task<Void> deleteUnsentReports() {
        this.reportActionProvided.trySetResult((Object)false);
        return this.unsentReportsHandled.getTask();
    }

    Task<Void> submitAllReports(final Task<AppSettingsData> appSettingsDataTask) {
        if (!this.reportingCoordinator.hasReportsToSend()) {
            Logger.getLogger().v("No crash reports are available to be sent.");
            this.unsentReportsAvailable.trySetResult((Object)false);
            return Tasks.forResult(null);
        }
        Logger.getLogger().v("Crash reports are available to be sent.");
        return this.waitForReportAction().onSuccessTask((SuccessContinuation)new SuccessContinuation<Boolean, Void>(){

            @NonNull
            public Task<Void> then(final @Nullable Boolean send) throws Exception {
                return CrashlyticsController.this.backgroundWorker.submitTask(new Callable<Task<Void>>(){

                    @Override
                    public Task<Void> call() throws Exception {
                        if (!send.booleanValue()) {
                            Logger.getLogger().v("Deleting cached crash reports...");
                            CrashlyticsController.deleteFiles(CrashlyticsController.this.listAppExceptionMarkerFiles());
                            CrashlyticsController.this.reportingCoordinator.removeAllReports();
                            CrashlyticsController.this.unsentReportsHandled.trySetResult(null);
                            return Tasks.forResult(null);
                        }
                        Logger.getLogger().d("Sending cached crash reports...");
                        boolean dataCollectionToken = send;
                        CrashlyticsController.this.dataCollectionArbiter.grantDataCollectionPermission(dataCollectionToken);
                        final Executor executor = CrashlyticsController.this.backgroundWorker.getExecutor();
                        return appSettingsDataTask.onSuccessTask(executor, (SuccessContinuation)new SuccessContinuation<AppSettingsData, Void>(){

                            @NonNull
                            public Task<Void> then(@Nullable AppSettingsData appSettingsData) throws Exception {
                                if (appSettingsData == null) {
                                    Logger.getLogger().w("Received null app settings at app startup. Cannot send cached reports");
                                    return Tasks.forResult(null);
                                }
                                CrashlyticsController.this.logAnalyticsAppExceptionEvents();
                                CrashlyticsController.this.reportingCoordinator.sendReports(executor);
                                CrashlyticsController.this.unsentReportsHandled.trySetResult(null);
                                return Tasks.forResult(null);
                            }
                        });
                    }
                });
            }
        });
    }

    void writeToLog(final long timestamp, final String msg) {
        this.backgroundWorker.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                if (!CrashlyticsController.this.isHandlingException()) {
                    CrashlyticsController.this.logFileManager.writeToLog(timestamp, msg);
                }
                return null;
            }
        });
    }

    void writeNonFatalException(final @NonNull Thread thread, final @NonNull Throwable ex) {
        final Date time = new Date();
        this.backgroundWorker.submit(new Runnable(){

            @Override
            public void run() {
                if (!CrashlyticsController.this.isHandlingException()) {
                    long timestampSeconds = CrashlyticsController.getTimestampSeconds(time);
                    String currentSessionId = CrashlyticsController.this.getCurrentSessionId();
                    if (currentSessionId == null) {
                        Logger.getLogger().w("Tried to write a non-fatal exception while no session was open.");
                        return;
                    }
                    CrashlyticsController.this.reportingCoordinator.persistNonFatalEvent(ex, thread, currentSessionId, timestampSeconds);
                }
            }
        });
    }

    void setUserId(String identifier) {
        this.userMetadata.setUserId(identifier);
        this.cacheUserData(this.userMetadata);
    }

    void setCustomKey(String key, String value) {
        try {
            this.userMetadata.setCustomKey(key, value);
        }
        catch (IllegalArgumentException ex) {
            if (this.context != null && CommonUtils.isAppDebuggable(this.context)) {
                throw ex;
            }
            Logger.getLogger().e("Attempting to set custom attribute with null key, ignoring.");
            return;
        }
        this.cacheKeyData(this.userMetadata.getCustomKeys());
    }

    void setCustomKeys(Map<String, String> keysAndValues) {
        this.userMetadata.setCustomKeys(keysAndValues);
        this.cacheKeyData(this.userMetadata.getCustomKeys());
    }

    private void cacheUserData(final UserMetadata userMetaData) {
        this.backgroundWorker.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                String currentSessionId = CrashlyticsController.this.getCurrentSessionId();
                if (currentSessionId == null) {
                    Logger.getLogger().d("Tried to cache user data while no session was open.");
                    return null;
                }
                CrashlyticsController.this.reportingCoordinator.persistUserId(currentSessionId);
                new MetaDataStore(CrashlyticsController.this.getFilesDir()).writeUserData(currentSessionId, userMetaData);
                return null;
            }
        });
    }

    private void cacheKeyData(final Map<String, String> keyData) {
        this.backgroundWorker.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                String currentSessionId = CrashlyticsController.this.getCurrentSessionId();
                new MetaDataStore(CrashlyticsController.this.getFilesDir()).writeKeyData(currentSessionId, keyData);
                return null;
            }
        });
    }

    void openSession() {
        this.backgroundWorker.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                CrashlyticsController.this.doOpenSession();
                return null;
            }
        });
    }

    @Nullable
    private String getCurrentSessionId() {
        List<String> sortedOpenSessions = this.reportingCoordinator.listSortedOpenSessionIds();
        return !sortedOpenSessions.isEmpty() ? sortedOpenSessions.get(0) : null;
    }

    boolean finalizeSessions() {
        this.backgroundWorker.checkRunningOnThread();
        if (this.isHandlingException()) {
            Logger.getLogger().w("Skipping session finalization because a crash has already occurred.");
            return Boolean.FALSE;
        }
        Logger.getLogger().v("Finalizing previously open sessions.");
        try {
            this.doCloseSessions(true);
        }
        catch (Exception e) {
            Logger.getLogger().e("Unable to finalize previously open sessions.", e);
            return false;
        }
        Logger.getLogger().v("Closed all previously open sessions.");
        return true;
    }

    private void doOpenSession() {
        long startedAtSeconds = CrashlyticsController.getCurrentTimestampSeconds();
        String sessionIdentifier = new CLSUUID(this.idManager).toString();
        Logger.getLogger().d("Opening a new session with ID " + sessionIdentifier);
        this.nativeComponent.openSession(sessionIdentifier);
        this.writeBeginSession(sessionIdentifier, startedAtSeconds);
        this.writeSessionApp(sessionIdentifier);
        this.writeSessionOS(sessionIdentifier);
        this.writeSessionDevice(sessionIdentifier);
        this.logFileManager.setCurrentSession(sessionIdentifier);
        this.reportingCoordinator.onBeginSession(sessionIdentifier, startedAtSeconds);
    }

    void doCloseSessions() {
        this.doCloseSessions(false);
    }

    private void doCloseSessions(boolean skipCurrentSession) {
        int offset = skipCurrentSession ? 1 : 0;
        List<String> sortedOpenSessions = this.reportingCoordinator.listSortedOpenSessionIds();
        if (sortedOpenSessions.size() <= offset) {
            Logger.getLogger().v("No open sessions to be closed.");
            return;
        }
        String mostRecentSessionIdToClose = sortedOpenSessions.get(offset);
        if (this.nativeComponent.hasCrashDataForSession(mostRecentSessionIdToClose)) {
            this.finalizePreviousNativeSession(mostRecentSessionIdToClose);
            if (!this.nativeComponent.finalizeSession(mostRecentSessionIdToClose)) {
                Logger.getLogger().w("Could not finalize native session: " + mostRecentSessionIdToClose);
            }
        }
        String currentSessionId = null;
        if (skipCurrentSession) {
            currentSessionId = sortedOpenSessions.get(0);
        }
        this.reportingCoordinator.finalizeSessions(CrashlyticsController.getCurrentTimestampSeconds(), currentSessionId);
    }

    File[] listNativeSessionFileDirectories() {
        return CrashlyticsController.ensureFileArrayNotNull(this.getNativeSessionFilesDir().listFiles());
    }

    File[] listAppExceptionMarkerFiles() {
        return this.listFilesMatching(APP_EXCEPTION_MARKER_FILTER);
    }

    private File[] listFilesMatching(FilenameFilter filter) {
        return CrashlyticsController.listFilesMatching(this.getFilesDir(), filter);
    }

    private static File[] listFilesMatching(File directory, FilenameFilter filter) {
        return CrashlyticsController.ensureFileArrayNotNull(directory.listFiles(filter));
    }

    private static File[] ensureFileArrayNotNull(File[] files) {
        return files == null ? new File[]{} : files;
    }

    private void finalizePreviousNativeSession(String previousSessionId) {
        Logger.getLogger().v("Finalizing native report for session " + previousSessionId);
        NativeSessionFileProvider nativeSessionFileProvider = this.nativeComponent.getSessionFileProvider(previousSessionId);
        File minidumpFile = nativeSessionFileProvider.getMinidumpFile();
        if (minidumpFile == null || !minidumpFile.exists()) {
            Logger.getLogger().w("No minidump data found for session " + previousSessionId);
            return;
        }
        long eventTime = minidumpFile.lastModified();
        LogFileManager previousSessionLogManager = new LogFileManager(this.context, this.logFileDirectoryProvider, previousSessionId);
        File nativeSessionDirectory = new File(this.getNativeSessionFilesDir(), previousSessionId);
        if (!nativeSessionDirectory.mkdirs()) {
            Logger.getLogger().w("Couldn't create directory to store native session files, aborting.");
            return;
        }
        this.doWriteAppExceptionMarker(eventTime);
        List<NativeSessionFile> nativeSessionFiles = CrashlyticsController.getNativeSessionFiles(nativeSessionFileProvider, previousSessionId, this.getFilesDir(), previousSessionLogManager.getBytesForLog());
        NativeSessionFileGzipper.processNativeSessions(nativeSessionDirectory, nativeSessionFiles);
        this.reportingCoordinator.finalizeSessionWithNativeEvent(previousSessionId, nativeSessionFiles);
        previousSessionLogManager.clearLog();
    }

    private static long getCurrentTimestampSeconds() {
        return CrashlyticsController.getTimestampSeconds(new Date());
    }

    private static long getTimestampSeconds(Date date) {
        return date.getTime() / 1000L;
    }

    private void doWriteAppExceptionMarker(long eventTime) {
        try {
            new File(this.getFilesDir(), APP_EXCEPTION_MARKER_PREFIX + eventTime).createNewFile();
        }
        catch (IOException e) {
            Logger.getLogger().w("Could not create app exception marker file.", e);
        }
    }

    private void writeBeginSession(String sessionId, long startedAtSeconds) {
        String generator = String.format(Locale.US, GENERATOR_FORMAT, CrashlyticsCore.getVersion());
        this.nativeComponent.writeBeginSession(sessionId, generator, startedAtSeconds);
    }

    private void writeSessionApp(String sessionId) {
        String appIdentifier = this.idManager.getAppIdentifier();
        String versionCode = this.appData.versionCode;
        String versionName = this.appData.versionName;
        String installUuid = this.idManager.getCrashlyticsInstallId();
        int deliveryMechanism = DeliveryMechanism.determineFrom(this.appData.installerPackageName).getId();
        this.nativeComponent.writeSessionApp(sessionId, appIdentifier, versionCode, versionName, installUuid, deliveryMechanism, this.unityVersion);
    }

    private void writeSessionOS(String sessionId) {
        String osRelease = Build.VERSION.RELEASE;
        String osCodeName = Build.VERSION.CODENAME;
        boolean isRooted = CommonUtils.isRooted(this.getContext());
        this.nativeComponent.writeSessionOs(sessionId, osRelease, osCodeName, isRooted);
    }

    private void writeSessionDevice(String sessionId) {
        Context context = this.getContext();
        StatFs statFs = new StatFs(Environment.getDataDirectory().getPath());
        int arch = CommonUtils.getCpuArchitectureInt();
        String model = Build.MODEL;
        int availableProcessors = Runtime.getRuntime().availableProcessors();
        long totalRam = CommonUtils.getTotalRamInBytes();
        long diskSpace = (long)statFs.getBlockCount() * (long)statFs.getBlockSize();
        boolean isEmulator = CommonUtils.isEmulator(context);
        int state = CommonUtils.getDeviceState(context);
        String manufacturer = Build.MANUFACTURER;
        String modelClass = Build.PRODUCT;
        this.nativeComponent.writeSessionDevice(sessionId, arch, model, availableProcessors, totalRam, diskSpace, isEmulator, state, manufacturer, modelClass);
    }

    UserMetadata getUserMetadata() {
        return this.userMetadata;
    }

    boolean isHandlingException() {
        return this.crashHandler != null && this.crashHandler.isHandlingException();
    }

    File getFilesDir() {
        return this.fileStore.getFilesDir();
    }

    File getNativeSessionFilesDir() {
        return new File(this.getFilesDir(), NATIVE_SESSION_DIR);
    }

    private Task<Void> logAnalyticsAppExceptionEvents() {
        File[] appExceptionMarkers;
        ArrayList<Task<Void>> events = new ArrayList<Task<Void>>();
        for (File markerFile : appExceptionMarkers = this.listAppExceptionMarkerFiles()) {
            try {
                long timestamp = Long.parseLong(markerFile.getName().substring(APP_EXCEPTION_MARKER_PREFIX.length()));
                events.add(this.logAnalyticsAppExceptionEvent(timestamp));
            }
            catch (NumberFormatException nfe) {
                Logger.getLogger().w("Could not parse app exception timestamp from file " + markerFile.getName());
            }
            markerFile.delete();
        }
        return Tasks.whenAll(events);
    }

    private Task<Void> logAnalyticsAppExceptionEvent(final long timestamp) {
        if (CrashlyticsController.firebaseCrashExists()) {
            Logger.getLogger().w("Skipping logging Crashlytics event to Firebase, FirebaseCrash exists");
            return Tasks.forResult(null);
        }
        Logger.getLogger().d("Logging app exception event to Firebase Analytics");
        ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1);
        return Tasks.call((Executor)executor, (Callable)new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                Bundle params = new Bundle();
                params.putInt(CrashlyticsController.FIREBASE_CRASH_TYPE, 1);
                params.putLong(CrashlyticsController.FIREBASE_TIMESTAMP, timestamp);
                CrashlyticsController.this.analyticsEventLogger.logEvent(CrashlyticsController.FIREBASE_APPLICATION_EXCEPTION, params);
                return null;
            }
        });
    }

    private static void deleteFiles(File[] files) {
        if (files == null) {
            return;
        }
        for (File file : files) {
            file.delete();
        }
    }

    private static boolean firebaseCrashExists() {
        try {
            Class<?> clazz = Class.forName("com.google.firebase.crash.FirebaseCrash");
            return true;
        }
        catch (ClassNotFoundException e) {
            return false;
        }
    }

    @NonNull
    static List<NativeSessionFile> getNativeSessionFiles(NativeSessionFileProvider fileProvider, String previousSessionId, File filesDir, byte[] logBytes) {
        MetaDataStore metaDataStore = new MetaDataStore(filesDir);
        File userFile = metaDataStore.getUserDataFileForSession(previousSessionId);
        File keysFile = metaDataStore.getKeysFileForSession(previousSessionId);
        ArrayList<NativeSessionFile> nativeSessionFiles = new ArrayList<NativeSessionFile>();
        nativeSessionFiles.add(new BytesBackedNativeSessionFile("logs_file", "logs", logBytes));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("crash_meta_file", "metadata", fileProvider.getMetadataFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("session_meta_file", "session", fileProvider.getSessionFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("app_meta_file", "app", fileProvider.getAppFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("device_meta_file", "device", fileProvider.getDeviceFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("os_meta_file", "os", fileProvider.getOsFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("minidump_file", "minidump", fileProvider.getMinidumpFile()));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("user_meta_file", "user", userFile));
        nativeSessionFiles.add(new FileBackedNativeSessionFile("keys_file", "keys", keysFile));
        return nativeSessionFiles;
    }
}

