/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.remoteconfig.internal;

import android.text.format.DateUtils;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import com.google.android.gms.common.util.Clock;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.analytics.connector.AnalyticsConnector;
import com.google.firebase.inject.Provider;
import com.google.firebase.installations.FirebaseInstallationsApi;
import com.google.firebase.installations.InstallationTokenResult;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigClientException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigFetchThrottledException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigServerException;
import com.google.firebase.remoteconfig.internal.ConfigCacheClient;
import com.google.firebase.remoteconfig.internal.ConfigContainer;
import com.google.firebase.remoteconfig.internal.ConfigFetchHttpClient;
import com.google.firebase.remoteconfig.internal.ConfigMetadataClient;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.HttpURLConnection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

public class ConfigFetchHandler {
    public static final long DEFAULT_MINIMUM_FETCH_INTERVAL_IN_SECONDS = TimeUnit.HOURS.toSeconds(12L);
    @VisibleForTesting
    static final int[] BACKOFF_TIME_DURATIONS_IN_MINUTES = new int[]{2, 4, 8, 16, 32, 64, 128, 256};
    @VisibleForTesting
    static final int HTTP_TOO_MANY_REQUESTS = 429;
    @VisibleForTesting
    static final String FIRST_OPEN_TIME_KEY = "_fot";
    private final FirebaseInstallationsApi firebaseInstallations;
    private final Provider<AnalyticsConnector> analyticsConnector;
    private final Executor executor;
    private final Clock clock;
    private final Random randomGenerator;
    private final ConfigCacheClient fetchedConfigsCache;
    private final ConfigFetchHttpClient frcBackendApiClient;
    private final ConfigMetadataClient frcMetadata;
    private final Map<String, String> customHttpHeaders;

    public ConfigFetchHandler(FirebaseInstallationsApi firebaseInstallations, Provider<AnalyticsConnector> analyticsConnector, Executor executor, Clock clock, Random randomGenerator, ConfigCacheClient fetchedConfigsCache, ConfigFetchHttpClient frcBackendApiClient, ConfigMetadataClient frcMetadata, Map<String, String> customHttpHeaders) {
        this.firebaseInstallations = firebaseInstallations;
        this.analyticsConnector = analyticsConnector;
        this.executor = executor;
        this.clock = clock;
        this.randomGenerator = randomGenerator;
        this.fetchedConfigsCache = fetchedConfigsCache;
        this.frcBackendApiClient = frcBackendApiClient;
        this.frcMetadata = frcMetadata;
        this.customHttpHeaders = customHttpHeaders;
    }

    public Task<FetchResponse> fetch() {
        return this.fetch(this.frcMetadata.getMinimumFetchIntervalInSeconds());
    }

    public Task<FetchResponse> fetch(long minimumFetchIntervalInSeconds) {
        return this.fetchedConfigsCache.get().continueWithTask(this.executor, cachedFetchConfigsTask -> this.fetchIfCacheExpiredAndNotThrottled((Task<ConfigContainer>)cachedFetchConfigsTask, minimumFetchIntervalInSeconds));
    }

    private Task<FetchResponse> fetchIfCacheExpiredAndNotThrottled(Task<ConfigContainer> cachedFetchConfigsTask, long minimumFetchIntervalInSeconds) {
        Task fetchResponseTask;
        Date currentTime = new Date(this.clock.currentTimeMillis());
        if (cachedFetchConfigsTask.isSuccessful() && this.areCachedFetchConfigsValid(minimumFetchIntervalInSeconds, currentTime)) {
            return Tasks.forResult((Object)FetchResponse.forLocalStorageUsed(currentTime));
        }
        Date backoffEndTime = this.getBackoffEndTimeInMillis(currentTime);
        if (backoffEndTime != null) {
            fetchResponseTask = Tasks.forException((Exception)((Object)new FirebaseRemoteConfigFetchThrottledException(this.createThrottledMessage(backoffEndTime.getTime() - currentTime.getTime()), backoffEndTime.getTime())));
        } else {
            Task installationIdTask = this.firebaseInstallations.getId();
            Task installationAuthTokenTask = this.firebaseInstallations.getToken(false);
            fetchResponseTask = Tasks.whenAllComplete((Task[])new Task[]{installationIdTask, installationAuthTokenTask}).continueWithTask(this.executor, completedInstallationsTasks -> {
                if (!installationIdTask.isSuccessful()) {
                    return Tasks.forException((Exception)((Object)new FirebaseRemoteConfigClientException("Firebase Installations failed to get installation ID for fetch.", installationIdTask.getException())));
                }
                if (!installationAuthTokenTask.isSuccessful()) {
                    return Tasks.forException((Exception)((Object)new FirebaseRemoteConfigClientException("Firebase Installations failed to get installation auth token for fetch.", installationAuthTokenTask.getException())));
                }
                String installationId = (String)installationIdTask.getResult();
                String installationToken = ((InstallationTokenResult)installationAuthTokenTask.getResult()).getToken();
                return this.fetchFromBackendAndCacheResponse(installationId, installationToken, currentTime);
            });
        }
        return fetchResponseTask.continueWithTask(this.executor, completedFetchTask -> {
            this.updateLastFetchStatusAndTime((Task<FetchResponse>)completedFetchTask, currentTime);
            return completedFetchTask;
        });
    }

    private boolean areCachedFetchConfigsValid(long cacheExpirationInSeconds, Date newFetchTime) {
        Date lastSuccessfulFetchTime = this.frcMetadata.getLastSuccessfulFetchTime();
        if (lastSuccessfulFetchTime.equals(ConfigMetadataClient.LAST_FETCH_TIME_NO_FETCH_YET)) {
            return false;
        }
        Date cacheExpirationTime = new Date(lastSuccessfulFetchTime.getTime() + TimeUnit.SECONDS.toMillis(cacheExpirationInSeconds));
        return newFetchTime.before(cacheExpirationTime);
    }

    @Nullable
    private Date getBackoffEndTimeInMillis(Date currentTime) {
        Date backoffEndTime = this.frcMetadata.getBackoffMetadata().getBackoffEndTime();
        if (currentTime.before(backoffEndTime)) {
            return backoffEndTime;
        }
        return null;
    }

    private String createThrottledMessage(long throttledDurationInMillis) {
        return String.format("Fetch is throttled. Please wait before calling fetch again: %s", DateUtils.formatElapsedTime((long)TimeUnit.MILLISECONDS.toSeconds(throttledDurationInMillis)));
    }

    private Task<FetchResponse> fetchFromBackendAndCacheResponse(String installationId, String installationToken, Date fetchTime) {
        try {
            FetchResponse fetchResponse = this.fetchFromBackend(installationId, installationToken, fetchTime);
            if (fetchResponse.getStatus() != 0) {
                return Tasks.forResult((Object)fetchResponse);
            }
            return this.fetchedConfigsCache.put(fetchResponse.getFetchedConfigs()).onSuccessTask(this.executor, putContainer -> Tasks.forResult((Object)fetchResponse));
        }
        catch (FirebaseRemoteConfigException frce) {
            return Tasks.forException((Exception)((Object)frce));
        }
    }

    @WorkerThread
    private FetchResponse fetchFromBackend(String installationId, String installationToken, Date currentTime) throws FirebaseRemoteConfigException {
        try {
            HttpURLConnection urlConnection = this.frcBackendApiClient.createHttpURLConnection();
            FetchResponse response = this.frcBackendApiClient.fetch(urlConnection, installationId, installationToken, this.getUserProperties(), this.frcMetadata.getLastFetchETag(), this.customHttpHeaders, this.getFirstOpenTime(), currentTime);
            if (response.getLastFetchETag() != null) {
                this.frcMetadata.setLastFetchETag(response.getLastFetchETag());
            }
            this.frcMetadata.resetBackoff();
            return response;
        }
        catch (FirebaseRemoteConfigServerException serverHttpError) {
            ConfigMetadataClient.BackoffMetadata backoffMetadata = this.updateAndReturnBackoffMetadata(serverHttpError.getHttpStatusCode(), currentTime);
            if (this.shouldThrottle(backoffMetadata, serverHttpError.getHttpStatusCode())) {
                throw new FirebaseRemoteConfigFetchThrottledException(backoffMetadata.getBackoffEndTime().getTime());
            }
            throw this.createExceptionWithGenericMessage(serverHttpError);
        }
    }

    private FirebaseRemoteConfigServerException createExceptionWithGenericMessage(FirebaseRemoteConfigServerException httpError) throws FirebaseRemoteConfigClientException {
        String errorMessage;
        switch (httpError.getHttpStatusCode()) {
            case 401: {
                errorMessage = "The request did not have the required credentials. Please make sure your google-services.json is valid.";
                break;
            }
            case 403: {
                errorMessage = "The user is not authorized to access the project. Please make sure you are using the API key that corresponds to your Firebase project.";
                break;
            }
            case 500: {
                errorMessage = "There was an internal server error.";
                break;
            }
            case 502: 
            case 503: 
            case 504: {
                errorMessage = "The server is unavailable. Please try again later.";
                break;
            }
            case 429: {
                throw new FirebaseRemoteConfigClientException("The throttled response from the server was not handled correctly by the FRC SDK.");
            }
            default: {
                errorMessage = "The server returned an unexpected error.";
            }
        }
        return new FirebaseRemoteConfigServerException(httpError.getHttpStatusCode(), "Fetch failed: " + errorMessage, (Throwable)((Object)httpError));
    }

    private ConfigMetadataClient.BackoffMetadata updateAndReturnBackoffMetadata(int statusCode, Date currentTime) {
        if (this.isThrottleableServerError(statusCode)) {
            this.updateBackoffMetadataWithLastFailedFetchTime(currentTime);
        }
        return this.frcMetadata.getBackoffMetadata();
    }

    private boolean isThrottleableServerError(int httpStatusCode) {
        return httpStatusCode == 429 || httpStatusCode == 502 || httpStatusCode == 503 || httpStatusCode == 504;
    }

    private void updateBackoffMetadataWithLastFailedFetchTime(Date lastFailedFetchTime) {
        int numFailedFetches = this.frcMetadata.getBackoffMetadata().getNumFailedFetches();
        long backoffDurationInMillis = this.getRandomizedBackoffDurationInMillis(++numFailedFetches);
        Date backoffEndTime = new Date(lastFailedFetchTime.getTime() + backoffDurationInMillis);
        this.frcMetadata.setBackoffMetadata(numFailedFetches, backoffEndTime);
    }

    private long getRandomizedBackoffDurationInMillis(int numFailedFetches) {
        long timeOutDurationInMillis = TimeUnit.MINUTES.toMillis(BACKOFF_TIME_DURATIONS_IN_MINUTES[Math.min(numFailedFetches, BACKOFF_TIME_DURATIONS_IN_MINUTES.length) - 1]);
        return timeOutDurationInMillis / 2L + (long)this.randomGenerator.nextInt((int)timeOutDurationInMillis);
    }

    private boolean shouldThrottle(ConfigMetadataClient.BackoffMetadata backoffMetadata, int httpStatusCode) {
        return backoffMetadata.getNumFailedFetches() > 1 || httpStatusCode == 429;
    }

    private void updateLastFetchStatusAndTime(Task<FetchResponse> completedFetchTask, Date fetchTime) {
        if (completedFetchTask.isSuccessful()) {
            this.frcMetadata.updateLastFetchAsSuccessfulAt(fetchTime);
            return;
        }
        Exception fetchException = completedFetchTask.getException();
        if (fetchException == null) {
            return;
        }
        if (fetchException instanceof FirebaseRemoteConfigFetchThrottledException) {
            this.frcMetadata.updateLastFetchAsThrottled();
        } else {
            this.frcMetadata.updateLastFetchAsFailed();
        }
    }

    @WorkerThread
    private Map<String, String> getUserProperties() {
        HashMap<String, String> userPropertiesMap = new HashMap<String, String>();
        AnalyticsConnector connector = (AnalyticsConnector)this.analyticsConnector.get();
        if (connector == null) {
            return userPropertiesMap;
        }
        for (Map.Entry userPropertyEntry : connector.getUserProperties(false).entrySet()) {
            userPropertiesMap.put((String)userPropertyEntry.getKey(), userPropertyEntry.getValue().toString());
        }
        return userPropertiesMap;
    }

    @WorkerThread
    private Long getFirstOpenTime() {
        AnalyticsConnector connector = (AnalyticsConnector)this.analyticsConnector.get();
        if (connector == null) {
            return null;
        }
        return (Long)connector.getUserProperties(true).get(FIRST_OPEN_TIME_KEY);
    }

    @VisibleForTesting
    public Provider<AnalyticsConnector> getAnalyticsConnector() {
        return this.analyticsConnector;
    }

    public static class FetchResponse {
        private final Date fetchTime;
        private final int status;
        private final ConfigContainer fetchedConfigs;
        @Nullable
        private final String lastFetchETag;

        private FetchResponse(Date fetchTime, int status, ConfigContainer fetchedConfigs, @Nullable String lastFetchETag) {
            this.fetchTime = fetchTime;
            this.status = status;
            this.fetchedConfigs = fetchedConfigs;
            this.lastFetchETag = lastFetchETag;
        }

        public static FetchResponse forBackendUpdatesFetched(ConfigContainer fetchedConfigs, String lastFetchETag) {
            return new FetchResponse(fetchedConfigs.getFetchTime(), 0, fetchedConfigs, lastFetchETag);
        }

        public static FetchResponse forBackendHasNoUpdates(Date fetchTime) {
            return new FetchResponse(fetchTime, 1, null, null);
        }

        public static FetchResponse forLocalStorageUsed(Date fetchTime) {
            return new FetchResponse(fetchTime, 2, null, null);
        }

        Date getFetchTime() {
            return this.fetchTime;
        }

        @Nullable
        String getLastFetchETag() {
            return this.lastFetchETag;
        }

        int getStatus() {
            return this.status;
        }

        public ConfigContainer getFetchedConfigs() {
            return this.fetchedConfigs;
        }

        @Retention(value=RetentionPolicy.SOURCE)
        public static @interface Status {
            public static final int BACKEND_UPDATES_FETCHED = 0;
            public static final int BACKEND_HAS_NO_UPDATES = 1;
            public static final int LOCAL_STORAGE_USED = 2;
        }
    }
}

