/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirFunctionCallChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.overriddenFunctions
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.containingClassLookupTag
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCallOrigin
import ksp.org.jetbrains.kotlin.fir.originalOrSelf
import ksp.org.jetbrains.kotlin.fir.references.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.types.isAny
import ksp.org.jetbrains.kotlin.fir.types.isNullableAny
import ksp.org.jetbrains.kotlin.fir.types.lowerBoundIfFlexible
import ksp.org.jetbrains.kotlin.name.CallableId
import ksp.org.jetbrains.kotlin.name.ClassId
import ksp.org.jetbrains.kotlin.util.OperatorNameConventions

/**
 * This checker detects if a call by operator 'contains' convention to a Java method violates the expected contract:
 * * "key in map" commonly resolves to stdlib extension that calls Map.containsKey(),
 * but there's a member in ConcurrentHashMap with acceptable signature that delegates to `containsValue` instead,
 * leading to an unexpected result. See KT-18053
 */
object FirJvmInconsistentOperatorFromJavaCallChecker : FirFunctionCallChecker(MppCheckerKind.Common) {
    private val CONCURRENT_HASH_MAP_CALLABLE_ID = CallableId(
        ClassId.fromString("java/util/concurrent/ConcurrentHashMap"),
        OperatorNameConventions.CONTAINS
    )

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirFunctionCall) {
        // Filter out non-operators
        if (expression.origin != FirFunctionCallOrigin.Operator) return
        val callableSymbol = expression.calleeReference.toResolvedCallableSymbol() as? FirNamedFunctionSymbol ?: return
        // Filter out non-contains
        if (callableSymbol.name != OperatorNameConventions.CONTAINS) return
        val valueParameterSymbol = callableSymbol.valueParameterSymbols.singleOrNull() ?: return
        val type = valueParameterSymbol.resolvedReturnTypeRef.coneType.lowerBoundIfFlexible()
        // Filter out handrolled contains with non-Any type
        if (!type.isAny && !type.isNullableAny) return
        callableSymbol.check(expression.calleeReference.source, context, reporter)
    }

    private fun FirNamedFunctionSymbol.check(source: KtSourceElement?, context: CheckerContext, reporter: DiagnosticReporter): Boolean {
        // Unwrap SubstitutionOverride origin if necessary
        if (originalOrSelf().callableId == CONCURRENT_HASH_MAP_CALLABLE_ID) {
            reporter.reportOn(source, FirJvmErrors.CONCURRENT_HASH_MAP_CONTAINS_OPERATOR_ERROR, context)
            return true
        }

        // Check explicitly overridden contains
        val containingClass = containingClassLookupTag()?.toRegularClassSymbol(context.session) ?: return false
        val overriddenFunctions = overriddenFunctions(containingClass, context)
        for (overriddenFunction in overriddenFunctions) {
            if (overriddenFunction is FirNamedFunctionSymbol && overriddenFunction.check(source, context, reporter)) {
                return true
            }
        }

        return false
    }
}
