/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.declaration

import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.classKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirFunctionChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getContainingClassSymbol
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirAnonymousFunction
import ksp.org.jetbrains.kotlin.fir.declarations.FirFunction
import ksp.org.jetbrains.kotlin.fir.declarations.getAnnotationByClassId
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isAbstract
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isInline
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isInlineOrValue
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isSuspend
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.fir.types.isSuspendOrKSuspendFunctionType
import ksp.org.jetbrains.kotlin.name.JvmStandardClassIds.SYNCHRONIZED_ANNOTATION_CLASS_ID

object FirSynchronizedAnnotationChecker : FirFunctionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirFunction) {
        val session = context.session
        val annotation = declaration.getAnnotationByClassId(SYNCHRONIZED_ANNOTATION_CLASS_ID, session) ?: return

        if (declaration.isInline) {
            reporter.reportOn(annotation.source, FirJvmErrors.SYNCHRONIZED_ON_INLINE)
            return
        }
        if (declaration.isSuspend ||
            (declaration as? FirAnonymousFunction)?.typeRef?.coneType?.isSuspendOrKSuspendFunctionType(session) == true
        ) {
            reporter.reportOn(annotation.source, FirJvmErrors.SYNCHRONIZED_ON_SUSPEND)
            return
        }

        val containingClass = declaration.getContainingClassSymbol() ?: return
        when {
            containingClass.classKind == ClassKind.INTERFACE ->
                reporter.reportOn(annotation.source, FirJvmErrors.SYNCHRONIZED_IN_INTERFACE)
            containingClass.classKind == ClassKind.ANNOTATION_CLASS ->
                reporter.reportOn(annotation.source, FirJvmErrors.SYNCHRONIZED_IN_ANNOTATION)
            containingClass.isInlineOrValue ->
                reporter.reportOn(annotation.source, FirJvmErrors.SYNCHRONIZED_ON_VALUE_CLASS)
            declaration.isAbstract ->
                reporter.reportOn(annotation.source, FirJvmErrors.SYNCHRONIZED_ON_ABSTRACT)
        }
    }
}
